/**
 * $Id: EmitInclude.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  java.util.Iterator;
import  java.util.Map;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.PropertySet;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableProjectComponent;
import  com.idaremedia.antx.ErrorSnapshot;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.parameters.FlexValueSupport;
import  com.idaremedia.antx.print.DiagnosticsValue;

/**
 * Helper used to nest Ant entities included with emitted messages. Allows inclusion
 * of comment items like properties, exported-variables, and references while being
 * extensible for additonal items later. Some examples:<pre>
 *   &lt;include property="ant.file"/&gt;
 *   &lt;include var="my.var"/&gt;
 *   &lt;include reference="compiling.err"/&gt;
 *   &lt;include taskdef="emitconfigure"/&gt;
 *   &lt;include typedef="all"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,helper
 * @see      EmitTask#addConfiguredInclude
 **/

public class EmitInclude extends AssertableProjectComponent implements FlexValueSupport
{
    /**
     * Initializes new EmitInclude instance.
     **/
    public EmitInclude()
    {
        super(AntX.feedback+"include");
    }

    /**
     * Sets this include's item's name.
     **/
    protected final void setKey(String key)
    {
        m_itemName = key;
    }


    /**
     * Returns this include's item's name. Returns <i>null</i>
     * if never set (through one of various 'set' methods).
     **/
    public final String getKey()
    {
        return m_itemName;
    }


    /**
     * Marks this as a project property value inclusion.
     **/
    public void setProperty(String property)
    {
        require_(property!=null,"setProp- nonzro name");
        m_xstring.set(property);
        m_xstring.setIsProperty(true);
        setKey(property);
    }


    /**
     * Marks this as a project reference value inclusion.
     **/
    public void setReference(String refId)
    {
        require_(refId!=null,"setRefId- nonzro name");
        m_xstring.set(refId);
        m_xstring.setIsReference(true);
        setKey(refId);
    }


    /**
     * Marks this as a project exported property value inclusion.
     **/
    public void setVariable(String variable)
    {
        require_(variable!=null,"setVar- nonzro name");
        m_xstring.set(variable);
        m_xstring.setIsExported(true);
        setKey(variable);
    }


    /**
     * Synonym for {@linkplain #setVariable setVariable()}.
     **/
    public final void setVar(String variable)
    {
        setVariable(variable);
    }


    /**
     * Marks this as a property set inclusion.
     * @param r existing property set reference (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setPropertySet(Reference r)
    {
        require_(r!=null,"setPropSet- nonzro ref");
        require_(r.getRefId()!=null,"setPropSet- nonzro refid");
        setKey(r.getRefId());
        m_Is = IS_PSET; //xclusive
    }


    /**
     * Marks this as a task-definition class name inclusion.
     **/
    public void setTaskdef(String taskName)
    {
        require_(taskName!=null,"setTaskdef- nonzro name");
        setKey(taskName);
        m_Is = IS_TASK; //xclusive
    }


    /**
     * Marks this as a type-definition class name inclusion.
     **/
    public void setTypedef(String typeName)
    {
        require_(typeName!=null,"setTypedef- nonzro name");
        setKey(typeName);
        m_Is = IS_TYPE; //xclusive
    }


    /**
     * Includes whatever this inclusion refers to the given
     * snapshot.
     * @param es the snapshot to be updated (non-null)
     **/
    public void apply(ErrorSnapshot es)
    {
        require_(es!=null,"apply- nonzro snapshot");
        if (!putFlexString(es)) {

            final Project P= getProject();
            final String item = getKey();
            Map mp;

            if (isPropertySet()) {
                es.addConfiguredPropertySet
                    ((PropertySet)FixtureExaminer.getReferencedObject
                     (P,null,item,PropertySet.class));
            }
            else if (isTaskDef()) {
                mp= P.getTaskDefinitions();
                if (Strings.ALL.equals(item)) {
                    putDefiners(es,mp,"TaskDef");
                } else {
                    putDefiner(es,item,(Class)mp.get(item),"TaskDef");
                }

            }
            else if (isTypeDef()) {
                mp= P.getDataTypeDefinitions();
                if (Strings.ALL.equals(item)) {
                    putDefiners(es,mp,"TypeDef");
                } else {
                    putDefiner(es,item,(Class)mp.get(item),"TypeDef");
                }
            }
        }
    }


    /**
     * Captures a single type or task definition to snapshot.
     **/
    private void putDefiner(ErrorSnapshot es, String key, Class c, String prefix)
    {
        if (c!=null) {
            es.setProperty(prefix+"."+key,c.getName());
        } else {
            es.setProperty(prefix+"."+key,null);
        }
    }


    /**
     * Captures all type or task definitions to snapshot.
     **/
    private void putDefiners(ErrorSnapshot es, Map mp, String prefix)
    {
        Iterator itr= mp.entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry mE= (Map.Entry)itr.next();
            if (mE!=null) {
                es.setProperty(prefix+"."+mE.getKey(),((Class)mE.getValue()).getName());
            }
        }
    }


    /**
     * If active, captures contents of flex string to snapshot.
     * Returns <i>false</i> if flex string was activated by any
     * setter method.
     **/
    private boolean putFlexString(ErrorSnapshot es)
    {
        if (!m_xstring.isUndefined()) {
            es.setProperty(m_xstring.get(), m_xstring.getValue(getProject()));
            return true;
        }
        return false;
    }


    /**
     * Returns <i>true</i> if this is task-definition inclusion.
     **/
    protected final boolean isTaskDef()
    {
        return (m_Is&IS_TASK)==IS_TASK;
    }


    /**
     * Returns <i>true</i> if this is type-definition inclusion.
     **/
    protected final boolean isTypeDef()
    {
        return (m_Is&IS_TYPE)==IS_TYPE;
    }


    /**
     * Returns <i>true</i> if this is &lt;propertyset&gt; inclusion.
     * @since JWare/AntX 0.4
     **/
    protected final boolean isPropertySet()
    {
        return (m_Is&IS_PSET)==IS_PSET;
    }



    private static final int IS_NONE=0;
    private static final int IS_TYPE=0x01;
    private static final int IS_TASK=0x02;
    private static final int IS_PSET=0x04;

    private String m_itemName;
    private int m_Is= IS_NONE;
    private DiagnosticsValue m_xstring = new DiagnosticsValue(true);//NB:most often done
}

/* end-of-EmitInclude.java */
