/**
 * $Id: EmitConfigureTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2003,2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.apis.DiagnosticsEmitter;
import  com.idaremedia.apis.DiagnosticsEmitterFactory;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.FixtureOverlay;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.apis.BuildError;
import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.helpers.Setting;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * A scoped configuration task that defines default emit configuration for its enclosed
 * tasks. The build script can inline the configuration definition using task parameters
 * or through a previously declared {@linkplain EmitConfiguration} object (to which the
 * task is made to refer).
 * <p>
 * <b>For example:</b><pre>
 *   &lt;<b>overlay-emit</b>
 *        from="builds.nightly.antx.compile"
 *        noiseLevel="verbose"
 *        echo="yes"
 *        timestamp="yes"
 *        properties="javac.debug,javac.opt,javac.depend,javac.classpath"
 *        emitterFactory="jware.antx.helpers.SilentEmitterFactory"
 *        timestampFormat="HH:MM:ss dd-MMM-YYYY"
 *        withDefaults="builds.nightly.log.configuration"&gt;
 *      &lt;emit msgid="BlastOff"/&gt;
 *      &lt;emitlogs&gt;
 *          ...
 *      &lt;/emitlogs&gt;
 *   &lt;/overlay-emit&gt;
 *
 * -OR- (better)
 *
 *   &lt;emitconfiguration id="antx.nightly.builds"
 *       from="builds.nightly.antx.compile".../&gt;
 *   &lt;<b>overlay-emit</b> with="antx.nightly.builds"&gt;
 *      &lt;emit msgid="BlastOff"/&gt;
 *      &lt;emitlogs&gt;
 *          ...
 *      &lt;/emitlogs&gt;
 *   &lt;/overlay-emit&gt;
 * </pre>
 * <b>Implementation Notes:</b><br>
 * EmitConfigureTasks change the current thread's iteration context. Their effect is
 * determined by the addivity features of nested emit-aware tasks; for example, the
 * default EmitTask will use an enclosing configuration to supply default settings
 * if its own parameters have not be explicitly defined.
 * <p>
 * Using the <i>defaultEmitterfactory</i> option effectively re-roots this configuration's
 * emitter to the root emitter returned by the custom factory. If a custom grouping
 * is also specified (<i>defaultFrom</i>), then this configuration's emitter is
 * obtained from the custom factory using the custom grouping. Any inherited emitter
 * is ignored.
 * <p>
 * Currently most of an EmitConfigureTask's attributes are latches-- they can be
 * modified repeatedly until the task is executed once. After the first execution,
 * subsequent attribute changes are basically ignored as the first execution creates
 * and caches several helpers that are never reset.
 * <p>
 * EmitConfigureTask is essentially a single-threaded object. Bad things will occur
 * if a nested task calls back to this task's target from either another thread or
 * from the same thread. In the case of a second thread, the second thread will be
 * blocked waiting for the perform-lock to be released from the first thread (which in
 * turn is waiting for the second thread to finish). In the case of a single thread
 * an infinite call-back loop would have been created. Both these conditions are captured
 * when a configure task is verified before execution; a BuildException is thrown iff
 * duplicate instances of the same configure would end up on the iteration stack.
 * <b>Note:</b> Nested parallel 'emit' tasks <em>can</em> happen-- accessing the
 * configuration state concurrently is fine, re-executing the configure task concurrently
 * is not.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2003,2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded (from EmitConfiguration interface once configured)
 * @.group   api,infra
 * @see      EmitTask
 * @see      DefaultEmitConfiguration
 **/

public class EmitConfigureTask extends TaskSet
    implements FixtureOverlay, EmitConfiguration, EmitConfigurable
{
    /**
     * Initializes a new EmitConfigureTask instance.
     **/
    public EmitConfigureTask()
    {
        super(AntX.feedback+"EmitConfigureTask:");
        m_ecImpl.setController(this);
    }


    /**
     * Initializes a new EmitConfigureTask subclass instance.
     **/
    protected EmitConfigureTask(String iam, boolean delayConfigure)
    {
        super(iam,delayConfigure);
        m_ecImpl.setController(this);
    }


    /**
     * Initializes the enclosing project of this task. Will
     * also update the internal project-dependent elements.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        m_ecImpl.setProject(P);
    }


//---------------------------------------------------------------------------------------------------------|
// Noise-level/Emitted-kruft management:
//---------------------------------------------------------------------------------------------------------|

    /**
     * Sets the list of property names automatically attached
     * to any emitted messages within this configurations scope.
     * @param nameList comma-delimited list of property names (non-null)
     **/
    public void setProperties(String nameList)
    {
        m_ecImpl.setProperties(nameList);
    }


    /**
     * Returns name list of properties automatically included in any
     * messages emitted within this configuration's scope. Returns
     * <i>null</i> (for none) if never set.
     **/
    public String getDefaultProperties()
    {
        return m_ecImpl.getOwnProperties();
    }


    /**
     * Adds any custom property names to list automatically included
     * with messages emitted within this configuration's scope. Returns
     * <i>false</i> (for none) if never nothing added. Uses comma as
     * delimiter.
     * @param list names buffer to update
     * @return <i>true</i> if list buffer modified
     **/
    public boolean getPropertiesNameList(StringBuffer list)
    {
        return m_ecImpl.getPropertiesNameList(list);
    }


    /**
     * Sets the default priority level of messages emitted within
     * this configuration's scope.
     * @param nl default message noise level (non-null)
     **/
    public void setNoiseLevel(NoiseLevel nl)
    {
        m_ecImpl.setNoiseLevel(nl);
    }


    /**
     * Returns this configuration's default message priority level.
     * Will return <i>null</i> if never set (this configuration's
     * emitter can be queried directly).
     **/
    public NoiseLevel getDefaultNoiseLevel()
    {
        return m_ecImpl.getOwnNoiseLevel();
    }


    /**
     * Returns this configuration's preferred noise level if one is
     * defined for it or any of its enclosing configurations.
     **/
    public NoiseLevel getNoiseLevel()
    {
        NoiseLevel nl= getDefaultNoiseLevel();
        if (nl==null) {
            EmitConfiguration defaults= getDefaults();
            //NB: We want callers to be able to slip project-based prefs between
            //    nothing(us) and the root-defaults for this attribute (ssmc)
            if (defaults!=DefaultEmitConfiguration.INSTANCE) {
                nl= defaults.getNoiseLevel();
            }
        }
        return nl;
    }


    /**
     * Sets whether any emissions within this configuration's scope should
     * also echo their messages to the standard Ant logging system.
     * @param wantIt preference setting (yes,no,inherit) (non-null)
     **/
    public void setEcho(String wantIt)
    {
        m_ecImpl.setEcho(wantIt);
    }


    /**
     * Returns the echo-to-Ant-log preference for this configuration.
     * Will return <i>Setting.INHERITED</i> if never set.
     **/
    public Setting getDefaultEcho()
    {
        return m_ecImpl.getOwnEcho();
    }


    /**
     * Returns <i>true</i> if emitted messages should echo their messages
     * to the standard Ant logging system. Result is based on this
     * configuration's settings and optionally its own inherited configuration.
     * @.safety multiple for nested tasks or after configured
     **/
    public boolean shouldEcho()
    {
        return m_ecImpl.shouldEcho();
    }

//---------------------------------------------------------------------------------------------------------|
// Grouping/Log4J Category management:
//---------------------------------------------------------------------------------------------------------|

    /**
     * Sets a custom emitter factory's grouping path separator.
     * Only useful if the emitter factory is also customized.
     * @since JWare/AntX 0.3
     **/
    public void setGroupingSeparator(String separator)
    {
        m_ecImpl.setGroupingSeparator(separator);
    }


    /**
     * Returns the default emitter grouping item separator for this
     * configuration.
     * @since JWare/AntX 0.3
     **/
    public String getGroupingPathSeparator()
    {
        return m_ecImpl.getGroupingPathSeparator();
    }


    /**
     * Sets the default grouping or log4j category of messages emitted by
     * tasks nested within this configuration's scope. By default there
     * is no scope and everything goes to the root emitter; this method
     * sets this configuration's category explicitly.
     * @param grpId this configuration's new default category
     **/
    public void setFrom(String grpId)
    {
        m_ecImpl.setFrom(grpId);
    }


    /**
     * Returns the grouping or log4j category of this configuration.
     * Will return <i>null</i> if no explicit scope defined.
     * @since JWare/AntX 0.3
     **/
    public String getDefaultFrom()
    {
        return m_ecImpl.getOwnFrom();
    }


    /**
     * Returns the calculated grouping for messages emitted by tasks
     * nested within this configuration's scope.
     **/
    public String getFrom()
    {
        return m_ecImpl.getFrom();
    }


    /**
     * Instructs this configuration to calculate its grouping relative
     * to something else.
     * @param wrt the something else (non-null)
     * @since JWare/AntX 0.3
     **/
    public void setWrt(String wrt)
    {
        require_(wrt!=null,"setDfltWrt- nonzro wrt");
        m_ecImpl.setWrt(wrt);
    }


    /**
     * Returns what this configuration's grouping is relative to.
     * Will return <i>null</i> if no explicit instruction defined
     * @since JWare/AntX 0.3
     **/
    public String getDefaultWrt()
    {
        return m_ecImpl.getOwnWrt();
    }

//---------------------------------------------------------------------------------------------------------|
// Emitter(Logger) management:
//---------------------------------------------------------------------------------------------------------|

    /**
     * Sets the custom emitter factory used by this configuration. By default
     * a log4j-based emitter factory is used.
     * @param emitterFactoryClass custom emitter factory implementation (non-null)
     * @throws BuildException if unable to create a DiagnosticsEmitterFactory
     *         instance from given class instance
     **/
    public void setEmitterFactory(Class emitterFactoryClass)
    {
        m_ecImpl.setEmitterFactory(emitterFactoryClass);
    }


    /**
     * Returns this configuration's custom emitter factory. Will return
     * <i>null</i> if a custom emitter factory has not be defined.
     * @see LJDiagnosticsEmitter#FACTORY
     **/
    public DiagnosticsEmitterFactory getDefaultEmitterFactory()
    {
        return m_ecImpl.getOwnEmitterFactory();
    }


    /**
     * Returns this configuration's default emitter. Cached on execution
     * for this taskset's scope.
     * @see #setFrom
     * @.safety multiple for nested tasks or after configured
     **/
    public DiagnosticsEmitter getEmitter()
    {
        ensure_(m_emitr!=null,"getEmtr- been exec'd at least once");
        return m_emitr;
    }


    /**
     * Returns a custom diagnostics emitter for specified grouping. If
     * this configuration has a custom emitter factory it is used to produce
     * the new emitter; otherwise, this configuration's inherited emitter
     * factory is used.
     * @param grpId category's name (non-null)
     * @.safety multiple for nested tasks or after configured
     **/
    public DiagnosticsEmitter getCustomEmitter(String grpId)
    {
        return m_ecImpl.getCustomEmitter(grpId);
    }


    /**
     * Setup this configuration's default emitter. Depending on whether
     * a custom emitter factory and a default 'from' have been defined,
     * this configuration's default emitter is either inherited or a
     * custom instance.
     **/
    private void setupDefaultEmitter()
    {
        EmitConfiguration ec = ignoreAllInherited() ? null : m_inheritedInstance;
        m_ecImpl.setDefaultConfiguration(ec);
        m_emitr = m_ecImpl.getEmitter();
    }


//---------------------------------------------------------------------------------------------------------|
// Timestamp management:
//---------------------------------------------------------------------------------------------------------|

    /**
     * Sets the default format of timestamps emitted by tasks nested within this
     * configuration's scope. An {@linkplain EmitTask <i>emit</i>} task can
     * define its own format as part of its definition to override this setting.
     * @param format new timestamp format (non-null)
     **/
    public void setTimestampFormat(String format)
    {
        m_ecImpl.setTimestampFormat(format);
    }


    /**
     * Returns the default format of timestamps emitted by tasks nested within
     * this task's scope. Will return <i>null</i> if not set.
     **/
    public final String getDefaultTimestampFormat()
    {
        return m_ecImpl.getOwnTimestampFormat();
    }


    /**
     * Creates readable date-time string of given timestamp (milliseconds).
     * If this configuration's timestamp format has been set, the formatted
     * string will be in this format; otherwise, a standard MEDIUM date/time
     * formatted string is returned.
     * @.safety guarded for nested tasks or after configured
     **/
    public String stampify(long ms)
    {
        return m_ecImpl.stampify(ms);
    }


    /**
     * Sets the include-timestamp preference for this configuration.
     * @param wantIt preference setting (yes,no,inherit) (non-null)
     **/
    public void setTimestamp(String wantIt)
    {
        m_ecImpl.setTimestamp(wantIt);
    }


    /**
     * Returns the include-timestamp preference for this configuration.
     * Returns <i>Setting.INHERITED</i> if never set.
     **/
    public final Setting getDefaultIncludeTimestamp()
    {
        return m_ecImpl.getOwnTimestamp();
    }


    /**
     * Returns <i>true</i> if timestamp information should be included in
     * any emitted messages. Result is based on this configuration's settings
     * and optionally its own inherited configuration.
     * @see #getDefaultIncludeTimestamp
     * @.safety multiple for nested tasks or after configured
     **/
    public boolean wantTimestamp()
    {
        return m_ecImpl.wantTimestamp();
    }


//---------------------------------------------------------------------------------------------------------|
// Taskset/controller management:
//---------------------------------------------------------------------------------------------------------|

    /**
     * Returns <i>true</i> if this configuration ignores its inherited
     * configuration and returns its top-level defaults configuration.
     * Is <i>false</i> by default.
     * @see DefaultEmitConfiguration
     **/
    protected final boolean ignoreAllInherited()
    {
        return m_noInherited;
    }


    /**
     * Set whether this configuration will use the configuration
     * hierarchy or short-circuit directly to its top-level defaults
     * configuration.
     * @param passthru <i>false</i> if this configuration should ignore
     *        the preceding configuration hierarchy.
     **/
    public void setInheritance(boolean passthru)
    {
        m_noInherited = !passthru;
    }


    /**
     * Set this configuration task to use a previously defined
     * EmitConfiguration data object as it <em>full</em> definition.
     * @param ecImplRef the configuration reference (non-null)
     * @since JWare/AntX 0.3
     **/
    public void setWith(Reference ecImplRef)
    {
        require_(ecImplRef!=null,"setRef- nonzro ref");
        m_ecImpl.setRefid(ecImplRef);
        m_ecImplReference = ecImplRef;
    }


    /**
     * Returns this configuration's delegate EmitConfiguration
     * reference. Returns <i>null</i> if no such reference ever set.
     * @since JWare/AntX 0.3
     **/
    public Reference getWith()
    {
        return m_ecImplReference;
    }


    /**
     * Initializes this configuration's own defaults to an existing
     * configuration's reference. Iff this configuration finds no
     * enclosing context configuration, will it use this configuration's
     * information. The reference id is evaluated at this task's
     * execution time.
     * @param dfltRef reference id of an EmitConfiguration (non-null)
     **/
    public void setWithDefaults(Reference dfltRef)
    {
        m_ecImpl.setDefaults(dfltRef);
    }


    /**
     * Returns the reference of this configuration's default settings.
     * Returns <i>null</i> if never set.
     **/
    public final Reference getWithDefaults()
    {
        return m_ecImpl.getOwnDefaults();
    }


    /**
     * Verifies this configuration's references (if any defined).
     * Ensures the refid refer to valid EmitConfiguration implementations.
     * @since JWare/AntX 0.3
     **/
    protected final void verifyReferences()
    {
        if (getWith()!=null) {
            EmitConfiguration definition = (EmitConfiguration)getReferencedObject
                (getProject(), getWith().getRefId(), EmitConfiguration.class);
            if (definition==this) {//Whups!
                String ermsg = uistrs().get("emit.circular.defaults");
                log(ermsg, Project.MSG_ERR);
                throw new BuildException(ermsg,getLocation());
            }
        }
        if (getWithDefaults()!=null) {
            EmitConfiguration defaults = (EmitConfiguration)getReferencedObject
                (getProject(), getWithDefaults().getRefId(), EmitConfiguration.class);
            if (defaults==this) {//Whups!
                String ermsg = uistrs().get("emit.circular.defaults");
                log(ermsg, Project.MSG_ERR);
                throw new BuildException(ermsg,getLocation());
            }
        }
    }


    /**
     * Returns this task's closest inherited configuration or, if none,
     * its default configuration. Can inherit from enclosing &ltemitconfigure&gt;
     * taskset or the {@linkplain EmitContext#getDefaultConfiguration default
     * configuration}.
     * @see #setWithDefaults
     * @.safety multiple for nested tasks or after configured
     **/
    public EmitConfiguration getDefaults()
    {
        EmitConfiguration ec = ignoreAllInherited() ? null : m_inheritedInstance;
        if (ec==null) {
            ec= m_ecImpl.getDefaults();
        }
        return ec;
    }



    /**
     * Installs this configuration as the frontmost configuration within
     * current thread's iteration environment before running nested tasks.
     * When tasks have finished (or failed), uninstalls self from
     * context stack.
     * @throws BuildError if emit-context stack becomes corrupted
     * @.safety guarded
     **/
    protected void performNestedTasks()
        throws BuildException
    {
        synchronized(m_runlock) {
            boolean installed=false;
            try {
                //NB: The ordering is critical here-- must setup our emitter
                //    *before* we install ourselves; otherwise an infinite loop
                //    can occur during emitter resolution if 'wrt' is "enclosing" (ssmc)
                m_inheritedInstance= EmitContext.getConfiguration();
                setupDefaultEmitter();
                EmitConfiguration cur= EmitContext.installConfiguration(this,m_problemHandler);
                installed=true;
                verify_(cur==m_inheritedInstance,"perform- same defaults");
                super.performNestedTasks();

            } finally {
                if (installed) {
                    m_inheritedInstance = null;
                    EmitConfiguration ec = EmitContext.getConfiguration();
                    if (ec!=this) {
                        String ME= uistrs().dget("task.emit.whoami","EmitConfiguration");
                        String ermsg = uistrs().get("context.stack.corrupted",ME);
                        m_problemHandler.problem(ermsg, Project.MSG_ERR);
                        throw new BuildError(ermsg,getLocation());
                    }
                    EmitContext.unwindConfiguration(m_problemHandler);
                }//installed
            }
        }
    }


    /**
     * Verifies in valid target/project and pre-loads any defaults
     * reference objects (latch) to ensure concurrent access by
     * nested-tasks is OK.
     **/
    protected synchronized void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        verifyReferences();
    }



    private EmitConfigurationType m_ecImpl= new EmitConfigurationType();
    private Object m_runlock= new int[0];
    private boolean m_noInherited;//NB:=>use inheritance hierarchy
    private EmitConfiguration m_inheritedInstance;//NB:setup by 'perform'
    private DiagnosticsEmitter m_emitr;//NB:setup by 'perform'
    private Reference m_ecImplReference;//NB:=>inlined-definition


    private ProblemHandler m_problemHandler = new ProblemHandler()
    {//NB:must be safe w/o context
        public void problem(Object nugget, int weight)
        {
            EmitConfigureTask.this.log(String.valueOf(nugget), weight);
            Emit.broadcast(EmitConfigureTask.this.getDefaults().getEmitter(),
                           nugget,null,NoiseLevel.fromNativeIndex(weight));
        }
    };
}

/* end-of-EmitConfigureTask.java */
