/**
 * $Id: IsSet.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  com.idaremedia.antx.parameters.Handling;
import  com.idaremedia.antx.parameters.IgnoreWhitespaceEnabled;
import  com.idaremedia.antx.parameters.IsA;
import  com.idaremedia.antx.parameters.MalformedCheckEnabled;

/**
 * Simple Is-Set condition check that supports references and AntX variables. Example
 * usage: <pre>
 *   &lt;isset property="jdk14.present"/&gt;
 *   &lt;isset property="module.id" whitespace="reject"/&gt;
 *   &lt;isset variable="__loopcount"/&gt;
 *   &lt;isset reference="build.num"/&gt;
 *   &lt;isset property="build.ROOT" whitespace="reject" malformed="reject"/&gt;
 *</pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 **/

public class IsSet extends SimpleFlexCondition
    implements IgnoreWhitespaceEnabled, MalformedCheckEnabled
{
    /**
     * Initializes new IsSet condition. A value type must be
     * specified before this condition is evaluated.
     **/
    public IsSet()
    {
        super();
    }


    /**
     * Initializes a fully-defined IsSet condition.
     * @param property the property against which condition checks
     **/
    public IsSet(String property)
    {
        setProperty(property);
    }


    /**
     * Initializes an IsSet condition to either a property or
     * a variable value check.
     * @param value property or variable against which condition checks
     * @param isP <i>true</i> if <span class="src">value</span> is
     *             property's name
     **/
    public IsSet(String value, boolean isP)
    {
        if (isP) {
            setProperty(value);
        } else {
            setVariable(value);
        }
    }



    /**
     * Tells this condition whether a property containing only
     * whitespace is considered "not set". If the choice is
     * one of "ignore", "reject", or "balk" then an all-whitespace
     * property are considered not set.
     **/
    public void setWhitespace(Handling response)
    {
        getValueHelper().setWhitespace(response);
    }


    /**
     * Returns <i>true</i> if values of all whitespace will be
     * ignored (as if not set). Defaults <i>false</i>.
     **/
    public final boolean ignoreWhitespace()
    {
        return getValueHelper().isIgnoreWhitespace();
    }


    /**
     * Tells this condition whether properties must be completely
     * resolved to be considered "set". If the choice is
     * either "ignore" or "accept" then unresolved property
     * values are considered okidoki (set).
     * @since JWare/AntX 0.4
     **/
    public void setMalformed(Handling response)
    {
        // malformed="accept|ignore"
        if (Handling.isYes(response,Handling.REJECT)) {
            m_checkUnresolved= false;
        }
        // malformed="reject|balk|inherit"
        else {
            m_checkUnresolved= true;
        }
    }


    /**
     * Returns how this condition will handle unresolved
     * properties. Returnes either "accept" or "reject".
     * @since JWare/AntX 0.4
     **/
    public Handling getMalformedHandling()
    {
        return m_checkUnresolved ? Handling.REJECT : Handling.ACCEPT;
    }



    /**
     * Marks this condition's property as the default field
     * set by a value URI.
     * @since JWare/AntX 0.5
     * @return IsA.PROPERTY always.
     */
    protected IsA getDefaultIsAForURI()
    {
        return IsA.PROPERTY;
    }



    /**
     * Returns <i>true</i> if this condition's property, reference,
     * or variable is explicitly set.
     **/
    public boolean eval()
    {
        verifyCanEvaluate_("eval");

        if (isProperty()) {
            if (m_checkUnresolved) {
                if (!LocalPropertyExaminer.verifyProperty
                    (getProject(), getValueHelper())) {
                    return false;
                }
            }
        }
        else if (isReference()) {
            Object o = getProject().getReference(getResolvedFlexValue());
            if (!(o instanceof String)) {//otherwise apply modifiers(ssmc)
                return o!=null;
            }
        }

        return getValueHelper().getValue()!=null;
    }


    private boolean m_checkUnresolved;//NB:=> even unresolved=="set"
}

/* end-of-IsSet.java */
