/**
 * $Id: IsNumeric.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2003 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  com.idaremedia.antx.helpers.Tk;

/**
 * Simple Is-Numeric condition check. IsNumeric can handle any number up to
 * Long.MAX_VALUE which means it can be used for timestamps as well as simplier (smaller)
 * numbers.
 * <p>
 * <b>Examples:</b><pre>
 *   &lt;isnumeric value="${build.number}"/&gt;
 *   &lt;isnumeric property="build.ITID"/&gt;
 *
 *   &lt;isnumeric variable="_loop.count"/&gt;
 *   &lt;isnumeric variable="_try.count" gte="0" lt="3"/&gt;
 *
 *   &lt;isnumeric variable="_totaltime" gte="0"/&gt;
 *</pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2003 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 **/

public class IsNumeric extends SimpleFlexCondition
{
    /**
     * Initializes a new IsNumeric condition.
     **/
    public IsNumeric()
    {
        super();
    }


    /**
     * Creates defined condition.
     * @param value the value against which condition checks
     **/
    public IsNumeric(String value)
    {
        setValue(value);
    }


    /**
     * Sets this condition to evaluate a literal value as-is.
     * @param value the literal value to check
     **/
    public void setValue(String value)
    {
        require_(value!=null,"setValu- nonzro");
        setLiteral(value);
    }


    /**
     * Sets a "greater-than" limit requirement on checked
     * value.
     **/
    public void setGT(long gt)
    {
        m_GT = gt;
        m_isLimited = true;
    }


    /**
     * Sets a "greater-than-or-equal" limit requirement on
     * checked value.
     **/
    public void setGTE(long gte)
    {
        m_GTE = gte;
        m_isLimited = true;
    }


    /**
     * Sets a "less-than" limit requirement on checked
     * value.
     **/
    public void setLT(long lt)
    {
        m_LT = lt;
        m_isLimited = true;
    }


    /**
     * Sets a "less-than-or-equal" limit requirement on checked
     * value.
     **/
    public void setLTE(long lte)
    {
        m_LTE = lte;
        m_isLimited = true;
    }


    /**
     * Returns <i>true</i> if this condition's value is a valid
     * number (and optionally within a specified range).
     **/
    public boolean eval()
    {
        String value = getValueHelper().getValue();

        long l= Tk.longFrom(value,NO_VALU);

        boolean ok = l!=NO_VALU;

        if (ok && m_isLimited) { //painful to look at but it works
            if (m_GTE!=NO_VALU) {
                if (l<m_GTE) {
                    return false;
                }
            }
            if (m_GT!=NO_VALU) {
                if (l<=m_GT) {
                    return false;
                }
            }
            if (m_LTE!=NO_VALU) {
                if (l>m_LTE) {
                    return false;
                }
            }
            if (m_LT!=NO_VALU) {
                if (l>=m_LT) {
                    return false;
                }
            }
        }

        return ok;
    }


    private static final long NO_VALU=Long.MIN_VALUE;
    private boolean m_isLimited;
    private long m_GTE=NO_VALU,m_GT=NO_VALU;
    private long m_LTE=NO_VALU,m_LT=NO_VALU;
}

/* end-of-IsNumeric.java */
