/**
 * $Id: IsBoolean.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.IgnoreCaseEnabled;
import  com.idaremedia.antx.parameters.SynonymsEnabled;

/**
 * Simple Is-A-Boolean condition check. Example usage:
 * <pre>
 *   &lt;isboolean value="${junit.present}"/&gt;
 *   &lt;isboolean property="is.dist.build"/&gt;
 *   &lt;isboolean variable="defaults.disabled" synonyms="off"/&gt;
 *</pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 **/

public final class IsBoolean extends SimpleFlexCondition
    implements IgnoreCaseEnabled, SynonymsEnabled
{
    /**
     * Initializes new IsBoolean condition. A value type must be
     * specified before this condition is evaluated.
     **/
    public IsBoolean()
    {
        super();
    }


    /**
     * Initializes a fully-defined IsBoolean condition.
     * @param property the property against which condition checks
     **/
    public IsBoolean(String property)
    {
        setProperty(property);
    }


    /**
     * Initializes an IsBoolean condition to either a property or
     * a variable value check.
     * @param value the property or varaible against which condition checks
     * @param isP <i>true</i> if <code>value</code> is property's name
     **/
    public IsBoolean(String value, boolean isP)
    {
        if (isP) {
            setProperty(value);
        } else {
            setVariable(value);
        }
    }


    /**
     * Sets whether value should be lowercased before checked.
     **/
    public void setIgnoreCase(boolean ignore)
    {
        getValueHelper().setIgnoreCase(ignore);
    }


    /**
     * Returns <i>true</i> if this condition will ignore value's
     * case when it's checked. Defaults <i>false</i>.
     **/
    public final boolean isIgnoreCase()
    {
        return getValueHelper().isIgnoreCase();
    }


    /**
     * Sets whether value will be checked for boolean synonyms
     * like "<i>off</i>" or "<i>yes</i>". Defaults <i>on</i>.
     * @param allowAll <i>true</i> if synonyms are enabled (default)
     **/
    public void setSynonyms(boolean allowAll)
    {
        m_allowAll = allowAll;
    }


    /**
     * Returns <i>true</i> if value allowed to be a boolean synonyms.
     * Defaults <i>true</i>.
     **/
    public final boolean allowSynonyms()
    {
        return m_allowAll;
    }



    /**
     * Sets this condition to evaluate a literal value as-is.
     * @param value the literal value to check
     **/
    public void setValue(String value)
    {
        require_(value!=null,"setValu- nonzro");
        setLiteral(value);
    }



    /**
     * Returns <i>true</i> if this condition's property, reference,
     * or variable value matches a boolean value name (explicitly).
     **/
    public boolean eval()
    {
        verifyCanEvaluate_("eval");

        String value = getValueHelper().getValue();
        if (value==null) {
            return false;
        }

        boolean isbool;
        if (allowSynonyms()) {
            isbool = Tk.string2PosBool(value).booleanValue() ||
                !Tk.string2NegBool(value).booleanValue();
        } else {
            isbool = Strings.TRUE.equals(value) ||
                Strings.FALSE.equals(value);
        }
        return isbool;
    }


    private boolean m_allowAll=true;
}

/* end-of-IsBoolean.java */
