/**
 * $Id: AnySet.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  java.util.Iterator;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.Handling;

/**
 * Shortcut condition that returns <i>true</i> if at least one of the nested items is
 * set in the project's environment. Empty sets always evaluate <i>false</i>.
 * <p>
 * <b>Example Usage:</b><pre>
 *   &lt;tally&gt;
 *     &lt;available classname="junit.framework.Assert" property="junit.present"/&gt;
 *     &lt;available classname="com.idaremedia.pet.PET" property="pet.present"/&gt;
 *     &lt;<b>isanyset</b> trueproperty="run.tests.enable"&gt;
 *       &lt;property name="junit.present/&gt;
 *       &lt;property name="pet.present/&gt;
 *     &lt;/isanyset&gt;
 *   &lt;/tally&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 * @see      AllSet
 * @see      NoneSet
 **/

public class AnySet extends CheckSetCondition implements URIable
{
    /**
     * Initializes new AnySet condition. Will return <i>false</i>
     * if evaluated and still empty.
     **/
    public AnySet()
    {
        super(AntX.rules+"anyset");
    }



    /**
     * Initializes a filled-in AnySet instance.
     * @param properties comma-delimited list of properties (non-null)
     * @param P condition's project
     **/
    public AnySet(String properties,Project P)
    {
        super(AntX.rules+"anyset",properties,P);
    }



    /**
     * Returns <i>true</i> if any of the condition's items are
     * defined.
     **/
    public boolean eval()
    {
        verifyInProject_("eval");

        final Project P = getProject();
        boolean istrue = false;//=> empty=failure, noneset=failure

        if (!getIncludes().isEmpty()) {

            boolean ignoreWS = ignoreWhitespace();
            boolean iffTrue  = isTruesOnly();
            boolean checkUnresolved = getMalformedHandling()==Handling.REJECT;
            Iterator itr = getIncludesIterator();

            while (itr.hasNext()) {
                FlexString xv = (FlexString)itr.next();
                String v = null;

                if (xv.isProperty() && checkUnresolved) {
                    if (LocalPropertyExaminer.verifyProperty(P, xv)) {
                        v = xv.getValue(P);
                    }
                } else {
                    v = xv.getValue(P);
                }
                if (v!=null) {
                    if (ignoreWS && Tk.isWhitespace(v)) {
                        v = null;
                    } else if (iffTrue && Tk.string2PosBool(v)!=Boolean.TRUE) {
                        v = null;
                    }
                }
                if (v!=null) {
                    istrue=true;
                    break;
                }
            }
        }

        if (istrue && getUpdateProperty()!=null) {
            String what = Strings.TRUE;
            if (isUpdateVariable()) {
                ExportedProperties.set(getUpdateProperty(),what);
            } else {
                getProject().setNewProperty(getUpdateProperty(),what);
            }
        }

        return istrue;
    }



    /**
     * Sets this condition's list of properties as part of 
     * a value URI.
     * @param fragment the value uri bits (non-null)
     * @since JWare/AntX 0.5
     */
    public void xsetFromURI(String fragment)
    {
        xsetFromURIFragment(fragment);
    }
}

/* end-of-AnySet.java */
