/**
 * $Id: CopyLoggedTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.capture;

import  java.io.BufferedOutputStream;
import  java.io.File;
import  java.io.FileOutputStream;
import  java.io.IOException;
import  java.io.OutputStream;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.parameters.FlexInformationSaver;
import  com.idaremedia.antx.parameters.TransformHelper;

/**
 * Helper task that transfers captured logs to a file, project property, or project
 * reference. Copy-to destinations can be specified in a single &lt;copylogged&gt;
 * declaration. The source of copied information is either the ant logs (default) or
 * the system output streams (see "{@linkplain LogsUsingTask#setFrom from}" attribute).
 * <p>
 * <b>Example Usage:</b><pre>
 *   &lt;capturelogs importantfrom="warning"&gt;
 *     &lt;show message="Whoa"/&gt;
 *     &lt;show message="Whoa" level="error"/&gt;
 *     &lt;copylogged toproperty="one.whoa"/&gt;
 *     &lt;assert equals="Whoa" property="one.whoa"/&gt;
 *     &lt;copylogged toproperty="two.whoa" important="no"/&gt;
 *     &lt;assert startswith="WhoaWhoa" property="two.whoa"/&gt;
 *   &lt;/capturelogs&gt;
 *
 *   &lt;capturelogs importantfrom="info"&gt;
 *     &lt;javadoc &#46;&#46;&#46;/&gt;
 *     &lt;copylogged tofile="${scratch.root}/javadoc.out"/&gt;
 *   &lt;/capturelogs&gt;
 *
 *   &lt;captureoutput&gt;
 *     &lt;<i>Your Tasks Here</i>&#46;&#46;&#46;/&gt;
 *     &lt;copylogged <b>from="stdio"</b> toreference="tasks.stdio.out"/&gt;
 *   &lt;/captureoutput&gt;
 *</pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 * @see      CaptureLogsTask
 **/

public final class CopyLoggedTask extends LogsUsingTask
    implements FlexInformationSaver
{
    /**
     * Initializes a new CopyLoggedTask instance.
     **/
    public CopyLoggedTask()
    {
        super(AntX.capture+"CopyLoggedTask:");
    }


    /**
     * Initializes a CV-labeled CopyLoggedTask instance.
     * @param iam CV-label (non-null)
     **/
    public CopyLoggedTask(String iam)
    {
        super(iam);
    }

// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets the target file for the copied logs. If the given
     * path is relative, the output file will be determined
     * relative to this task's project's base directory.
     * @param fileRef the output file's path or file URL (non-null)
     **/
    public void setToFile(String fileRef)
    {
        require_(fileRef!=null,"setToFil- nonzro fileref");
        fileRef = TransformHelper.toPath(fileRef,getProject());
        m_toFile= new File(fileRef);

    }


    /**
     * Sets the target file for the copied logs. File reference
     * is used as-is.
     * @param outputFile the output file (non-null)
     * @since JWare/AntX 0.4
     **/
    public final void setToProjectFile(File outputFile)
    {
        require_(outputFile!=null,"setToProjFil- nonzro file");
        m_toFile= outputFile;
    }


    /**
     * Returns this task's target file for output. Returns
     * <i>null</i> if never set.
     * @since JWare/AntX 0.4
     **/
    public final String getToFilePath()
    {
        File f = getToFile();
        return f!=null ? f.getPath() : null;
    }


    /**
     * Returns this target file for output. Returns <i>null</i>
     * if never set.
     **/
    public File getToFile()
    {
        return m_toFile;
    }


    /**
     * Marks whether this task should try to append its information
     * to its output stream.
     * @param append <i>true</i> if should try to append
     **/
    public void setAppend(boolean append)
    {
        m_tryAppend= append;
    }


    /**
     * Returns <i>true</i> if this task will try to append any new
     * information to an existing output sink. By default is
     * <i>false</i>.
     **/
    public final boolean willTryAppend()
    {
        return m_tryAppend;
    }


    /**
     * Sets the property to which logs copied.
     * @param property the property to create (non-null)
     **/
    public void setToProperty(String property)
    {
        require_(property!=null,"setPropty- nonzro nam");
        m_updateProperty = property;
    }


    /**
     * Returns the name of the property this task will set with
     * the captured log information. Returns <i>null</i> if
     * never set.
     * @see #setToProperty setToProperty
     **/
    public String getToProperty()
    {
        return m_updateProperty;
    }


    /**
     * Sets the reference to which logs copied.
     * @param name the name of the (new) reference to update (non-null)
     **/
    public void setToReference(String name)
    {
        require_(name!=null,"setRef- nonzro nam");
        m_updateReference = name;
    }


    /**
     * Returns the name of the reference this task will update with
     * the captured log information. Returns <i>null</i> if
     * never set.
     * @see #setToReference setToReference
     **/
    public String getToReference()
    {
        return m_updateReference;
    }


// ---------------------------------------------------------------------------------------
// Execution:
// ---------------------------------------------------------------------------------------

    /**
     * Returns an output stream base on this task's <i>file</i>
     * and <i>append</i> parameters. Will return <i>null</i> if the
     * file parameter never set. The returned stream is the raw
     * file output stream; it is <em>not</em> a buffered stream.
     **/
    protected OutputStream getOutputStream()
    {
        File f = getToFile();
        if (f!=null) {
            try {
                return new FileOutputStream(f.getPath(), willTryAppend());
            } catch(IOException ioX) {
                String ermsg = uistrs().get("task.echo.unable.use.file",
                                            f.getPath(), ioX.getMessage());
                log(ermsg,Project.MSG_ERR);
                throw new BuildException(ermsg,getLocation());
            }
        }
        return null;
    }



    /**
     * Tries to copy this nearest task's enclosing logs recorder current
     * contents to a file and/or project property.
     * @throws BuildException if neither a file nor a property specified
     *          or unable to write information to file
     **/
    public void execute()
    {
        verifyCanExecute_("execute");

        String info = getVUTLog();

        if (getToProperty()!=null) {
            checkIfProperty_(getToProperty(), true);
            getProject().setNewProperty(getToProperty(),info);
        }

        if (getToReference()!=null) {
            checkIfReference_(getToReference(),true);
            getProject().addReference(getToReference(),info);
        }

        OutputStream fos = getOutputStream();
        if (fos!=null) {
            try {
                BufferedOutputStream bos = new BufferedOutputStream(fos, 512);
                byte[] raw = info.getBytes();
                bos.write(raw,0,raw.length);
                bos.flush();
                raw = null;
            } catch(IOException ioX) {
                String ermsg = uistrs().get("task.echo.unable.use.file",
                                            getToFile().getPath(),
                                            ioX.getMessage());
                log(ermsg,Project.MSG_ERR);
                throw new BuildException(ermsg,getLocation());
            } finally {
                try { fos.close(); } catch(IOException igx) {/*burp*/}
            }
        }

        if (willReset()) {
            LogsRecorder r = getRecorder(true);
            if (r!=null) {
                r.clearLogs();
            }
        }
    }


    /**
     * Verify this task is enclosed by at least one {@linkplain LogsRecorder
     * logs recording} taskset and has a fully defined condition.
     * @throws BuildException if neither a value or a sequence of values defined
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        String pn = getToProperty();

        if (getToFile()==null && pn==null) {
            String ermsg = getAntXMsg("task.needs.this.attr",getTaskName(),
                                      "tofile|toproperty");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        if (pn!=null) {
            checkIfProperty_(pn,true);
        }
    }

    private String m_updateProperty, m_updateReference;
    private File m_toFile;
    private boolean m_tryAppend;//NB:false=>overwrite existin'
}

/* end-of-CopyLoggedTask.java */
