/**
 * $Id: StringEquality.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;
import  org.apache.tools.ant.types.RegularExpression;
import  org.apache.tools.ant.util.regexp.RegexpMatcher;

import  com.idaremedia.antx.parameters.IgnoreCaseEnabled;
import  com.idaremedia.antx.parameters.TrimEnabled;
import  com.idaremedia.antx.parameters.ValueMatchEnabled;

/**
 * Helper condition that can replace the standard Ant <span class="src">&lt;equals&gt;</span>
 * condition with something thats works better with the default AntX &lt;assert&gt; inlined
 * <span class="src">equals</span> and <span class="src">notequals</span> options.
 * Note that this object is not intended as a script-facing nestable condition element;
 * however, instances of this class can be dynamically configured with a internal
 * <span class="src">RuntimeConfigurable</span> helper.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 * @see      FlexString
 **/

public final class StringEquality extends AssertableProjectComponent
    implements Condition, IgnoreCaseEnabled, TrimEnabled, ValueMatchEnabled
{
    /** Operation name for default 'equals' condition. **/
    public static final String OP_EQUALS= "equals";
    /** Operation name for 'startswith' condition. **/
    public static final String OP_STARTSWITH= "startswith";
    /** Operation name for 'endswith' condition. **/
    public static final String OP_ENDSWITH= "endswith";
    /** Operation name for 'contains' condition. **/
    public static final String OP_CONTAINS= "contains";
    /** Operation name for 'matches' condition. **/
    public static final String OP_MATCHES= "matches";


    /**
     * Initializes new string matching condition.
     **/
    public StringEquality()
    {
        super();
        m_knownArg.setLenient(false);
        m_unknownArg.setLenient(false);
    }


    /**
     * Initializes new string matching condition with known
     * argument specified.
     **/
    public StringEquality(String known)
    {
        this();
        setKnownArg(known);
    }


    /**
     * Initializes new string matching condition with known
     * argument optionally specified as a property name.
     * @param isP <i>true</i> if value is a property name
     **/
    public StringEquality(String known, boolean isP)
    {
        this();
        setKnownArg(known,isP);
    }


    /**
     * Initializes the enclosing project of this component.
     * Updates the internal arguments also.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        m_knownArg.setProject(P);
        m_unknownArg.setProject(P);
    }


    /**
     * Returns <i>true</i> if this equality is missing either
     * one or both arguments.
     **/
    public final boolean isUndefined()
    {
        return m_knownArg.isUndefined() || m_unknownArg.isUndefined();
    }


    /**
     * Set this condition's known or first argument to as-is string.
     **/
    public final void setKnownArg(String arg1)
    {
        setKnownArg(arg1,false);
    }


    /**
     * Set this condition's known or first argument with option to
     * specify value is a property name (which is determined at
     * evaluation time).
     * @param isP <i>true</i> if is a property name
     **/
    public void setKnownArg(String arg1, boolean isP)
    {
        m_knownArg.set(arg1);
        m_knownArg.setIsLiteral();
        m_knownArg.setIsProperty(isP);
    }


    /**
     * Returns this condition's known argument; returns <i>null</i>
     * if never set.
     * @see #getKnownValue
     **/
    public final String getKnownArg()
    {
        return m_knownArg.get();
    }


    /**
     * Returns the processed known argument's value as it's used in
     * comparision. This value has all the pre-processing instructions
     * applied (like trimming, lower-casing, etc.)
     **/
    public final String getKnownValue()
    {
        return m_knownArg.getValue();
    }


    /**
     * Script-facing delegatee for {@linkplain #setKnownArg
     * setKnownArg(String)}.
     * @since JWare/AntX 0.4
     **/
    public final void setMatch(String pattern)
    {
        setKnownArg(pattern,false);
        setOperator(OP_MATCHES);
    }


    /**
     * Set this condition's unknown or second argument to as-is string.
     **/
    public final void setUnknownArg(String arg2)
    {
        setUnknownArg(arg2, false);
    }


    /**
     * Set this condition's unknown or second argument with option to
     * specify value is a property name (which is determined at
     * evaluation time).
     * @param isP <i>true</i> if is a property name
     **/
    public void setUnknownArg(String arg1, boolean isP)
    {
        m_unknownArg.set(arg1);
        m_unknownArg.setIsLiteral();
        m_unknownArg.setIsProperty(isP);
    }


    /**
     * Returns this condition's unknown argument; returns <i>null</i>
     * if never set.
     * @see #getUnknownValue
     **/
    public final String getUnknownArg()
    {
        return m_unknownArg.get();
    }


    /**
     * Returns the processed unknown argument's value as it's used in
     * comparision. This value has all the pre-processing instructions
     * applied (like trimming, lower-casing, etc.)
     **/
    protected final String getUnknownValue()
    {
        return m_unknownArg.getValue();
    }


    /**
     * Script-facing delegatee for {@linkplain #setUnknownArg
     * setUnknownArg(String)}.
     * @since JWare/AntX 0.4
     **/
    public final void setValue(String value)
    {
        setUnknownArg(value);
    }


    /**
     * Sets the main operator for this condition. Must be one of
     * the declared "OP_" constants.
     **/
    public final void setOperator(String op)
    {
        require_(op!=null,"setOp- nonzro op");
        if      (OP_EQUALS.equals(op))     { m_Op=0; }
        else if (OP_CONTAINS.equals(op))   { m_Op=1; }
        else if (OP_STARTSWITH.equals(op)) { m_Op=2; }
        else if (OP_MATCHES.equals(op))    { m_Op=3; }
        else if (OP_ENDSWITH.equals(op))   { m_Op=4; }
        else { require_(false, "setOp- known operator"); }
    }


    /**
     * Set whether both arguments should be trimmed before being
     * compared.
     **/
    public void setTrim(boolean trim)
    {
        m_knownArg.setTrim(trim);
        m_unknownArg.setTrim(trim);
    }


    /**
     * Returns <i>true</i> if both arguments will be trimmed before
     * comparision.
     **/
    public boolean willTrim()
    {
        return m_knownArg.isTrimmed();
    }


    /**
     * Set whether the string comparision should be case-insensitive.
     * Defaults to case-sensitive comparisions.
     * @param b <i>true</i> if case sensitive (default)
     **/
    public void setIgnoreCase(boolean b)
    {
        m_knownArg.setIgnoreCase(b);
        m_unknownArg.setIgnoreCase(b);
    }


    /**
     * Returns <i>true</i> if case insensitive string comparision
     * will be done. Defaults to case-sensitive comparisions.
     **/
    public boolean isIgnoreCase()
    {
        return m_knownArg.isIgnoreCase();
    }


    /**
     * Set whether this comparision is for inequality. Shortcut for
     * embedding &lt;not&gt; in inlined condition attributes.
     **/
    public void setNegate(boolean b)
    {
        m_Not = b;
    }


    /**
     * Returns <i>true</i> if this is a not-a-match comparision
     * condition.
     **/
    public boolean isNegate()
    {
        return m_Not;
    }


    /**
     * Converts the known RE pattern string into the Regexp object
     * used for matching.
     **/
    private RegexpMatcher getRE()
    {
        if (m_RE==null) {
            RegularExpression redt= new RegularExpression();
            redt.setPattern(getKnownValue());
            m_RE= redt.getRegexp(getProject());
        }
        return m_RE;
    }


    /**
     * Compares two strings for some kind-of match.
     * @see #setOperator
     **/
    public boolean eval() throws BuildException
    {
        if (isUndefined()) {
            String ermsg = AntX.uistrs().get("brul.assert.need.2args");
            if (getProject()!=null) {
                log(ermsg,Project.MSG_ERR);
            }
            throw new BuildException(ermsg);
        }

        String unknown = getUnknownValue();
        String known   = getKnownValue();

        if (known==null || unknown==null) {//=> a missing property!
            return false;
        }

        boolean hit = true;
        switch(m_Op) {
            case 0: {
                hit = known.equals(unknown);
                break;
            }
            case 1: {
                hit = unknown.indexOf(known)>=0;
                break;
            }
            case 2: {
                hit = unknown.startsWith(known);
                break;
            }
            case 3: {
                hit = getRE().matches(unknown);
                break;
            }
            case 4: {
                hit = unknown.endsWith(known);
                break;
            }
        }

        return isNegate() ? !hit : hit;
    }


    /**
     * Retuns the known argument's underlying flexstring so
     * users can customize individually. Never returns <i>null</i>.
     **/
    public final FlexString getKnownValueGetter()
    {
        return m_knownArg;
    }


    /**
     * Returns the unknown argument's underlying flexstring so
     * users can customize individually. Never returns <i>null</i>.
     **/
    public final FlexString getUnknownValueGetter()
    {
        return m_unknownArg;
    }

    private FlexString m_knownArg = new FlexString();
    private FlexString m_unknownArg = new FlexString();
    private boolean m_Not;
    private int m_Op = 0;//equals
    private RegexpMatcher m_RE;
}

/* end-of-StringEquality.java */
