/**
 * $Id: FlexString.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  java.util.Locale;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.Handling;

/**
 * Wrapper for a string value that is determined from a literal, a property's value,
 * an AntX variable's value, or a reference's string form. A FlexString also encapsulates
 * many of the common manipulation instructions associated with string comparision so
 * every component doesn't have to. A FlexString uses a {@linkplain Stringifier} to
 * convert references to their string forms.
 * <p>
 * Usually FlexStrings reflect implementation-details of other classes; however, as
 * of AntX 0.3, FlexStrings can be exposed as (rather verbose) elements within other
 * types and tasks.
 * <p>
 * A <span class="src">FlexString</span> will try to resolve any property references in
 * its raw value before using it. This means you can use property variables to hold
 * the property/reference/variable names themselves like:
 * <span class="src">property="build.@{build.type}"</span> where
 * "<span class="src">build.type</span>" is determined by the flex string at
 * evaluation time.
 * <p>
 * Implementation Note: if both the <i>is-property</i> and <i>is-exported</i> options
 * are activated, the {@linkplain #getValue getValue} method will first read the
 * property's value and then use that output as the exported property's name. This is
 * usually what the desired effect is if both these options are checked; otherwise, make
 * sure only one of these options is checked. Note that a Flexstring cannot be both an
 * variable property and a reference; these are exclusive definitions.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 * @see      Stringifier
 **/

public class FlexString extends AssertableProjectComponent implements Cloneable
{
    /**
     * Creates a new null flex string.
     **/
    public FlexString()
    {
        m_normalizeLower= true;
    }

    /**
     * Creates a new defined flex string.
     * @param value string's raw value
     **/
    public FlexString(String value)
    {
        this();
        set(value);
    }

    /**
     * Creates a new property flex string.
     * @param value string's raw value
     * @param isP <i>true</i> if value is property name
     **/
    public FlexString(String value, boolean isP)
    {
        this();
        set(value);
        setIsProperty(isP);
    }


    /**
     * Creates a new flex string that uppercases string to
     * normalize for case-insensitivity. By default strings
     * are lower-cased to normalize.
     * @param uppercaseToNormalize <i>true</i> to use UPPER case
     **/
    public FlexString(boolean uppercaseToNormalize)
    {
        m_normalizeLower= !uppercaseToNormalize;
    }


    /**
     * Returns a clone of this flex string.
     * @since JWare/AntX 0.3
     **/
    public Object clone()
    {
        try {
            return super.clone();
        } catch(CloneNotSupportedException clnx) {
            throw new Error(uistrs().get(AntX.CLONE_BROKEN_MSGID));
        }
    }


    /**
     * Returns <i>true</i> if this flexstring has not been defined.
     **/
    public final boolean isUndefined()
    {
        return m_actualString==null;
    }


    /**
     * Sets this flexstring's raw underlying value. All
     * process instructions are applied to this string.
     **/
    public void set(String value)
    {
        m_actualString= value;
    }


    /**
     * Script-facing synonym for {@linkplain #set set()}.
     * @since JWare/AntX 0.3
     **/
    public final void setString(String value)
    {
        set(value);
    }


    /**
     * Returns this flexstring's raw underlying value. Returns
     * <i>null</i> if never set.
     **/
    public String get()
    {
        return m_actualString;
    }


    /**
     * Marks this flexstring's value as a property name. The
     * property is read by {@linkplain #getValue getValue}.
     **/
    public void setIsProperty(boolean b)
    {
        m_isProperty= b;
    }


    /**
     * Returns <i>true</i> if this flexstring's value is
     * actually a property name. Is <i>false</i> by default.
     **/
    public boolean isProperty()
    {
        return m_isProperty;
    }


    /**
     * Marks this flexstring's value as an exported property's name.
     * The exported property's value is read by {@linkplain #getValue
     * getValue}.
     **/
    public void setIsExported(boolean b)
    {
        m_isExported= b;
    }


    /**
     * Returns <i>true</i> if this flexstring's value is actually
     * an exported property's name. Is <i>false</i> by default.
     **/
    public boolean isExported()
    {
        return m_isExported;
    }


    /**
     * Marks this flexstring's value as a referenced object's name.
     * The referenced value is read (and stringified) by {@linkplain
     * #getValue getValue}.
     **/
    public void setIsReference(boolean b)
    {
        m_isReference= b;
    }


    /**
     * Returns <i>true</i> if this flexstring's value is actually
     * an reference object's name. Is <i>false</i> by default.
     **/
    public boolean isReference()
    {
        return m_isReference;
    }


    /**
     * Forces this string to consider its underlying value as
     * a literal. Resets all the source options.
     * @since JWare/AntX 0.2
     **/
    public void setIsLiteral()
    {
        m_isProperty = false;
        m_isReference = false;
        m_isExported = false;
    }


    /**
     * Returns <i>true</i> if this flexstring's value is literal;
     * no other "is-a" options apply.
     * @since JWare/AntX 0.2
     **/
    public boolean isLiteral()
    {
        return !m_isProperty && !m_isExported && !m_isReference;
    }


    /**
     * Tells this string to only use <em>guaranteed</em> string
     * values. So references are considered <i>null</i> unless
     * the refer to string-like things (like a String, Enum,
     * Environment.Variable, etc.) Flex strings are lenient by
     * default (uses the 'toString()' method to stringify everything).
     * @since JWare/AntX 0.2
     **/
    public void setLenient(boolean lenient)
    {
        m_isLenient = lenient;
    }


    /**
     * Returns <i>true</i> if this flexstring will tolerate any
     * type of reference; defaulting to the standard 'toString()'
     * method to stringify it. Defaults <i>true</i>.
     * @since JWare/AntX 0.2
     * @see Stringifier
     **/
    public boolean isLenient()
    {
        return m_isLenient;
    }


    /**
     * Tells this flexstring to trim whitespace from the
     * underlying value.
     **/
    public void setTrim(boolean b)
    {
        m_isTrimmed= b;
    }


    /**
     * Returns <i>true</i> if this flexstring's value will be
     * trimmed of whitespace. Is <i>false</i> by default.
     **/
    public boolean isTrimmed()
    {
        return m_isTrimmed;
    }


    /**
     * Tells this flexstring to normalize case of underlying
     * value for case-insensitive comparisions.
     **/
    public void setIgnoreCase(boolean ignored)
    {
        m_ignoreCase= ignored;
    }


    /**
     * Returns <i>true</i> if this flexstring will try to
     * normalize the case of its underlying string. Usually either
     * lowercases or UPPERCASES the string to normalize.
     * By default leaves string's case unchanged.
     **/
    public boolean isIgnoreCase()
    {
        return m_ignoreCase;
    }


    /**
     * Sets whether a value containing only whitespace is
     * considered same as an undefined value.
     * @since JWare/AntX 0.2
     **/
    public void setIgnoreWhitespace(boolean ignored)
    {
        m_ignoreWS= ignored;
    }


    /**
     * Script-facing synonym for {@linkplain #setIgnoreWhitespace
     * setIgnoreWhitespace()}. If the choice is one of "ignore",
     * "reject", or "balk", then whitespace is considered as
     * undefined.
     **/
    public void setWhitespace(Handling response)
    {
        response = Handling.simplifyIgnoreOrNot
            (response,Handling.ACCEPT);

        setIgnoreWhitespace(Handling.IGNORE.equals(response));
    }


    /**
     * Returns <i>true</i> if values of all whitespace
     * should be ignored (as if undefined). Defaults <i>false</i>.
     * @since JWare/AntX 0.2
     **/
    public boolean isIgnoreWhitespace()
    {
        return m_ignoreWS;
    }


    /**
     * Sets a starts-with filter for this flexstring's value. If
     * the value begins with this string it considered not-set.
     * @param filter the filtering prefix (<i>null</i> to clear)
     * @since JWare/AntX 0.3
     **/
    public void setIgnorePrefix(String filter)
    {
        m_swFilter = filter;
    }


    /**
     * Returns this flex string's starts-with filter or <i>null</i>
     * if never set.
     * @since JWare/AntX 0.3
     **/
    public String getPrefixFilter()
    {
        return m_swFilter;
    }


    /**
     * Marks this flexstring's value as a path (to be normalized).
     * This setting affects the {@linkplain #targetString targetString}
     * form of this flexstring.
     * @param isPath <i>true</i> to normalize value like path
     * @since JWare/AntX 0.4
     **/
    public void setIsPath(boolean isPath)
    {
        if (isPath) {
            m_isMask |= ISPATH;
        } else {
            m_isMask &= ~ISPATH;
        }
    }


    /**
     * Returns <i>true</i> if this value represent a file system path.
     * @since JWare/AntX 0.4
     **/
    public final boolean isPath()
    {
        return (m_isMask&ISPATH)==ISPATH;
    }


    /**
     * Returns this flexstring's processed value. All instructions
     * are applied to returned value. Returns <i>null</i> if
     * value never set or the referenced property doesn't exist.
     * Returns the empty string if trimming is turned on and the
     * underlying value was all whitespace. If this flexstring is a
     * property name and <span class="src">P</span> is undefined,
     * always returns <i>null</i>. This method is implemented as a
     * template method that first calls: {@linkplain #sourceString
     * sourceString(&#8230;)}, {@linkplain #targetString targetString(&#8230;)},
     * and {@linkplain #modifiedString modifiedString(&#8230;)} in order.
     * @param P [optional] the project from which properties read
     * @throws IllegalStateException if definition is contradictory
     **/
    public String getValue(final Project P)
    {
        verify_(!(isExported() && isReference()), "getvalu- exported|reference");

        String value = sourceString(P);

        if (value!=null) {
            value = targetString(P,value);
            if (value!=null) {
                value = modifiedString(P,value);
            }
        }

        return value;
    }


    /**
     * Same as {@linkplain #getValue(Project) getValue(Project)} but
     * using this component's assigned project.
     **/
    public String getValue()
    {
        return getValue(getProject());
    }


    /**
     * Figure out what to do with generic reference types. Never
     * returns <i>null</i>. Subclasses can customize here to use
     * other mechanism to map from object to string. By default
     * just uses a {@linkplain Stringifier}.
     * @param o object to stringify (non-null)
     * @param P [optional] project context
     * @since JWare/AntX 0.4 (made protected from private)
     **/
    protected String stringFrom(Object o, Project P)
    {
        return Stringifier.get(isLenient()).stringFrom(o,P);
    }


    /**
     * Returns the value string that a {@linkplain #getValue getValue}
     * will use to determine the target value. By default returns
     * the {@linkplain #get raw string} or the current value
     * of the raw string if that string was itself a variable.
     * @param P [optional] project context
     * @see #get
     * @since JWare/AntX 0.4
     **/
    public String sourceString(Project P)
    {
        String value= get();

        if (value==null) {
            return null;
        }
        return Tk.resolveString(P,value,true);//Like if/unless
    }



    /**
     * Returns the value string on which this item's modifiers
     * should be applied.
     * @param P [optional] project context
     * @param source item's {@linkplain #sourceString source} string
     * @since JWare/AntX 0.4
     **/
    public String targetString(Project P, String source)
    {
        String value = source;

        if (isProperty()) {
            value = (P!=null) ? P.getProperty(value) : null;
            if (value==null) {
                return null;
            }
        }

        if (isExported()) {
            value = ExportedProperties.readstring(value);
            if (value==null) {
                return null;
            }
        } else if (isReference()) {
            Object object = P.getReference(value);
            if (object==null) {
                return null;
            }
            value = stringFrom(object,P);
            if (value==null) {
                return null;
            }
        }

        if (value!=null && isPath()) {
            value = AntXFixture.fileUtils().normalize(value).getPath();
        }

        return value;
    }


    /**
     * Applies all of this string's modifiers to the given string.
     * Usually called by {@linkplain #getValue getValue} after referenced
     * fixture item has been retrieved.
     * @param P [optional] project context
     * @param value item's {@linkplain #targetString target} string
     * @since JWare/AntX 0.4
     **/
    protected String modifiedString(Project P, String value)
    {
        if (isIgnoreWhitespace()) {
            if (Tk.isWhitespace(value)) {
                return null;
            }
        }

        if (isTrimmed()) {
            value= value.trim();
            if (value.length()==0) {
                return value;
            }
        }

        if (isIgnoreCase()) {
            value= m_normalizeLower ?
                value.toLowerCase(Locale.US) : value.toUpperCase(Locale.US);
        }

        if (getPrefixFilter()!=null) {
            if (value.startsWith(getPrefixFilter())) {
                return null;
            }
        }

        return value;
    }


    /**
     * Returns a diagnostics string of this flex value.
     * @see #toAttributeName
     **/
    public String toString()
    {
        StringBuffer sb= AntXFixture.newSmallStringBuffer();
        if (isLiteral()) {
            sb.append("Literal/");
        } else if (isReference()) {
            sb.append("Reference/");
        } else {
            if (isProperty()) {
                sb.append("Property/");
            }
            if (isExported()) {
                sb.append("Variable/");
            }
        }
        sb.append("[");
        sb.append(get());
        sb.append("]");
        return sb.toString();
    }



    /**
     * Returns the standard name for the value parameter
     * of this flex value. Returns one of "value", "property",
     * "reference", or "variable".
     * @since JWare/AntX 0.5
     **/
    public String toAttributeName()
    {
        if (isReference()) {
            return "reference";
        }
        if (isProperty()) {
            if (isExported()) {
                return "variable";
            }
            return "property";
        }
        return "value";
   }


    private final boolean m_normalizeLower;
    private String  m_actualString;
    private boolean m_isProperty, m_isExported, m_isReference;
    private boolean m_ignoreCase, m_ignoreWS, m_isTrimmed;
    private boolean m_isLenient = true;
    private String  m_swFilter;

    private static final int ISPATH=0x01;
    private int m_isMask;
}

/* end-of-FlexString.java */
