<!--
  Indexing transformations for JPEG images.

  The source XML follows the structure below:

  ```
    <index-data mediatype="image/jpeg">
      <document level="metadata"> ... </document>
      <workflow> ... </workflow>
      <content source="tika"> ... </content>
    </index-data>
  ```

  @author Christophe Lauret
  @author Jean-Baptiste Reure

  @version 5.9000
-->
<xsl:transform  xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="2.0"
                xmlns:xhtml="http://www.w3.org/1999/xhtml"
                xmlns:psf="http://www.pageseeder.com/function"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                exclude-result-prefixes="#all">

<!-- We extend the default index module -->
<xsl:import href="index.xsl" />

<!-- indexXFields group property -->
<xsl:param name="ps-indexXFields" select="'false'" />


<!--
  Generate the fields specific to JPEG from content extracted by Tika.

  ```
  <content source="tika">
    <html xmlns="http://www.w3.org/1999/xhtml">
      <head>
        <meta name="Image Width" content="xxx pixels"/>
        <meta name="Image Height" content="yyy pixels"/>
        ...
      </head>
      <body/>
    </html>
  </content>
  ```
-->
<xsl:template match="index-data[@mediatype='image/jpeg']/content" mode="ixml" priority="2">
  <xsl:for-each select="descendant::xhtml:head">

    <xsl:variable name="rotated" select="
       contains(xhtml:meta[@name = 'Orientation']/@content, 'Rotate 90')
    or contains(xhtml:meta[@name = 'Orientation']/@content, 'Rotate 270')
    or matches(xhtml:meta[@name = 'tiff:Orientation']/@content, '5|6|7|8')"/>

    <xsl:variable name="exif-width" select="xhtml:meta[@name = 'Exif Image Width']/@content"/>
    <xsl:variable name="exif-height" select="xhtml:meta[@name = 'Exif Image Height']/@content"/>
    <xsl:variable name="width" select="xhtml:meta[@name = 'Image Width']/@content"/>
    <xsl:variable name="height" select="xhtml:meta[@name = 'Image Height']/@content"/>

    <!-- Image dimensions (Different for JPEG) -->
    <xsl:choose>
      <xsl:when test="$exif-width and $exif-height">
        <xsl:sequence select="psf:ps-jpeg-dimension(psf:format-dimension($exif-width), psf:format-dimension($exif-height), $rotated)"/>
      </xsl:when>
      <xsl:when test="$width and $height">
        <xsl:sequence select="psf:ps-jpeg-dimension(psf:format-dimension($width), psf:format-dimension($height), $rotated)"/>
      </xsl:when>
    </xsl:choose>

    <xsl:sequence select="psf:exif-media-dates(., /index-data/document/documentinfo/uri/@created)" />

    <!-- Metadata reported by Tika if any -->
    <xsl:for-each select="xhtml:meta[not(@name = 'X-Parsed-By' or @name = 'Content-type' or ends-with(@name, ' TRC'))][$ps-indexXFields = 'true']">
      <xsl:sort select="lower-case(@name)"/>
      <xsl:sequence select="psf:x-field(@name, @content)"/>
    </xsl:for-each>

  </xsl:for-each>
</xsl:template>

<xsl:function name="psf:format-dimension">
  <xsl:param name="dimension" />
  <xsl:choose>
    <xsl:when test="ends-with($dimension, ' pixels')"><xsl:value-of select="substring-before($dimension, ' pixels')" /></xsl:when>
    <xsl:otherwise><xsl:value-of select="$dimension" /></xsl:otherwise>
  </xsl:choose>
</xsl:function>

<!--
  Generates the `width`, `height` and `pixelcount` fields - for images only.

  @param width  The graphic's width in pixels
  @param height The graphic's height in pixels
-->
<xsl:function name="psf:ps-jpeg-dimension">
  <xsl:param name="width"  as="xs:integer"/>
  <xsl:param name="height" as="xs:integer"/>
  <xsl:param name="rotated" as="xs:boolean"/>
  <xsl:choose>
    <xsl:when test="$rotated">
      <xsl:sequence select="psf:ps-dimension($height, $width)"/>
    </xsl:when>
    <xsl:otherwise>
      <xsl:sequence select="psf:ps-dimension($width, $height)"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:function>


</xsl:transform>
