<?xml version="1.0"?>
<!--
  This schematron defines the general best practice recommendations by PageSeeder for
  general purpose PSML documents (which do not use complex templates).

  The schematron rules defined here are meant to provide guidance for users on how to
  produce quality PSML documents.

  @version 6.0000
-->
<sch:schema xmlns:sch="http://purl.oclc.org/dsdl/schematron">

  <sch:title>Best practice</sch:title>

  <!--
    Set of rules applying to the entire document
  -->
  <sch:pattern name="Document">

    <sch:rule context="/document">

      <!-- The document title should match the first heading -->
      <sch:assert id="mismatching_title" test="documentinfo/uri/displaytitle = (//heading)[1]">The title of this document '<sch:value-of select="documentinfo/uri/displaytitle"/>' does not match
      the first heading of the content '<sch:value-of select="(//heading)[1]"/>'.</sch:assert>

      <!-- The document should not have more than one heading 1 -->
      <sch:assert id="multiple_heading1" test="count(//heading[@level=1]) le 1">Your document should not have more than one heading 1.</sch:assert>

      <!-- Documents should not have more than 20 sections -->
      <sch:assert id="too_many_sections" test="count(section) le 20">Your document contains over 20 sections.
      Having too many sections in a document make it difficult to read and edit and may lead to performance issues.
      You should consider splitting your document into multiple documents.</sch:assert>

      <!-- Document IDs can improve documents -->
      <sch:report id="no_documentid" test="not(documentinfo/uri/@docid)"
                  flag="tip">Your document does not have a Doc ID.
      Specifying a Doc ID makes it easier to identify and find your document in PageSeeder.</sch:report>

      <!-- Labels can improve documents -->
      <sch:report id="no_labels" test="not(documentinfo/uri/labels)"
                  flag="tip">Your document has no label.
      Attaching labels to a document makes it easier to categorise and find your document in PageSeeder.</sch:report>

      <!-- It may be useful to check the number of words in a document -->
      <sch:report id="word_count" test="section"
                  flag="info">Document contains '<sch:value-of select="count(tokenize(normalize-space(string-join(section//text(), ' ')), ' '))"/>' words.</sch:report>

    </sch:rule>

  </sch:pattern>


  <!--
    Set of rules applying to the document fragments
  -->
  <sch:pattern name="Fragments">

    <!-- Rule matching regular fragments -->
    <sch:rule context="fragment">

      <!-- Fragments should start with a heading -->
      <sch:report id="no_heading_start" test="not(name(*[1]) = 'heading')"
                  flag="tip" properties="fragment">Fragment '<sch:value-of select="@id"/>' has no heading.
      It is better practice to start each fragment with a heading.</sch:report>

      <!-- Fragments should avoid including multiple headings -->
      <sch:report id="multiple_headings" test="count(heading) gt 1"
                  flag="info" properties="fragment">Fragment '<sch:value-of select="@id"/>' has multiple headings!</sch:report>

      <!-- Fragments should not be too long (over 2000 chars) -->
      <sch:assert id="long_fragment" test="string-length(string-join(.//text(), '')) le 2000"
                  flag="tip" properties="fragment">Fragment '<sch:value-of select="@id"/>' has <sch:value-of select="string-length(string-join(.//text(), ''))"/> characters (more than 2000).
      You should consider splitting this fragment.</sch:assert>

      <!-- Large fragments display change tracking (over 2000 elements/attributes/text nodes) -->
      <sch:assert id="complex_fragment" test="count(.//*) * 2 + count(.//@*) + count(.//text()[normalize-space(.)!='']) le 2000"
                  flag="tip" properties="fragment">Fragment '<sch:value-of select="@id"/>' has <sch:value-of
        select="count(.//*) * 2 + count(.//@*) + count(.//text()[normalize-space(.)!=''])"/> elements/attributes/text nodes (more than 2000).
      You should consider splitting this fragment to enable change tracking.</sch:assert>

      <!-- "code" style on multiple lines should use "pre" style instead -->
      <sch:assert id="code_multiple" test="not(para[monospace[. = ..]][following-sibling::*[1][self::para][monospace[. = ..]]])"
                  flag="tip" properties="fragment"
       >Fragment '<sch:value-of select="@id"/>' has two consecutive paragraphs containing only "code" content, consider using the "pre" style.</sch:assert>

    </sch:rule>

    <!-- Rule matching table rows -->
    <sch:rule context="row">

      <!-- Table header rows should only be at the top of the table -->
      <sch:assert id="misplaced_table_header_row" test="not(*[position()=last() and self::hcell] and preceding-sibling::row/*[position()=last() and self::cell])"
                  flag="warning" properties="fragment">Fragment
        '<sch:value-of select="ancestor::fragment/@id"/>' has a table header row not at the top of the table
        starting with content '<sch:value-of select="*[1]"/>' </sch:assert>

    </sch:rule>

    <!-- Rule matching blockxref -->
    <sch:rule context="blockxref">

      <!-- Embedded single fragments should not be used -->
      <sch:assert id="embed_single_fragment" test="not(@type='embed' and @frag!='default')"
                  properties="fragment">Fragment
        '<sch:value-of select="ancestor::fragment/@id|ancestor::xref-fragment/@id"/>' has a cross-reference
        "<sch:value-of select="."/>" embedding a single fragment.
        Replace this with a transclude.</sch:assert>

      <!-- Embed XRefs in publications should be reverse links. -->
      <sch:assert id="embed_xref_not_reverse" test="not(@type='embed' and @reverselink='false')"
                  properties="fragment">Fragment
        '<sch:value-of select="ancestor::fragment/@id|ancestor::xref-fragment/@id"/>' has an embed cross-reference
        "<sch:value-of select="."/>" that is not a reverse link.
        Publications only support embeds that are reverse links.</sch:assert>

    </sch:rule>

  </sch:pattern>

  <sch:properties>

    <sch:property id="fragment"><sch:value-of select="ancestor-or-self::fragment/@id"/></sch:property>

  </sch:properties>

</sch:schema>
