<!--
  ~ Copyright (c) 1999-2020 Allette Systems Pty Ltd

  Module dealing with presentation of PSML content (within regular fragments)

  A-Z listing of content template elements for read (and edit mode)
-->
<xsl:stylesheet version="3.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:psml="http://pageseeder.com/PSML"
                xmlns:t="http://pageseeder.com/psml/template"
                xmlns:diff="https://www.pageseeder.org/diffx"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:v-bind="https://vuejs.org"
                exclude-result-prefixes="#all">

<!-- If the label config is supplied, we use it for the labels -->
<xsl:variable name="labels" select="//labels" as="element(labels)*"/>

<!-- Default template for elements that shouldn't be processed in content -->
<xsl:template match="*" mode="psml psml-edit">
  <xsl:message>Ignoring `<xsl:value-of select="name()" />` element found in PSML.</xsl:message>
</xsl:template>

<!-- Element `anchor` -->
<xsl:template match="anchor" mode="psml psml-edit psml-pre">
  <!-- Please leave comment to prevent duplicate bug when element is left empty -->
  <a id="{@name}" data-psml="anchor"><xsl:comment/></a>
  <!-- TODO diff @name -->
</xsl:template>

<!--
  Element `Block` for block labels mapped to a `<div>` element.

  The original label is preserved in a data-label attribute, the class attribute is
  generated using the psml:label-class function.
-->
<xsl:template match="block" mode="psml psml-edit">
  <xsl:variable name="label" select="string(@label)" />
  <xsl:variable name="description" select="psml:label-description($label, 'content-block')"/>
  <div data-psml="block" data-label="{$label}"
       class="psml-block {psml:label-class($label)} color-{psml:label-color($label, 'content-block')}"
       title="Label '{$label}'{if ($description) then concat(': ', $description) else ''}">
    <!-- We insert a space to ensure that the block can be used for editing -->
    <xsl:if test="not(node())"><xsl:text> </xsl:text></xsl:if>
    <xsl:apply-templates mode="#current"/>
  </div>
</xsl:template>

<!--
  Element `blockxref` display the block xref
-->
<xsl:template match="blockxref" mode="psml">
  <p>
    <psml-xref>
      <xsl:call-template name="xref-component-attributes"/>
      <xsl:apply-templates mode="#current"/>
    </psml-xref>
  </p>
</xsl:template>

<!--
  Transclusions
-->
<xsl:template match="blockxref[@type = 'transclude']|xref[@type = 'math']" mode="psml">
  <xsl:variable name="is-diff"
                select="not(document or locator or fragment or media-fragment or xref-fragment or properties-fragment)"/>
  <xsl:variable name="is-archived-fragment"
                select="locator and not(fragment or media-fragment or xref-fragment or properties-fragment)"/>
  <xsl:variable name="is-empty"
                select="fragment
                 and normalize-space(.) = ''
                 and not(descendant::image)
                 and not(fragment[* except (para,heading)])"/>
  <component is="{if (@type='math') then 'psml-math' else 'psml-transclusion'}">
    <xsl:if test="$is-diff"><xsl:attribute name="diff" /></xsl:if>
    <xsl:call-template name="xref-component-attributes"/>
    <xsl:choose>
      <xsl:when test="starts-with(@mediatype, 'video/')">
        <video controls="" title="{text()}">
          <source src="{$site-prefix}/uri/{@uriid}?group={$current-group}" type="{@mediatype}"/>
        </video>
      </xsl:when>
      <xsl:when test="starts-with(@mediatype, 'audio/')">
        <audio controls="" title="{text()}">
          <source src="{$site-prefix}/uri/{@uriid}?group={$current-group}" type="{@mediatype}"/>
        </audio>
      </xsl:when>
      <xsl:when test="starts-with(@mediatype, 'image/')">
        <img src="{$site-prefix}/uri/{@uriid}?group={$current-group}">
          <xsl:copy-of select="@width | @height | @alt"/>
        </img>
      </xsl:when>
      <xsl:otherwise>
        <xsl:choose>
          <xsl:when test="$is-archived-fragment">
            <i class="has-no-content">(Archived fragment)</i>
          </xsl:when>
          <xsl:when test="$is-empty">
            <i class="has-no-content">(Empty)</i>
          </xsl:when>
          <xsl:when test="$is-diff">
            <i class="has-no-content">Transcluded content not available</i>
          </xsl:when>
          <xsl:otherwise>
            <xsl:apply-templates mode="psml-transclude"/>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </component>
</xsl:template>

<!-- Directly maps to HTML `<b>` tag. -->
<xsl:template match="bold" mode="psml psml-edit psml-pre">
  <xsl:if test="node()">
    <b><xsl:apply-templates mode="#current"/></b>
  </xsl:if>
</xsl:template>

<!-- Break mapped to a `<br>` element. -->
<xsl:template match="br" mode="psml">
  <!-- In view mode, we don't include the <br> in empty paragraphs -->
  <xsl:if test="count(../node()) gt 1"><span class="psml-br"/><br/></xsl:if>
</xsl:template>

<!-- Break mapped to a `<br>` element. -->
<xsl:template match="br" mode="psml-pre">
  <span class="psml-br"/><br/>
</xsl:template>

<!-- Element `caption` mapped to a `<caption>` element. -->
<xsl:template match="caption" mode="psml psml-edit">
  <caption><xsl:apply-templates mode="#current"/></caption>
</xsl:template>

<!--
  Element `<cell>` cell mapped to `<td>` or `<th>` element.

  The cell will be considered a header cell if it is a PSML <hcell> element or it is part of a column
  or header row.

      cell = element cell {
        attribute role?,
        attribute colspan?,
        attribute rowspan?,
        attribute align?,
        text
      }

  @param cols  An array of `<col>` to compute applicable column properties to the cell

  @context row
-->
<xsl:template match="cell|hcell" mode="psml psml-edit">
  <xsl:param name="column-pos" tunnel="yes" /> <!-- Don't do as="xs:integer" just in case it's empty -->
  <xsl:param name="cols" select="psml:normalize-cols(ancestor::table[1])" tunnel="yes" as="element(col)*"/>
  <xsl:variable name="col" select="$cols[number($column-pos)]" as="element(col)?"/>
  <xsl:element name="{if (self::hcell or parent::row/@part='header' or $col/@part = 'header') then 'th' else 'td'}">
    <xsl:copy-of select="@width | @rowspan | @colspan | @valign"/>
    <!-- Roles add up -->
    <xsl:variable name="classNames" as="xs:string*">
      <xsl:if test="@role"><xsl:value-of select="concat('cell-', @role)"/></xsl:if>
      <xsl:if test="$col/@role"><xsl:value-of select="concat('col-', $col/@role)"/></xsl:if>
    </xsl:variable>
    <xsl:if test="count($classNames) gt 0">
      <xsl:attribute name="class" select="string-join($classNames, ' ')"/>
    </xsl:if>
    <xsl:apply-templates select="@role" mode="psml-data"/>
    <!-- Alignments override -->
    <xsl:choose>
      <xsl:when test="@align"><xsl:attribute name="style" select="psml:to-align-css(@align)"/></xsl:when>
      <xsl:when test="not(parent::row/@align) and $col/@align"><xsl:attribute name="style" select="psml:to-align-css($col/@align)"/></xsl:when>
    </xsl:choose>
    <xsl:apply-templates mode="#current"/>
  </xsl:element>
</xsl:template>

<!--
  Element `<col>` (column definition) mapped to `<col>` element.

  The 'part', 'role' and 'align' attributes are preserved as 'data-*' attributes.

  The 'width' attribute will be transformed into inline CSS.

     col = element col {
       attribute role?,
       attribute span?,
       attribute width?,
       attribute align?,
       attribute part?,
     }

  @context table
-->
<xsl:template match="col" mode="psml psml-edit">
  <col>
    <xsl:copy-of select="@span"/>
    <!-- We keep track of these column attributes using a data attributes -->
    <xsl:apply-templates select="@part|@role|@align" mode="psml-data"/>
    <!-- The width can be specified directly at the column level -->
    <xsl:if test="@width"><xsl:attribute name="style" select="concat('width:', @width)"/></xsl:if>
  </col>
</xsl:template>

<!--
  Element `heading` mapped to the corresponding HTML heading elements `h1`, `h2`, `h3`,  `h4`, `h5`, `h6`
-->
<xsl:template match="heading" name="psml-heading" mode="psml psml-edit">
  <xsl:element name="h{if (@level) then @level else '6'}"><!-- handle old bug in schema which allowed no @level -->
    <xsl:variable name="top-fragment"            select="(ancestor::fragment | ancestor::xref-fragment)[1]" />
    <xsl:variable name="previous-headings-paras" select="(preceding::heading|preceding::para)[(ancestor::fragment | ancestor::xref-fragment)[generate-id() = generate-id($top-fragment)]]" />
    <xsl:attribute name="id" select="psml:heading-ref-anchor($top-fragment/@id , string(count($previous-headings-paras) + 1))"/>
    <xsl:apply-templates select="@prefix|@numbered" mode="psml-data"/>
    <!-- TODO Distinguish between manual and auto, add class for long prefixes? -->
    <xsl:apply-templates mode="#current"/>
  </xsl:element>
</xsl:template>

<!--
  Element `<image>`
-->
<xsl:template match="image" mode="psml">
  <psml-image>
    <xsl:copy-of select="@alt|@docid"/>
    <xsl:attribute name="v-bind:local" select="starts-with(@src, concat($site-prefix,'/',translate($current-group, '-', '/'),'/'))"/>
    <xsl:for-each select="@unresolved|@archived">
      <xsl:attribute name="v-bind:{name()}"><xsl:value-of select=". = 'true'"/></xsl:attribute>
    </xsl:for-each>
    <xsl:for-each select="@height|@uriid|@width">
      <xsl:attribute name="v-bind:{name()}"><xsl:value-of select="."/></xsl:attribute>
    </xsl:for-each>
    <xsl:for-each select="@labels">
      <xsl:attribute name="v-bind:{name()}">[<xsl:for-each select="tokenize(., ',')">'<xsl:value-of select="."/>',</xsl:for-each>]</xsl:attribute>
    </xsl:for-each>
    <!-- publisher session files -->
    <xsl:choose>
      <xsl:when test="ancestor::document/parent::content[@generator = 'com.pageseeder.layout.ui.ServletAdaptor'] and
                  //parameter[@name='path'] and
                  string(@uriid) = ''">
        <xsl:attribute name="src" select="concat(replace(//parameter[@name='path'], '/[^/]+$', '/'), @src)" />
      </xsl:when>
      <xsl:otherwise>
        <xsl:copy-of select="@src" />
      </xsl:otherwise>
    </xsl:choose>
  </psml-image>
</xsl:template>

<!--
  Element `<inline>` (inline label) mapped to a `<span>` element.

  The original label is preserved in a data-label attribute, the class attribute is
  generated using the psml:label-class function.
-->
<xsl:template match="inline" mode="psml psml-edit psml-pre">
  <xsl:variable name="label"       select="string(@label)" />
  <xsl:variable name="description" select="psml:label-description($label, 'content-inline')"/>
  <span data-psml="inline" data-label="{$label}"
        class="psml-inline {psml:label-class($label)} color-{psml:label-color($label, 'content-inline')}"
        title="Label '{$label}'{if ($description) then concat(': ', $description) else ''}">
    <xsl:apply-templates mode="#current"/>
  </span>
</xsl:template>

<!--
  Element `<inline>` for tex content (not available in edit mode)
-->
<xsl:template match="inline[@label='tex']" mode="psml psml-pre">
  <!-- handle possible del/ins from compare -->
  <xsl:variable name="content"><xsl:apply-templates mode="psml" /></xsl:variable>
  <math-katex expression="{serialize($content)}"/>
</xsl:template>

<!--
  Element `<inline>` for asciimatch content (not available in edit mode)
-->
<xsl:template match="inline[@label='asciimath']" mode="psml psml-pre">
  <!-- handle possible del/ins from compare -->
  <xsl:variable name="content"><xsl:apply-templates mode="psml" /></xsl:variable>
  <math-ascii expression="{serialize($content)}"/>
</xsl:template>

<!-- Element `<italic>` directly maps to HTML `<i>` tag. -->
<xsl:template match="italic" mode="psml psml-edit psml-pre">
  <xsl:if test="node()">
    <i><xsl:apply-templates mode="#current"/></i>
  </xsl:if>
</xsl:template>

<!--
  Element `<item>` (list item) mapped to `<li>` element.
-->
<xsl:template match="item" mode="psml psml-edit">
  <li><xsl:apply-templates mode="#current"/></li>
</xsl:template>

<!--
  Element `<link>` mapped to simple one way HTML link `<a>`
-->
<xsl:template match="link" mode="psml psml-pre">
  <psml-link id="link-{count(preceding::link)}" href="{@href}">
    <xsl:if test="@uriid">
      <xsl:attribute name="v-bind:uriid" select="@uriid" />
    </xsl:if>
    <xsl:copy-of select="@role|@mediatype|@frag|@title" />
    <!-- TODO Link class -->
    <xsl:for-each select="@unresolved|@archived">
      <xsl:attribute name="v-bind:{name()}"><xsl:value-of select=". = 'true'"/></xsl:attribute>
    </xsl:for-each>
    <xsl:for-each select="@labels">
      <xsl:attribute name="v-bind:{name()}">[<xsl:for-each select="tokenize(., ',')">'<xsl:value-of select="."/>',</xsl:for-each>]</xsl:attribute>
    </xsl:for-each>
    <xsl:attribute name="group-name" select="$current-group" />
    <xsl:apply-templates mode="#current"/>
  </psml-link>
</xsl:template>

<!-- Element `<list>` (Unordered list) mapped to `<ul>` element. -->
<xsl:template match="list" mode="psml">
  <ul>
    <xsl:apply-templates select="@type|@role" mode="psml-data"/>
    <xsl:variable name="classNames" as="xs:string*">
      <xsl:if test="@role"><xsl:value-of select="concat('list-', @role)"/></xsl:if>
      <xsl:if test="@type"><xsl:value-of select="@type"/></xsl:if>
    </xsl:variable>
    <xsl:if test="count($classNames) gt 0">
      <xsl:attribute name="class"><xsl:value-of select="$classNames" separator=" "/></xsl:attribute>
    </xsl:if>
    <xsl:apply-templates mode="#current"/>
  </ul>
</xsl:template>

<!-- Element `<monospace>` directly maps to HTML `<code>` tag. -->
<xsl:template match="monospace" mode="psml psml-edit psml-pre">
  <xsl:if test="node()">
    <code><xsl:apply-templates mode="#current"/></code>
  </xsl:if>
</xsl:template>

<!--
  Element `<nlist>` ordered list mapped to `<ol>` element.
-->
<xsl:template match="nlist" mode="psml">
  <ol>
    <xsl:apply-templates select="@type|@role" mode="psml-data"/>
    <xsl:copy-of select="@start"/>
    <xsl:variable name="classNames" as="xs:string*">
      <xsl:if test="@role"><xsl:value-of select="concat('list-', @role)"/></xsl:if>
      <xsl:if test="@type"><xsl:value-of select="@type"/></xsl:if>
    </xsl:variable>
    <xsl:if test="count($classNames) gt 0">
      <xsl:attribute name="class"><xsl:value-of select="$classNames" separator=" "/></xsl:attribute>
    </xsl:if>
    <xsl:apply-templates mode="#current"/>
  </ol>
</xsl:template>

<!-- Element `<para>` paragraph mapped to a `<p>` element. -->
<xsl:template match="para" mode="psml">
  <p>
    <xsl:apply-templates select="@indent|@prefix|@numbered" mode="psml-data"/>
    <xsl:if test="@indent"><xsl:attribute name="class">indent-<xsl:value-of select="@indent"/></xsl:attribute></xsl:if>
    <xsl:variable name="top-fragment"            select="(ancestor::fragment | ancestor::xref-fragment)[1]" />
    <xsl:variable name="previous-headings-paras" select="(preceding::heading|preceding::para)[(ancestor::fragment | ancestor::xref-fragment)[generate-id() = generate-id($top-fragment)]]" />
    <xsl:attribute name="id" select="psml:heading-ref-anchor($top-fragment/@id , string(count($previous-headings-paras) + 1))"/>
    <xsl:apply-templates mode="#current"/>
  </p>
</xsl:template>

<!-- Element `<placeholder>` paragraph mapped to a `<var>` element. -->
<xsl:template match="placeholder" mode="psml psml-pre">
  <psml-placeholder name="{@name}"><xsl:apply-templates mode="#current"/></psml-placeholder>
</xsl:template>

<!-- Element `<preformat>` (preformatted text) mapped to a `<pre>` element. -->
<xsl:template match="preformat" mode="psml">
  <psml-preformat><xsl:if test="monospace"><xsl:attribute name="ignore-lang"/></xsl:if><xsl:copy-of select="@role"/><xsl:apply-templates select="*|text()" mode="psml-pre"/></psml-preformat>
</xsl:template>

<xsl:template match="text()" mode="psml-pre">
  <xsl:for-each select="tokenize(., '&#xa;')">
    <xsl:value-of select="."/>
    <xsl:if test="position() != last()"><span class="psml-br"/><br/></xsl:if>
  </xsl:for-each>
</xsl:template>

<!--
  Element `<row>` mapped to `<tr>` element.

  The 'part', 'role' and 'align' attributes are preserved as 'data-*' attributes.

  The 'width' attribute will be transformed into inline CSS.

     row = element row {
       attribute role?,
       attribute align?,
       attribute part?,
       (cell | hcell)+
     }

  @param cols The columns definitions for all cells in that row (may be supplied by table template)
  @context table
- ->
<xsl:template match="row" mode="psml psml-edit">
  <xsl:param name="cols" select="psml:normalize-cols(parent::table)" tunnel="yes" as="element(col)*"/>
  <tr>
    <xsl:for-each select="@role"><xsl:attribute name="class" select="."/></xsl:for-each>
    <xsl:if test="@align"><xsl:attribute name="style" select="psml:to-align-css(@align)"/></xsl:if>
    <xsl:apply-templates select="@role" mode="psml-data"/>
    <xsl:apply-templates select="cell|hcell" mode="#current"/>
  </tr>
</xsl:template> -->

<!-- Element `<sub>` directly maps to HTML `<sub>` tag. -->
<xsl:template match="sub" mode="psml psml-edit psml-pre">
  <sub><xsl:apply-templates mode="#current"/></sub>
</xsl:template>

<!-- Element `<sup>` directly maps to HTML `<sup>` tag. -->
<xsl:template match="sup" mode="psml psml-edit psml-pre">
  <xsl:if test="node()">
    <sup><xsl:apply-templates mode="#current"/></sup>
  </xsl:if>
</xsl:template>

<xsl:template match="t:value" mode="psml psml-edit">
  <xsl:if test="node()">
    <var><xsl:value-of select="@name"/></var>
  </xsl:if>
</xsl:template>

<!--
  Element `<table>` is assumed to be denormalized.

  The default default template will normalize the table first in order to correctly compute the
  rules applied to columns.

  Normalization is not necessary if the table does not have any column definitions or does not
  include any row or column span.

  After the table is normalized, the XSLT template in mode "psml-normalized" is applied.

     table = element table {
       attribute role?,
       attribute width?,
       attribute height?,
       caption?
       col*,
       row+,
     }
-->
<xsl:template match="table" mode="psml psml-edit">
  <xsl:variable name="cols" select="psml:normalize-cols(.)" as="element(col)*"/>
  <!-- initialise with ones -->
  <xsl:variable name="prev-rowspans">
    <xsl:for-each select="$cols">
      <c span="1"/>
    </xsl:for-each>
  </xsl:variable>
  <table>
    <xsl:copy-of select="@summary | @cellspacing | @cellpadding | @border"/>
    <xsl:apply-templates select="@role" mode="psml-data"/>
    <xsl:if test="@role">
      <xsl:attribute name="class" select="concat('table-', @role)"/>
    </xsl:if>
    <xsl:if test="@width or @height">
      <xsl:attribute name="style">
        <xsl:if test="@width">width:<xsl:value-of select="@width"/>;</xsl:if>
        <xsl:if test="@height">height:<xsl:value-of select="@height"/></xsl:if>
      </xsl:attribute>
    </xsl:if>
    <xsl:apply-templates select="@height" mode="psml-data"/>
    <xsl:if test="col">
      <colgroup><xsl:apply-templates select="col" mode="#current"/></colgroup>
    </xsl:if>
    <xsl:if test="caption">
      <caption><xsl:apply-templates select="caption/node()" mode="#current"/></caption>
    </xsl:if>
    <xsl:if test="row[@part='header']">
      <thead>
        <!-- <xsl:apply-templates select="row[@part='header']" mode="#current"> -->
        <xsl:call-template name="create-row" >
          <xsl:with-param name="rows" select="row[@part='header']" />
          <xsl:with-param name="row-num" select="1" />
          <xsl:with-param name="prev-rowspans" select="$prev-rowspans" />
          <xsl:with-param name="cols" select="$cols" tunnel="yes" />
        </xsl:call-template>
      </thead>
    </xsl:if>
    <xsl:if test="row[@part='footer']">
      <tfoot>
        <!-- <xsl:apply-templates select="row[@part='footer']" mode="#current"> -->
        <xsl:call-template name="create-row" >
          <xsl:with-param name="rows" select="row[@part='footer']" />
          <xsl:with-param name="row-num" select="1" />
          <xsl:with-param name="prev-rowspans" select="$prev-rowspans" />
          <xsl:with-param name="cols" select="$cols" tunnel="yes" />
        </xsl:call-template>
      </tfoot>
    </xsl:if>
    <tbody>
      <!-- <xsl:apply-templates select="row[not(@part) or @part = 'body']" mode="#current"> -->
      <xsl:call-template name="create-row" >
        <xsl:with-param name="rows" select="row[not(@part) or @part = 'body']" />
        <xsl:with-param name="row-num" select="1" />
        <xsl:with-param name="prev-rowspans" select="$prev-rowspans" />
        <xsl:with-param name="cols" select="$cols" tunnel="yes" />
      </xsl:call-template>
    </tbody>
  </table>
</xsl:template>

<!-- create rows recursively -->
<xsl:template name="create-row">
  <xsl:param name="rows" as="element(row)*"/>
  <xsl:param name="row-num" as="xs:integer"/>
  <xsl:param name="prev-rowspans"/>
  <xsl:param name="cols" tunnel="yes" as="element(col)*"/>
  <!-- <xsl:message>$prev-rowspans=<xsl:copy-of select="$prev-rowspans" /></xsl:message> -->

  <xsl:variable name="row" select="$rows[$row-num]"/>

  <xsl:if test="$row">
    <xsl:variable name="cur-rowspans">
      <xsl:for-each select="$row/*">
        <!-- assign cell to the next normalized cell position with no rowspan (i.e. @span=1) -->
        <xsl:variable name="cell-span" select="$prev-rowspans/c[@span='1'][psml:cell-position(current())]" />
        <c pos="{count($cell-span/preceding-sibling::*) + 1}"
           span="{if (@rowspan) then @rowspan else 1}" />
      </xsl:for-each>
    </xsl:variable>
    <!-- <xsl:message>$cur-rowspans=<xsl:copy-of select="$cur-rowspans" /><xsl:text>&#xA;</xsl:text></xsl:message> -->
    <tr>
      <xsl:for-each select="$row/@role"><xsl:attribute name="class" select="concat('row-', .)"/></xsl:for-each>
      <xsl:if test="$row/@align"><xsl:attribute name="style" select="psml:to-align-css($row/@align)"/></xsl:if>
      <xsl:apply-templates select="$row/@role" mode="psml-data"/>
      <xsl:for-each select="$row/*">
        <xsl:variable name="cpos" select="position()" />
        <xsl:apply-templates select="." mode="#current">
          <xsl:with-param name="cols" select="$cols" tunnel="yes" />
          <xsl:with-param name="column-pos" select="$cur-rowspans/c[$cpos]/@pos" tunnel="yes"/>
        </xsl:apply-templates>
      </xsl:for-each>
    </tr>
    <!-- pass on the previous rowspans minus one or current rowspans -->
    <xsl:variable name="next-rowspans">
      <xsl:for-each select="$prev-rowspans/c">
        <xsl:variable name="cpos" select="position()" />
        <!-- current rowspan is from the last cell with @pos <= $cpos due to colspans -->
        <c span="{if (@span=1) then number(($cur-rowspans/c[number(@pos)le$cpos])[last()]/@span)
                  else number(@span) - 1}" />
      </xsl:for-each>
    </xsl:variable>
    <!-- create next row -->
    <xsl:call-template name="create-row" >
      <xsl:with-param name="rows" select="$rows" />
      <xsl:with-param name="row-num" select="$row-num + 1" />
      <xsl:with-param name="prev-rowspans" select="$next-rowspans" />
      <xsl:with-param name="cols" select="$cols" tunnel="yes" />
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<!-- Element `<underline>` directly maps to HTML `<u>` tag. -->
<xsl:template match="underline" mode="psml psml-edit psml-pre">
  <xsl:if test="node()">
    <u><xsl:apply-templates mode="#current"/></u>
  </xsl:if>
</xsl:template>

<!--
  Element `<xref>`
-->
<xsl:template match="xref" mode="psml psml-pre">
  <psml-xref>
    <xsl:call-template name="xref-component-attributes"/>
    <xsl:apply-templates mode="#current"/>
  </psml-xref>
</xsl:template>

<!--
  Serializes the attribute for the xrefs and blockxrefs components

  @context xref|blockxref
-->
<xsl:template name="xref-component-attributes">
  <xsl:if test="contains(@href, '/archive/')"><xsl:attribute name="archived" /></xsl:if>
  <xsl:if test="starts-with(@href, concat($site-prefix,'/',translate($current-group, '-', '/'),'/'))"><xsl:attribute name="local" /></xsl:if>
  <xsl:copy-of select="@config|@display|@docid|@documenttype|@frag|@href|@mediatype|@reversetitle|@reversetype|@reversefrag|@title|@type"/>
  <xsl:choose>
    <xsl:when test="string(@urititle) = '' and @type = 'transclude'">
      <xsl:attribute name="urititle"><xsl:apply-templates mode="psml-xref-urititle" /></xsl:attribute>
    </xsl:when>
    <xsl:otherwise><xsl:copy-of select="@urititle" /></xsl:otherwise>
  </xsl:choose>
  <xsl:for-each select="@external|@reverselink|@unresolved|@archived">
    <xsl:attribute name="v-bind:{name()}"><xsl:value-of select=". = 'true'"/></xsl:attribute>
  </xsl:for-each>
  <xsl:for-each select="@labels|@urilabels">
    <xsl:attribute name="v-bind:{name()}">[<xsl:for-each select="tokenize(., ',')">'<xsl:value-of select="."/>',</xsl:for-each>]</xsl:attribute>
  </xsl:for-each>
  <xsl:for-each select="@id|@level|@uriid">
    <xsl:attribute name="v-bind:{name()}"><xsl:value-of select="."/></xsl:attribute>
  </xsl:for-each>
  <xsl:attribute name="group-name" select="$current-group" />
</xsl:template>
<!-- ignore deleted parts of the xref content to load the uri title -->
<xsl:template match="diff:del" mode="psml-xref-urititle" />

<!-- Data-* Attribute ==================================================================================== -->

<!--
  Returns a `data-*` attribute with the same value and the name prefixed by `data-`
-->
<xsl:template match="@*" mode="psml-data" as="attribute()">
  <xsl:attribute name="data-{local-name()}" select="."/>
</xsl:template>

<!-- Utility Functions =======================================================================  -->

<xsl:function name="psml:cell-position" as="xs:integer">
  <xsl:param name="cell" />
  <xsl:sequence select="count($cell/preceding-sibling::*[not(@colspan)]) + xs:integer(sum($cell/preceding-sibling::*/@colspan)) + 1" />
</xsl:function>

<!--
  Returns the CSS class for for the given label name.

  This function turns the label to lower case, replaces non-word characters by a dash and
  prefix the label by 'label-'.

  @param label The name of the label
  @return the corresponding CSS class.
-->
<xsl:function name="psml:label-class" as="xs:string">
  <xsl:param name="label" as="xs:string"/>
  <xsl:value-of select="concat('label-', lower-case(replace($label, '\W', '-')))"/>
</xsl:function>

<!--
  Returns the CSS for the align attribute as a `text-align` CSS property.

  @param align The align attribute
  @return the corresponding CSS.
-->
<xsl:function name="psml:to-align-css" as="xs:string?">
  <xsl:param name="align" as="attribute(align)"/>
  <xsl:if test="matches($align, 'left|center|right|justify')"><xsl:value-of select="concat('text-align:',$align,';')"/></xsl:if>
</xsl:function>

<!--
  Return a unique ID for heading references
-->
<xsl:function name="psml:heading-ref-anchor" as="xs:string">
  <xsl:param name="fragment-id"  as="xs:string?"/>
  <xsl:param name="index"        as="xs:string"/>
  <xsl:sequence select="concat('fragment-', if ($fragment-id) then $fragment-id else 'default', '-toc-', $index)"/>
</xsl:function>

<!--
  Returns the CSS class for the color a specific label name.

  @param label The name of the label
  @return the corresponding CSS class.
-->
<xsl:function name="psml:label-color" as="xs:string">
  <xsl:param name="label"  as="xs:string"/>
  <xsl:param name="type"   as="xs:string"/>
  <xsl:variable name="color" select="$labels[@type = $type]/label[@name = $label]/@color"/>
  <xsl:value-of select="if ($color) then $color else 'default'"/>
</xsl:function>

<xsl:function name="psml:label-description" as="xs:string">
  <xsl:param name="label"  as="xs:string"/>
  <xsl:param name="type"   as="xs:string"/>
  <xsl:variable name="description" select="$labels[@type = $type]/label[@name = $label]/@description"/>
  <xsl:value-of select="if ($description) then $description else ''"/>
</xsl:function>

<!-- Table normalization ===============================================================================  -->

<!--
  Normalized `<col>` elements so that the column definition can be used for individual cells using
  their position.

  @param table The table
  @return an array of columns
-->
<xsl:function name="psml:normalize-cols" as="element(col)*">
  <xsl:param name="table" as="element(table)?"/>
  <xsl:for-each select="$table/col">
    <xsl:variable name="attr" select="@align|@role|@part"/>
    <xsl:variable name="count" select="if (@span) then @span else 1" as="xs:integer"/>
    <xsl:for-each select="1 to $count">
      <col><xsl:copy-of select="$attr"/></col>
    </xsl:for-each>
  </xsl:for-each>
</xsl:function>

</xsl:stylesheet>