<!--
  ~ Copyright (c) 1999-2019 Allette Systems Pty Ltd

  This shared module provides common JSON definitions for PageSeeder objects.

  All templates in this module MUST use mode `json`.

  They SHOULD provide a `key` parameter to specify the JSON array or object property name.
-->
<xsl:stylesheet version="3.0"
                xmlns:f="http://www.pageseeder.com/function"
                xmlns:json="http://www.w3.org/2005/xpath-functions"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:t="http://pageseeder.com/psml/template"
                exclude-result-prefixes="#all">

  <xsl:template match="account-home/link" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="href"><xsl:value-of select="@href"/></json:string>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
    </json:map>
  </xsl:template>

  <xsl:template match="access-token" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="clientId"><xsl:value-of select="@client-id"/></json:string>
      <json:string key="expires"><xsl:value-of select="@expires"/></json:string>
      <json:boolean key="expired"><xsl:value-of select="@expired = 'true'"/></json:boolean>
      <json:string key="issued"><xsl:value-of select="@issued"/></json:string>
      <json:number key="lifetime"><xsl:value-of select="@lifetime"/></json:number>
      <json:number key="memberId"><xsl:value-of select="@member-id"/></json:number>
      <json:string key="scope"><xsl:value-of select="@scope"/></json:string>
      <json:string key="token"><xsl:value-of select="@token"/></json:string>
      <json:string key="type"><xsl:value-of select="name()"/></json:string>
      <xsl:apply-templates select="client" mode="json">
        <xsl:with-param name="key" select="'client'"/>
      </xsl:apply-templates>
      <xsl:apply-templates select="member" mode="json">
        <xsl:with-param name="key" select="'member'"/>
      </xsl:apply-templates>
    </json:map>
  </xsl:template>

  <!--
    Similar to member (identical?)
  -->
  <xsl:template match="assignedto|author|modifiedby|acceptedby|draftauthor" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <xsl:if test="@id">
        <json:number key="id"><xsl:value-of select="@id"/></json:number>
      </xsl:if>
      <json:string key="firstname"><xsl:value-of select="@firstname"/></json:string>
      <json:string key="surname"><xsl:value-of select="@surname"/></json:string>
      <xsl:if test="@username">
        <json:string key="username"><xsl:value-of select="@username"/></json:string>
      </xsl:if>
      <xsl:if test="@status">
        <json:string key="status"><xsl:value-of select="@status"/></json:string>
      </xsl:if>
      <xsl:if test="@email">
        <json:string key="email"><xsl:value-of select="@email"/></json:string>
      </xsl:if>
      <json:string key="fullname"><xsl:value-of select="fullname"/></json:string>
      <xsl:for-each select="@created|@activated|@lastlogin|@date">
        <json:string key="{name()}"><xsl:value-of select="."/></json:string>
      </xsl:for-each>
      <xsl:if test="@admin">
        <json:boolean key="admin"><xsl:value-of select="@admin = 'true'"/></json:boolean>
      </xsl:if>
    </json:map>
  </xsl:template>

  <!-- TODO Was used in compare fragment, consolidate with template above -->
  <xsl:template match="edit/author | draft/author" mode="json">
    <json:map key="author">
      <xsl:if test="@id"><json:number key="id"><xsl:value-of select="@id"/></json:number></xsl:if>
      <xsl:if test="@firstname"><json:string key="firstname"><xsl:value-of select="@firstname"/></json:string></xsl:if>
      <xsl:if test="@surname"><json:string key="surname"><xsl:value-of select="@surname"/></json:string></xsl:if>
      <xsl:if test="@username"><json:string key="username"><xsl:value-of select="@username"/></json:string></xsl:if>
      <xsl:if test="@email"><json:string key="email"><xsl:value-of select="@email"/></json:string></xsl:if>
      <json:string key="fullname"><xsl:value-of select="fullname"/></json:string>
    </json:map>
  </xsl:template>

  <!--
    <authorization-code code="6e82b98de4e069552c723b90ca495fbd"
                   client-id="c66757f652315c6a"
                   member-id="1"
                       scope="dfdfdasfd"
                redirect-uri=""
                      issued="2018-07-04T15:49:13+10:00"
                     expires="2018-07-04T15:59:13+10:00"/>
  -->
  <xsl:template match="authorization-code" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string  key="code"><xsl:value-of select="@code"/></json:string>
      <json:string  key="clientId"><xsl:value-of select="@client-id"/></json:string>
      <json:string  key="expires"><xsl:value-of select="@expires"/></json:string>
      <json:boolean key="expired"><xsl:value-of select="@expired = 'true'"/></json:boolean>
      <json:string  key="issued"><xsl:value-of select="@issued"/></json:string>
      <json:number  key="memberId"><xsl:value-of select="@member-id"/></json:number>
      <json:string  key="scope"><xsl:value-of select="@scope"/></json:string>
      <json:string  key="redirectUri"><xsl:value-of select="@redirect-uri"/></json:string>
      <xsl:apply-templates select="client" mode="json">
        <xsl:with-param name="key" select="'client'"/>
      </xsl:apply-templates>
      <xsl:apply-templates select="member" mode="json">
        <xsl:with-param name="key" select="'member'"/>
      </xsl:apply-templates>
    </json:map>
  </xsl:template>

  <xsl:template match="discussion" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:number key="id"><xsl:value-of select="@id"/></json:number>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
      <xsl:if test="@replycount">
        <json:number key="replyCount"><xsl:value-of select="@replycount"/></json:number>
      </xsl:if>
      <xsl:if test="@lastcommentid">
        <json:number key="lastCommentId"><xsl:value-of select="@lastcommentid"/></json:number>
      </xsl:if>
      <xsl:if test="@lastcommentdate">
        <json:string key="lastCommentDate"><xsl:value-of select="@lastcommentdate"/></json:string>
      </xsl:if>
      <xsl:variable name="lastcomment" select="comment[@id = ../@lastcommentid]" />
      <xsl:if test="$lastcomment/@status">
        <json:string key="status"><xsl:value-of select="$lastcomment/@status"/></json:string>
      </xsl:if>
      <xsl:if test="$lastcomment/@priority">
        <json:string key="priority"><xsl:value-of select="$lastcomment/@priority"/></json:string>
      </xsl:if>
      <xsl:if test="$lastcomment/@due">
        <json:string key="due"><xsl:value-of select="$lastcomment/@due"/></json:string>
      </xsl:if>
      <xsl:if test="$lastcomment/assignedto">
        <xsl:apply-templates select="$lastcomment/assignedto" mode="json">
          <xsl:with-param name="key">assignedto</xsl:with-param>
        </xsl:apply-templates>
      </xsl:if>
      <xsl:apply-templates select="group" mode="json">
        <xsl:with-param name="key">group</xsl:with-param>
      </xsl:apply-templates>
      <xsl:apply-templates select="uri" mode="json">
        <xsl:with-param name="key">uri</xsl:with-param>
      </xsl:apply-templates>
      <xsl:if test="locator/@fragment">
        <json:map key="locator">
          <json:string key="fragment"><xsl:value-of select="locator/@fragment"/></json:string>
        </json:map>
      </xsl:if>
      <json:array key="comments">
        <xsl:apply-templates select="comment" mode="json" />
      </json:array>
    </json:map>
  </xsl:template>

  <!--
    <client id="5" identifier="c66757f652315c6a" requires-consent="false" confidential="false"
            name="ReviewSpace App 2017-09-18T14:13:39.581" grant-type="password"
            last-token="2018-07-04T15:00:00+10:00" description="Review Space application"
            client-uri="http://localhost:8771/rs" scope="openid email profile" access-token-max-age="21600"
            refresh-token-max-age="2592000">
      <member id="1" firstname="Christophe" surname="Lauret" username="clauret" status="activated">
        <fullname>Christophe Lauret</fullname>
      </member>
    </client>
  -->
  <xsl:template match="client" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key">
        <xsl:attribute name="key" select="$key"/>
      </xsl:if>
      <json:number  key="accessTokenMaxAge"><xsl:value-of select="@access-token-max-age"/></json:number>
      <json:string  key="app"><xsl:value-of select="@app"/></json:string>
      <json:string  key="clientUri"><xsl:value-of select="@client-uri"/></json:string>
      <json:boolean key="confidential"><xsl:value-of select="@confidential = 'true'"/></json:boolean>
      <json:string  key="created"><xsl:value-of select="@created"/></json:string>
      <json:string  key="description"><xsl:value-of select="@description"/></json:string>
      <json:string  key="grantType"><xsl:value-of select="@grant-type"/></json:string>
      <json:number  key="id"><xsl:value-of select="@id"/></json:number>
      <json:string  key="identifier"><xsl:value-of select="@identifier"/></json:string>
      <json:string  key="lastToken"><xsl:value-of select="@last-token"/></json:string>
      <json:string  key="name"><xsl:value-of select="@name"/></json:string>
      <json:string  key="redirectUri"><xsl:value-of select="@redirect-uri"/></json:string>
      <json:number  key="refreshTokenMaxAge"><xsl:value-of select="@refresh-token-max-age"/></json:number>
      <json:boolean key="requiresConsent"><xsl:value-of select="@requires-consent = 'true'"/></json:boolean>
      <json:string  key="scope"><xsl:value-of select="@scope"/></json:string>
      <json:string  key="webhookSecret"><xsl:value-of select="@webhook-secret"/></json:string>
      <xsl:apply-templates select="member" mode="json">
        <xsl:with-param name="key">member</xsl:with-param>
      </xsl:apply-templates>
    </json:map>
  </xsl:template>

  <xsl:template match="comment" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:number key="id"><xsl:value-of select="@id"/></json:number>
      <xsl:if test="@contentrole">
        <json:string key="contentrole"><xsl:value-of select="@contentrole"/></json:string>
      </xsl:if>
      <xsl:if test="@type">
        <json:string key="type"><xsl:value-of select="@type"/></json:string>
      </xsl:if>
      <xsl:if test="@created">
        <json:string key="created"><xsl:value-of select="@created"/></json:string>
      </xsl:if>
      <xsl:if test="@priority">
        <json:string key="priority"><xsl:value-of select="@priority"/></json:string>
      </xsl:if>
      <xsl:if test="@status">
        <json:string key="status"><xsl:value-of select="@status"/></json:string>
      </xsl:if>
      <xsl:if test="@due">
        <json:string key="due"><xsl:value-of select="@due"/></json:string>
      </xsl:if>
      <json:array key="labels">
        <xsl:for-each select="tokenize(@labels, ',')">
          <json:string><xsl:value-of select="."/></json:string>
        </xsl:for-each>
        <xsl:for-each select="tokenize(labels, ',')">
          <json:string><xsl:value-of select="."/></json:string>
        </xsl:for-each>
      </json:array>
      <json:boolean key="moderated"><xsl:value-of select="@moderated = 'true'"/></json:boolean>
      <json:boolean key="draft"><xsl:value-of select="@draft = 'true'"/></json:boolean>
      <json:string key="title"><xsl:value-of select="@title | title"/></json:string>

      <xsl:apply-templates select="author" mode="json">
        <xsl:with-param name="key" select="'author'"/>
      </xsl:apply-templates>
      <xsl:apply-templates select="modifiedby" mode="json">
        <xsl:with-param name="key" select="'modifiedby'"/>
      </xsl:apply-templates>
      <xsl:apply-templates select="assignedto" mode="json">
        <xsl:with-param name="key" select="'assignedto'"/>
      </xsl:apply-templates>

      <xsl:for-each select="properties">
        <json:array key="properties">
          <xsl:for-each select="property">
            <json:map>
              <json:string key="name"><xsl:value-of select="@name"/></json:string>
              <json:string key="value"><xsl:value-of select="@value"/></json:string>
            </json:map>
          </xsl:for-each>
        </json:array>
      </xsl:for-each>

      <json:array key="content">
        <xsl:for-each select="content">
          <json:map>
            <json:string key="type"><xsl:value-of select="@type"/></json:string>
            <json:string key="value"><xsl:value-of select="serialize(text()|*)"/></json:string>
          </json:map>
        </xsl:for-each>
      </json:array>

      <xsl:if test="attachment">
        <json:array key="attachments">
          <xsl:for-each select="attachment">
            <json:map>
              <xsl:apply-templates select="uri" mode="json">
                <xsl:with-param name="key">uri</xsl:with-param>
              </xsl:apply-templates>
            </json:map>
          </xsl:for-each>
        </json:array>
      </xsl:if>

    </json:map>
  </xsl:template>

  <xsl:template match="document-ref" mode="json">
    <json:map>
      <json:string key="type">document</json:string>
      <json:string key="documenttype"><xsl:value-of select="@documenttype"/></json:string>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
      <json:string key="fragment"><xsl:value-of select="@fragment"/></json:string>
      <json:number key="position"><xsl:value-of select="@position"/></json:number>
      <json:boolean key="children"><xsl:value-of select="@children = 'true'"/></json:boolean>
      <xsl:if test="@uriid">
        <json:number key="uriid"><xsl:value-of select="@uriid"/></json:number>
      </xsl:if>
      <xsl:if test="@numbered">
        <json:boolean key="numbered"><xsl:value-of select="@numbered = 'true'"/></json:boolean>
        <json:string key="canonical"><xsl:value-of select="@canonical"/></json:string>
      </xsl:if>
      <xsl:if test="@prefix">
        <json:string key="prefix"><xsl:value-of select="@prefix"/></json:string>
      </xsl:if>
      <xsl:if test="@path">
        <json:string key="path"><xsl:value-of select="@path" /></json:string>
      </xsl:if>
    </json:map>
  </xsl:template>

  <xsl:template match="element" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="key"><xsl:value-of select="@key"/></json:string>
      <json:string key="version"><xsl:value-of select="@version"/></json:string>
      <json:string key="hitCount"><xsl:value-of select="@hit-count"/></json:string>
      <json:string key="timeToIdle"><xsl:value-of select="f:duration-short(@time-to-idle)"/></json:string>
      <json:string key="timeToLive"><xsl:value-of select="f:duration-short(@time-to-live)"/></json:string>
      <json:boolean key="eternal"><xsl:value-of select="if (@eternal) then @eternal else 'false'"/></json:boolean>
      <json:boolean key="expired"><xsl:value-of select="if (@expired) then @expired else 'false'"/></json:boolean>
      <json:string key="creationTime"><xsl:value-of select="@creation-time"/></json:string>
      <json:string key="expirationTime"><xsl:value-of select="@expiration-time"/></json:string>
      <json:string key="lastAccessTime"><xsl:value-of select="@last-access-time"/></json:string>
    </json:map>
  </xsl:template>

  <!-- Log events (access, process, publish) ...but not index -->
  <xsl:template match="event" mode="json">
    <xsl:param name="first-event"/>
    <xsl:param name="previous-event" select="preceding-sibling::event[1]"/>
    <json:map>
      <json:string key="datetime"><xsl:value-of select="@datetime"/></json:string>
      <json:string key="level"><xsl:value-of select="@level"/></json:string>
      <json:number key="timestamp"><xsl:value-of select="f:timestamp-micro(.)"/></json:number>
      <json:number key="delta"><xsl:value-of select="@timestamp - $first-event/@timestamp"/></json:number>
      <xsl:if test="$previous-event"><json:number key="deltaWithLast"><xsl:value-of select="@timestamp - $previous-event/@timestamp" /></json:number></xsl:if>
      <json:boolean key="isSameTime"><xsl:value-of select="$previous-event/@datetime != @datetime"/></json:boolean>
      <json:string key="thread"><xsl:value-of select="@thread"/></json:string>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
      <json:string key="username"><xsl:value-of select="@username"/></json:string>

      <xsl:if test="@groupid">
        <json:number key="groupId"><xsl:value-of select="@groupid"/></json:number>
      </xsl:if>
      <xsl:if test="@group">
        <json:string key="group"><xsl:value-of select="@group"/></json:string>
      </xsl:if>
      <xsl:apply-templates select="access|job|publish|thread|mail" mode="json">
        <!-- We assume there can only be one -->
        <xsl:with-param name="key" select="name((access|job|publish|thread|mail)[1])"/>
      </xsl:apply-templates>
    </json:map>
  </xsl:template>

  <xsl:template match="event/access" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="type"><xsl:value-of select="@type"/></json:string>
      <json:string key="component"><xsl:value-of select="@component"/></json:string>
      <json:string key="path"><xsl:value-of select="@path"/></json:string>
      <json:string key="method"><xsl:value-of select="@method"/></json:string>
      <json:string key="status"><xsl:value-of select="@status"/></json:string>
      <xsl:if test="@uriid">
        <json:number key="uriid"><xsl:value-of select="@uriid"/></json:number>
      </xsl:if>
      <xsl:if test="@permission">
        <json:string key="permission"><xsl:value-of select="@permission"/></json:string>
      </xsl:if>
      <xsl:if test="@ip">
        <json:string key="ip"><xsl:value-of select="@ip"/></json:string>
      </xsl:if>
      <xsl:if test="@time">
        <json:number key="time"><xsl:value-of select="@time"/></json:number>
      </xsl:if>
      <xsl:if test="@docpath">
        <json:string key="docpath"><xsl:value-of select="@docpath"/></json:string>
      </xsl:if>
      <xsl:if test="@doctitle">
        <json:string key="doctitle"><xsl:value-of select="@doctitle"/></json:string>
      </xsl:if>
      <xsl:if test="@query">
        <json:string key="query"><xsl:value-of select="@query"/></json:string>
      </xsl:if>
    </json:map>
  </xsl:template>

  <xsl:template match="event/mail" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="type"><xsl:value-of select="if (@incoming = 'true') then 'incoming' else 'outgoing'"/></json:string>
      <json:string key="from"><xsl:value-of select="@from"/></json:string>
      <json:string key="to"><xsl:value-of select="@to"/></json:string>
      <json:string key="subject"><xsl:value-of select="@subject"/></json:string>
      <json:string key="message"><xsl:value-of select="@message | @error"/></json:string>
      <xsl:apply-templates select="throwable" mode="json">
        <xsl:with-param name="key">throwable</xsl:with-param>
      </xsl:apply-templates>
    </json:map>
  </xsl:template>

  <xsl:template match="event/thread" mode="json">
    <json:string key="message"><xsl:value-of select="if (threadid) then 'See other process' else if (string(@message) != '') then @message else
                                                     if (@status = 'INITIALISED') then 'Process started' else '--'" /></json:string>
    <xsl:if test="threadid">
      <json:string key="threadid"><xsl:value-of select="threadid" /></json:string>
      <xsl:if test="threadid/@group"><json:string key="threadgroup"><xsl:value-of select="threadid/@group" /></json:string></xsl:if>
    </xsl:if>
    <xsl:apply-templates select="throwable" mode="json">
      <xsl:with-param name="key">throwable</xsl:with-param>
    </xsl:apply-templates>
  </xsl:template>

  <xsl:template match="validation/file" mode="json">
    <json:map>
      <json:string key="name"><xsl:value-of select="@name" /></json:string>
      <json:string key="path"><xsl:value-of select="@path" /></json:string>
      <json:string key="location"><xsl:value-of select="@location" /></json:string>
      <json:string key="status"><xsl:value-of select="@status" /></json:string>
      <xsl:if test="message">
        <json:array key="messages">
          <xsl:for-each select="message">
            <json:string><xsl:value-of select="." /></json:string>
          </xsl:for-each>
        </json:array>
      </xsl:if>
    </json:map>
  </xsl:template>

  <xsl:template match="group|project" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:number key="id"><xsl:value-of select="@id"/></json:number>
      <json:string key="name"><xsl:value-of select="@name"/></json:string>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
      <json:string key="description"><xsl:value-of select="@description"/></json:string>
      <json:string key="type"><xsl:value-of select="name()"/></json:string>
      <json:string key="owner"><xsl:value-of select="@owner"/></json:string>
      <json:string key="indexversion"><xsl:value-of select="@indexversion"/></json:string>
      <xsl:for-each select="@access|@visibility|@template|@commenting|@moderation|@registration|@defaultrole|@defaultnotify|@detailstype|@relatedurl|message">
        <json:string key="{name()}"><xsl:value-of select="."/></json:string>
      </xsl:for-each>
      <xsl:for-each select="@common|@editurls">
        <json:boolean key="{name()}"><xsl:value-of select="."/></json:boolean>
      </xsl:for-each>
      <xsl:if test="properties">
        <json:map key="properties">
          <xsl:for-each select="properties/property">
            <json:string key="{@name}"><xsl:value-of select="@value" /></json:string>
          </xsl:for-each>
        </json:map>
      </xsl:if>
    </json:map>
  </xsl:template>

  <xsl:template match="properties" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <xsl:for-each select="property">
        <json:string key="{@name}"><xsl:value-of select="@value" /></json:string>
      </xsl:for-each>
    </json:map>
  </xsl:template>

  <xsl:template match="heading-ref" mode="json">
    <json:map>
      <json:string key="type">heading</json:string>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
      <json:string key="fragment"><xsl:value-of select="@fragment"/></json:string>
      <json:number key="index"><xsl:value-of select="@index"/></json:number>
      <xsl:if test="@uriid">
        <json:number key="uriid"><xsl:value-of select="@uriid"/></json:number>
      </xsl:if>
      <xsl:if test="@numbered">
        <json:boolean key="numbered"><xsl:value-of select="@numbered = 'true'"/></json:boolean>
        <json:string key="canonical"><xsl:value-of select="@canonical"/></json:string>
      </xsl:if>
      <xsl:if test="@prefix">
        <json:string key="prefix"><xsl:value-of select="@prefix"/></json:string>
      </xsl:if>
      <xsl:variable name="path" select="if (ancestor::part[document-ref/@path]) then ancestor::part[document-ref/@path][1]/document-ref/@path else ancestor::publication-tree/@path" />
      <xsl:if test="$path">
        <json:string key="path"><xsl:value-of select="$path"/></json:string>
      </xsl:if>
    </json:map>
  </xsl:template>

  <xsl:template match="host" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:number key="id"><xsl:value-of select="@id"/></json:number>
      <json:string key="name"><xsl:value-of select="@name"/></json:string>
      <json:boolean key="external"><xsl:value-of select="string(@external) = 'true'"/></json:boolean>
      <json:boolean key="virtual"><xsl:value-of select="string(@virtual) = 'true'"/></json:boolean>
      <json:array key="aliases">
        <xsl:for-each select="aliases/alias">
          <json:map>
            <json:number key="id"><xsl:value-of select="@id"/></json:number>
            <json:string key="name"><xsl:value-of select="@name"/></json:string>
          </json:map>
        </xsl:for-each>
      </json:array>
    </json:map>
  </xsl:template>

  <xsl:template match="identity-config" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="defaultAuthentication"><xsl:value-of select="@default-authentication" /></json:string>
      <xsl:if test="@default-provider">
        <json:string key="defaultProvider"><xsl:value-of select="@default-provider" /></json:string>
      </xsl:if>
      <xsl:if test="portal">
        <json:map key="portal">
          <json:string key="title"><xsl:value-of select="portal/@title" /></json:string>
          <json:string key="href"><xsl:value-of select="portal/@href" /></json:string>
          <xsl:if test="portal/@display">
            <json:string key="display"><xsl:value-of select="portal/@display" /></json:string>
          </xsl:if>
        </json:map>
      </xsl:if>
      <json:array key="providers">
        <xsl:for-each select="provider">
          <json:map>
            <xsl:if test="@id">
              <json:string key="id"><xsl:value-of select="@id"/></json:string>
            </xsl:if>
            <json:string key="clientId"><xsl:value-of select="@client-id"/></json:string>
            <json:string key="authorityUrl"><xsl:value-of select="@authority-url"/></json:string>
            <json:string key="description"><xsl:value-of select="@description"/></json:string>
            <json:string key="title"><xsl:value-of select="@title"/></json:string>
          </json:map>
        </xsl:for-each>
      </json:array>
      <json:array key="domains">
        <xsl:for-each select="domain">
          <json:map>
            <json:string key="name"><xsl:value-of select="@name"/></json:string>
            <json:string key="authentication"><xsl:value-of select="@authentication"/></json:string>
            <xsl:if test="@providers">
              <json:array key="providers">
                <xsl:for-each select="tokenize(@providers, ',')">
                  <json:string><xsl:value-of select="."/></json:string>
                </xsl:for-each>
              </json:array>
            </xsl:if>
            <!-- Legacy attribute might be removed -->
            <xsl:if test="@externalOnly">
              <json:boolean key="externalOnly"><xsl:value-of select="@external-only = 'true'"/></json:boolean>
            </xsl:if>
          </json:map>
        </xsl:for-each>
      </json:array>
    </json:map>
  </xsl:template>

  <xsl:template match="label-config" mode="json">
    <json:map>
      <json:string key="scope"><xsl:value-of select="@scope"/></json:string>
      <json:string key="template"><xsl:value-of select="@template"/></json:string>
      <xsl:if test="@updated">
        <json:string key="updated"><xsl:value-of select="@updated"/></json:string>
      </xsl:if>
      <json:array key="labelTypes">
        <xsl:for-each select="labels">
          <json:map>
            <json:string key="type"><xsl:value-of select="@type"/></json:string>
            <json:array key="labels">
              <xsl:apply-templates select="label" mode="json" />
            </json:array>
          </json:map>
        </xsl:for-each>
      </json:array>
    </json:map>
  </xsl:template>

  <xsl:template match="label" mode="json">
    <json:map>
      <json:string key="name"><xsl:value-of select="@name"/></json:string>
      <json:string key="description"><xsl:value-of select="@description"/></json:string>
      <xsl:if test="@color">
        <json:string key="color"><xsl:value-of select="@color"/></json:string>
      </xsl:if>
    </json:map>
  </xsl:template>

  <xsl:template match="legal" mode="json">
    <json:array>
      <xsl:for-each select="document">
        <json:map>
          <json:string key="name"><xsl:value-of select="@name"/></json:string>
          <json:string key="title"><xsl:value-of select="@title"/></json:string>
        </json:map>
      </xsl:for-each>
    </json:array>
  </xsl:template>

  <xsl:template match="portal" mode="json">
    <json:map>
      <json:string key="href"><xsl:value-of select="@href"/></json:string>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
      <json:string key="display"><xsl:value-of select="@display"/></json:string>
    </json:map>
  </xsl:template>

  <xsl:template match="server-message" mode="json">
    <json:map>
      <json:string key="text"><xsl:value-of select="."/></json:string>
      <json:string key="md5"><xsl:value-of select="@md5"/></json:string>
    </json:map>
  </xsl:template>

  <!--
    <document-type name="product" definedby="jb">
      <document-config>
        <creation>
          <title>product</title>
          <folder context="/assets" path="{0P100}"/>
          <filename value="{NN}-product" transform="charfolding,underscore,trim" editable="false"/>
          <document title="Product {N} - {$ui.title}" labels="mylabel1,mylabel2"/>
        </creation>
        ...
      </document-config>
      <document-template type="psml">
        <document level="portable">
          <t:param name="ps.title" default="[Untitled]"/>
          <t:param name="ps.author" default="Anonymous"/>
          ...
        </document>
      </document-template>
    </document-type>
  -->
  <xsl:template match="document-type" mode="json">
    <xsl:variable name="site-prefix" select="/root/header/path/@context"/>
    <json:map>
      <json:string key="name"><xsl:value-of select="@name" /></json:string>
      <xsl:if test="@css and not(ends-with(@css, 'psml-content.css'))">
        <json:string key="css"><xsl:value-of select="$site-prefix" /><xsl:value-of select="@css" /></json:string>
      </xsl:if>
      <!-- creation element -->
      <xsl:variable name="creation" select="document-config/creation" />
      <xsl:if test="$creation/@disable = 'true'"><json:boolean key="disabled">true</json:boolean></xsl:if>
      <json:string key="title"><xsl:value-of select="$creation/title" /></json:string>
      <json:string key="description"><xsl:value-of select="$creation/description" /></json:string>
      <json:string key="folder"><xsl:value-of select="$creation/folder/@context" /></json:string>
      <json:string key="path"><xsl:value-of select="$creation/folder/@path" /></json:string>
      <json:map key="filename">
        <json:string key="value"><xsl:value-of select="$creation/filename/@value" /></json:string>
        <xsl:if test="$creation/filename/@transform">
          <json:string key="transform"><xsl:value-of select="$creation/filename/@transform" /></json:string>
        </xsl:if>
        <json:boolean key="editable"><xsl:value-of select="if ($creation/filename[@editable = 'false']) then 'false' else 'true'" /></json:boolean>
      </json:map>
      <json:map key="document">
        <json:string key="title"><xsl:value-of select="$creation/document/@title" /></json:string>
        <json:string key="labels"><xsl:value-of select="$creation/document/@labels"/></json:string>
        <json:string key="docid"><xsl:value-of select="$creation/document/@docid"/></json:string>
        <json:string key="description"><xsl:value-of select="$creation/document/@description"/></json:string>
        <xsl:variable name="type-labels"    select="tokenize($creation/document/@labels, ',')[normalize-space(.) != '']" />
        <xsl:variable name="general-labels" select="document-config/labeling/labels[@type='document']/label[normalize-space(@name) != '']/@name" />
        <json:array key="allLabels">
          <xsl:for-each select="distinct-values(json:tokenize(concat(string-join($type-labels, ','), ',', string-join($general-labels, ',')), ','))[normalize-space(.) != '']">
            <xsl:sort select="." />
            <json:string><xsl:value-of select="normalize-space(.)" /></json:string>
          </xsl:for-each>
        </json:array>
        <json:array key="typeLabels">
          <xsl:for-each select="$type-labels">
            <xsl:sort select="." />
            <json:string><xsl:value-of select="normalize-space(.)" /></json:string>
          </xsl:for-each>
        </json:array>
      </json:map>
      <xsl:if test="document-template/document/t:param">
        <json:array key="parameters">
          <xsl:for-each select="document-template/document/t:param">
            <json:map>
              <json:string key="name"><xsl:value-of select="@name" /></json:string>
              <json:string key="title"><xsl:value-of select="@title" /></json:string>
              <json:string key="type"><xsl:value-of select="@type" /></json:string>
              <json:string key="default"><xsl:value-of select="@default" /></json:string>
              <json:string key="description"><xsl:value-of select="@description" /></json:string>
            </json:map>
          </xsl:for-each>
        </json:array>
        <xsl:if test="editor-configs/editor-config[@name = 'PSMLCreation']">
          <json:array key="config">
            <xsl:apply-templates select="editor-configs/editor-config[@name = 'PSMLCreation']/field" mode="json-config"/>
          </json:array>
        </xsl:if>
      </xsl:if>
    </json:map>
  </xsl:template>

  <!--
    <member id="1" firstname="Christophe" surname="Lauret" username="clauret" status="activated">
      <fullname>Christophe Lauret</fullname>
    </member>
  -->
  <xsl:template match="member" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <xsl:if test="@id">
        <json:number key="id"><xsl:value-of select="@id"/></json:number>
      </xsl:if>
      <json:string key="firstname"><xsl:value-of select="@firstname"/></json:string>
      <json:string key="surname"><xsl:value-of select="@surname"/></json:string>
      <json:string key="username"><xsl:value-of select="@username"/></json:string>
      <xsl:if test="@status">
        <json:string key="status"><xsl:value-of select="@status"/></json:string>
      </xsl:if>
      <xsl:if test="@email">
        <json:string key="email"><xsl:value-of select="@email"/></json:string>
      </xsl:if>
      <json:string key="fullname"><xsl:value-of select="fullname"/></json:string>
      <xsl:for-each select="@created|@activated|@lastlogin|externalid|@lastpasswordchange">
        <json:string key="{name()}"><xsl:value-of select="."/></json:string>
      </xsl:for-each>
      <xsl:for-each select="@locked|@onvacation|@attachments">
        <json:boolean key="{name()}"><xsl:value-of select=". = 'true'"/></json:boolean>
      </xsl:for-each>
      <xsl:if test="@admin">
        <json:boolean key="admin"><xsl:value-of select="@admin = 'true'"/></json:boolean>
      </xsl:if>
    </json:map>
  </xsl:template>

  <xsl:template match="membership" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <xsl:if test="@id">
        <json:number key="id"><xsl:value-of select="@id"/></json:number>
      </xsl:if>
      <json:boolean key="emailListed"><xsl:value-of select="@email-listed = 'true'"/></json:boolean>
      <json:string key="notification"><xsl:value-of select="@notification"/></json:string>
      <json:string key="status"><xsl:value-of select="@status"/></json:string>
      <xsl:if test="@deleted">
        <json:boolean key="deleted"><xsl:value-of select="@deleted = 'true'"/></json:boolean>
      </xsl:if>
      <xsl:for-each select="@role|@subgroups|@override|@created">
        <json:string key="{name()}"><xsl:value-of select="."/></json:string>
      </xsl:for-each>
      <xsl:apply-templates select="member" mode="json">
        <xsl:with-param name="key">member</xsl:with-param>
      </xsl:apply-templates>
      <xsl:apply-templates select="group" mode="json">
        <xsl:with-param name="key">group</xsl:with-param>
      </xsl:apply-templates>
      <xsl:apply-templates select="project" mode="json">
        <xsl:with-param name="key">project</xsl:with-param>
      </xsl:apply-templates>
      <xsl:apply-templates select="details" mode="json">
        <xsl:with-param name="key">details</xsl:with-param>
      </xsl:apply-templates>
    </json:map>
  </xsl:template>

  <xsl:template match="membership/details" mode="json">
    <xsl:param name="key"/>
    <json:array>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <xsl:for-each select="field">
        <json:map>
          <json:string key="name"><xsl:value-of select="@name"/></json:string>
          <json:string key="title"><xsl:value-of select="@title"/></json:string>
          <json:number key="position"><xsl:value-of select="@position"/></json:number>
          <json:boolean key="editable"><xsl:value-of select="@editable = 'true'"/></json:boolean>
          <xsl:if test="@type">
            <json:string key="type"><xsl:value-of select="@type"/></json:string>
          </xsl:if>
          <json:string key="value"><xsl:value-of select="."/></json:string>
        </json:map>
      </xsl:for-each>
    </json:array>
  </xsl:template>

  <xsl:template match="para-ref" mode="json">
    <json:map>
      <json:string key="type">para</json:string>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
      <json:string key="fragment"><xsl:value-of select="@fragment"/></json:string>
      <json:number key="index"><xsl:value-of select="@index"/></json:number>
      <xsl:if test="@uriid">
        <json:number key="uriid"><xsl:value-of select="@uriid"/></json:number>
      </xsl:if>
      <xsl:if test="@numbered">
        <json:boolean key="numbered"><xsl:value-of select="@numbered = 'true'"/></json:boolean>
      </xsl:if>
      <xsl:if test="@prefix">
        <json:string key="prefix"><xsl:value-of select="@prefix"/></json:string>
      </xsl:if>
      <xsl:variable name="path" select="if (ancestor::part[document-ref/@path]) then ancestor::part[document-ref/@path][1]/document-ref/@path else ancestor::publication-tree/@path" />
      <xsl:if test="$path">
        <json:string key="path"><xsl:value-of select="$path"/></json:string>
      </xsl:if>
    </json:map>
  </xsl:template>

  <!--

   status="activated" admin="true"
  -->

  <xsl:template match="part" mode="json">
    <xsl:param name="key"/>
    <xsl:param name="prefix"/>
    <xsl:variable name="id" select="if ($prefix) then concat($prefix, '-', position()) else position()"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="partid"><xsl:value-of select="$id"/></json:string>
      <json:number key="level"><xsl:value-of select="@level"/></json:number>
      <xsl:if test="@uriid">
        <json:number key="uriid"><xsl:value-of select="@uriid"/></json:number>
      </xsl:if>
      <json:array key="refs">
        <xsl:apply-templates select="heading-ref|para-ref|document-ref|transclusion" mode="json" />
      </json:array>
      <xsl:variable name="prefix" select="if (heading-ref) then $id else document-ref/@uriid"/>
      <json:array key="parts">
        <xsl:apply-templates select="part" mode="json">
          <xsl:with-param name="prefix" select="$prefix"/>
        </xsl:apply-templates>
      </json:array>
    </json:map>
  </xsl:template>

  <xsl:template match="persistent-token" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="data"><xsl:value-of select="@data"/></json:string>
      <json:string key="expires"><xsl:value-of select="@expires"/></json:string>
      <json:boolean key="expired"><xsl:value-of select="@expired = 'true'"/></json:boolean>
      <json:number key="id"><xsl:value-of select="@id"/></json:number>
      <json:string key="issued"><xsl:value-of select="@issued"/></json:string>
      <json:number key="lifetime"><xsl:value-of select="@lifetime"/></json:number>
      <json:string key="scope"><xsl:value-of select="@scope"/></json:string>
      <json:string key="type"><xsl:value-of select="name()"/></json:string>
      <!-- Only available when it is issued -->
      <xsl:if test="@token">
        <json:string key="token"><xsl:value-of select="@token"/></json:string>
      </xsl:if>
      <xsl:apply-templates select="client" mode="json">
        <xsl:with-param name="key" select="'client'"/>
      </xsl:apply-templates>
      <xsl:apply-templates select="member" mode="json">
        <xsl:with-param name="key" select="'member'"/>
      </xsl:apply-templates>
    </json:map>
  </xsl:template>

  <xsl:template match="publication" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="id"><xsl:value-of select="@id"/></json:string>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
      <json:string key="type"><xsl:value-of select="@type"/></json:string>
      <xsl:if test="@hostid">
        <json:number key="hostid"><xsl:value-of select="@hostid"/></json:number>
      </xsl:if>
      <xsl:if test="@rooturiid">
        <json:number key="rooturiid"><xsl:value-of select="@rooturiid"/></json:number>
      </xsl:if>
      <xsl:if test="@defaultgroupid">
        <json:number key="defaultgroupid"><xsl:value-of select="@defaultgroupid"/></json:number>
      </xsl:if>
      <xsl:apply-templates select="publication-tree" mode="json">
        <xsl:with-param name="key">tree</xsl:with-param>
      </xsl:apply-templates>
      <!-- for webhook events -->
      <xsl:for-each select="added|modified|removed|deleted">
        <json:array key="{name()}">
          <xsl:apply-templates select="uri" mode="json" />
        </json:array>
      </xsl:for-each>
    </json:map>
  </xsl:template>

  <xsl:template match="publication-tree" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:number key="uriid"><xsl:value-of select="@uriid"/></json:number>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
      <xsl:if test="@path">
        <json:string key="path"><xsl:value-of select="@path"/></json:string>
      </xsl:if>
      <json:boolean key="content"><xsl:value-of select="@content = 'true'"/></json:boolean>
      <json:array key="refs">
        <xsl:apply-templates select="heading-ref|para-ref|document-ref" mode="json" />
      </json:array>
      <json:array key="parts">
        <xsl:apply-templates select="part" mode="json">
          <xsl:with-param name="prefix" select="@uriid"/>
        </xsl:apply-templates>
      </json:array>
    </json:map>
  </xsl:template>

  <xsl:template match="publication-type" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="name"><xsl:value-of select="@name"/></json:string>
      <json:string key="definedby"><xsl:value-of select="@definedby"/></json:string>
    </json:map>
  </xsl:template>

  <xsl:template match="throwable" mode="json">
    <xsl:param name="key"/>
    <json:string>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <xsl:choose>
        <xsl:when test="@message">
          <xsl:if test="@classname">
            <xsl:value-of select="@classname" />
            <xsl:text>: </xsl:text>
          </xsl:if>
          <xsl:value-of select="@message"/>
          <xsl:text>&#13;</xsl:text>
          <xsl:for-each select="step">
            <xsl:text>&#xA0;&#xA0;&#xA0;&#xA0;</xsl:text>
            <xsl:value-of select="."/>&#xA0;[<xsl:value-of select="@location"/>]<xsl:text>&#13;</xsl:text>
          </xsl:for-each>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="."/>
        </xsl:otherwise>
      </xsl:choose>
    </json:string>
  </xsl:template>

  <xsl:template match="ticket" mode="json">
    <xsl:param name="key"/>
    <xsl:param name="indexing" select="false()"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="id"><xsl:value-of select="@id"/></json:string>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
      <json:number key="infos"><xsl:value-of select="if ($indexing) then floor(@infos div 2) else @infos"/></json:number>
      <json:number key="warnings"><xsl:value-of select="@warnings"/></json:number>
      <json:number key="errors"><xsl:value-of select="@errors"/></json:number>
      <json:string key="started"><xsl:value-of select="@started"/></json:string>
      <json:string key="ended"><xsl:value-of select="@ended"/></json:string>
      <json:number key="duration"><xsl:value-of select="@duration"/></json:number>
      <xsl:if test="@size"><json:number key="size"><xsl:value-of select="@size"/></json:number></xsl:if>
      <xsl:if test="@hour"><json:number key="hour"><xsl:value-of select="@hour"/></json:number></xsl:if>
      <xsl:if test="@author"><json:string key="author"><xsl:value-of select="@author"/></json:string></xsl:if>
      <json:boolean key="hasStart"><xsl:value-of select="if (@start = 'false') then 'false' else 'true'"/></json:boolean>
      <json:boolean key="hasEnd"><xsl:value-of select="if (@complete = 'false') then 'false' else 'true'"/></json:boolean>
      <xsl:if test="../self::group[@id = '-1']">
        <json:boolean key="server">true</json:boolean>
        <xsl:if test="ends-with(@id, '-0')">
          <json:string key="ticketId"><xsl:value-of select="string-join(tokenize(@id, '-')[position() lt last()], '-')"/></json:string>
        </xsl:if>
      </xsl:if>
      <xsl:if test="../self::group[@id != '-1']">
        <json:number key="groupid"><xsl:value-of select="../@id"/></json:number>
        <json:string key="groupname"><xsl:value-of select="../@name"/></json:string>
        <xsl:if test="ends-with(@id, concat('-', ../@id))">
          <json:string key="ticketId"><xsl:value-of select="string-join(tokenize(@id, '-')[position() lt last()], '-')"/></json:string>
        </xsl:if>
      </xsl:if>
    </json:map>
  </xsl:template>

  <xsl:template match="uri" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <xsl:if test="@archived">
        <json:boolean key="archived"><xsl:value-of select="@archived = 'true'"/></json:boolean>
      </xsl:if>
      <json:string key="created"><xsl:value-of select="@created"/></json:string>
      <json:string key="decodedpath"><xsl:value-of select="@decodedpath"/></json:string>
      <json:string key="description"><xsl:value-of select="description"/></json:string>
      <json:string key="displaytitle"><xsl:value-of select="displaytitle"/></json:string>
      <json:string key="docid"><xsl:value-of select="@docid"/></json:string>
      <json:string key="documenttype"><xsl:value-of select="@documenttype"/></json:string>
      <json:string key="urltype"><xsl:value-of select="@urltype"/></json:string>
      <json:boolean key="external"><xsl:value-of select="@external = 'true'"/></json:boolean>
      <json:boolean key="folder"><xsl:value-of select="@folder = 'true' or @mediatype = 'folder'"/></json:boolean><!-- TODO Check which one is correct! -->
      <json:string key="host"><xsl:value-of select="@host"/></json:string>
      <xsl:if test="@id"><json:number key="id"><xsl:value-of select="@id"/></json:number></xsl:if>
      <json:array key="labels">
        <xsl:for-each select="tokenize(labels, ',')[. != '']">
          <json:string><xsl:value-of select="." /></json:string>
        </xsl:for-each>
      </json:array>
      <xsl:if test="@mediatype">
        <json:string key="mediatype"><xsl:value-of select="@mediatype"/></json:string>
      </xsl:if>
      <json:string key="modified"><xsl:value-of select="@modified"/></json:string>
      <json:string key="path"><xsl:value-of select="@path"/></json:string>
      <xsl:if test="@port">
        <json:number key="port"><xsl:value-of select="@port"/></json:number>
      </xsl:if>
      <json:string key="scheme"><xsl:value-of select="@scheme"/></json:string>
      <xsl:if test="@size">
        <json:number key="size"><xsl:value-of select="@size"/></json:number>
      </xsl:if>
      <xsl:if test="@source">
        <json:string key="source"><xsl:value-of select="@source"/></json:string>
      </xsl:if>
      <json:string key="title"><xsl:value-of select="@title"/></json:string>
      <xsl:apply-templates select="sharing" mode="json">
        <xsl:with-param name="key">sharing</xsl:with-param>
      </xsl:apply-templates>
    </json:map>
  </xsl:template>

  <xsl:template match="groupfolder" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:boolean key="external"><xsl:value-of select="@external"/></json:boolean>
      <json:boolean key="public"><xsl:value-of select="@public"/></json:boolean>
      <json:string key="host"><xsl:value-of select="@host"/></json:string>
      <json:number key="id"><xsl:value-of select="@id"/></json:number>
      <json:string key="path"><xsl:value-of select="@path"/></json:string>
      <json:number key="port"><xsl:value-of select="@port"/></json:number>
      <json:string key="scheme"><xsl:value-of select="@scheme"/></json:string>
      <xsl:apply-templates select="sharing" mode="json">
        <xsl:with-param name="key">sharing</xsl:with-param>
      </xsl:apply-templates>
    </json:map>
  </xsl:template>

  <xsl:template match="sharing" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="status"><xsl:value-of select="@status"/></json:string>
      <json:boolean key="public"><xsl:value-of select="@public = 'true'"/></json:boolean>
      <xsl:if test="@hidden">
        <json:number key="hidden"><xsl:value-of select="@hidden"/></json:number>
      </xsl:if>
      <json:array key="share">
        <xsl:for-each select="share">
          <json:map>
            <json:boolean key="owner"><xsl:value-of select="@owner = 'true'"/></json:boolean>
            <json:boolean key="editable"><xsl:value-of select="@editable = 'true'"/></json:boolean>
            <json:boolean key="addcomments"><xsl:value-of select="@addcomments = 'true'"/></json:boolean>
            <json:boolean key="shareedits"><xsl:value-of select="@shareedits = 'true'"/></json:boolean>
            <json:boolean key="sharexrefs"><xsl:value-of select="@sharexrefs = 'true'"/></json:boolean>
            <json:string key="groupspecific"><xsl:value-of select="@groupspecific"/></json:string>
            <xsl:apply-templates select="group" mode="json">
              <xsl:with-param name="key">group</xsl:with-param>
            </xsl:apply-templates>
          </json:map>
        </xsl:for-each>
      </json:array>
    </json:map>
    <!--  <sharing status="private">-->
    <!--    <share owner="true" editable="true" addcomments="true" shareedits="true" sharexrefs="true">-->
    <!--      <group id="5" name="test-clauret" title="Christophe's tests" description="Christophe's testing group" owner="test" access="member" common="false"/>-->
    <!--    </share>-->
    <!--  </sharing>-->
  </xsl:template>

  <xsl:template match="part/transclusion" mode="json">
    <json:map>
      <json:string key="type">transclusion</json:string>
      <json:number key="uriid"><xsl:value-of select="@uriid"/></json:number>
      <json:string key="fragment"><xsl:value-of select="@fragment"/></json:string>
      <json:number key="position"><xsl:value-of select="@position"/></json:number>
    </json:map>
  </xsl:template>

  <xsl:template match="version" mode="json">
    <json:map>
      <json:number key="id"><xsl:value-of select="@id"/></json:number>
      <json:string key="name"><xsl:value-of select="@name"/></json:string>
      <json:string key="created"><xsl:value-of select="@created"/></json:string>
      <xsl:if test="@publicationid">
        <json:string key="publicationid"><xsl:value-of select="@publicationid"/></json:string>
      </xsl:if>
      <xsl:apply-templates select="author" mode="json">
        <xsl:with-param name="key" select="'author'"/>
      </xsl:apply-templates>
      <json:string key="description"><xsl:value-of select="description"/></json:string>
      <xsl:if test="labels">
        <json:array key="labels">
          <xsl:for-each select="tokenize(labels, ',')">
            <json:string><xsl:value-of select="."/></json:string>
          </xsl:for-each>
        </json:array>
      </xsl:if>
    </json:map>
  </xsl:template>

  <xsl:template match="webevent" mode="json">
    <xsl:param name="key"/>
    <xsl:variable name="indented-xml" as="element()">
      <output:serialization-parameters xmlns:output="http://www.w3.org/2010/xslt-xquery-serialization">
        <output:method value="xml"/>
        <output:indent value="yes"/>
      </output:serialization-parameters>
    </xsl:variable>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="id"><xsl:value-of select="@id"/></json:string>
      <json:string key="datetime"><xsl:value-of select="@datetime"/></json:string>
      <json:string key="type"><xsl:value-of select="@type"/></json:string>
      <json:boolean key="queued"><xsl:value-of select="@queued = 'true'"/></json:boolean>
      <json:array key="groups">
        <xsl:for-each select="event/group">
          <json:map>
            <json:number key="id"><xsl:value-of select="@id"/></json:number>
            <json:string key="name"><xsl:value-of select="@name"/></json:string>
          </json:map>
        </xsl:for-each>
      </json:array>
      <xsl:for-each select="(comment|group|groupfolder|properties|member|membership|project|publication|task|uri|webhook|workflow)[1]">
        <xsl:apply-templates select="." mode="json">
          <xsl:with-param name="key" select="name(.)"/>
        </xsl:apply-templates>
      </xsl:for-each>
      <json:string key="xml"><xsl:value-of select="serialize(., $indented-xml)"/></json:string>
    </json:map>
  </xsl:template>

  <xsl:template match="webhook" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:number key="id"><xsl:value-of select="@id"/></json:number>
      <json:string key="created"><xsl:value-of select="@created"/></json:string>
      <json:array key="events">
        <xsl:for-each select="tokenize(@events, ',')">
          <json:string><xsl:value-of select="."/></json:string>
        </xsl:for-each>
      </json:array>
      <json:string key="format"><xsl:value-of select="@format"/></json:string>
      <json:array key="groups">
        <xsl:for-each select="tokenize(@groups, ',')">
          <json:string><xsl:value-of select="."/></json:string>
        </xsl:for-each>
      </json:array>
      <json:boolean key="insecuressl"><xsl:value-of select="@insecuressl = 'true'"/></json:boolean>
      <json:string key="modified"><xsl:value-of select="@modified"/></json:string>
      <json:string key="name"><xsl:value-of select="@name"/></json:string>
      <json:string key="object"><xsl:value-of select="@object"/></json:string>
      <json:array key="projects">
        <xsl:for-each select="tokenize(@projects, ',')">
          <json:string><xsl:value-of select="."/></json:string>
        </xsl:for-each>
      </json:array>
      <json:boolean key="server"><xsl:value-of select="@server = 'true'"/></json:boolean>
      <json:string key="status"><xsl:value-of select="@status"/></json:string>
      <json:string key="url"><xsl:value-of select="@url"/></json:string>
      <xsl:apply-templates select="client" mode="json">
        <xsl:with-param name="key">client</xsl:with-param>
      </xsl:apply-templates>
    </json:map>
  </xsl:template>

  <xsl:template match="widget" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <json:string key="name"><xsl:value-of select="@name"/></json:string>
      <xsl:if test="@start">
        <json:string key="start"><xsl:value-of select="@start"/></json:string>
      </xsl:if>
      <xsl:if test="@size">
        <json:string key="size"><xsl:value-of select="@size"/></json:string>
      </xsl:if>
      <xsl:if test="@rowspan and @rowspan castable as xs:positiveInteger">
        <json:number key="rowspan"><xsl:value-of select="@rowspan"/></json:number>
      </xsl:if>
      <json:map key="parameters">
        <xsl:for-each select="parameters[1]/@*[not(name() = 'is' or name() = 'size')]">
          <xsl:choose>
            <xsl:when test=". = 'true' or . = 'false'">
              <json:boolean key="{name()}"><xsl:value-of select="."/></json:boolean>
            </xsl:when>
            <xsl:when test=". castable as xs:decimal">
              <json:number key="{name()}"><xsl:value-of select="."/></json:number>
            </xsl:when>
            <xsl:otherwise>
              <json:string key="{name()}"><xsl:value-of select="."/></json:string>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:for-each>
      </json:map>
    </json:map>
  </xsl:template>

  <xsl:template match="workflow|task" mode="json">
    <xsl:param name="key"/>
    <json:map>
      <xsl:if test="$key"><xsl:attribute name="key" select="$key"/></xsl:if>
      <xsl:if test="@id">
        <json:number key="id"><xsl:value-of select="@id"/></json:number>
      </xsl:if>
      <json:string key="status"><xsl:value-of select="@status"/></json:string>
      <xsl:if test="@priority">
        <json:string key="priority"><xsl:value-of select="@priority"/></json:string>
      </xsl:if>
      <xsl:if test="@due">
        <json:string key="due"><xsl:value-of select="@due"/></json:string>
      </xsl:if>
      <xsl:if test="assignedto">
        <xsl:apply-templates select="assignedto" mode="json">
          <xsl:with-param name="key">assignedto</xsl:with-param>
        </xsl:apply-templates>
      </xsl:if>
      <json:array key="comments">
        <xsl:apply-templates select="comment" mode="json"/>
      </json:array>
    </json:map>
  </xsl:template>


  <xsl:template match="action/target" mode="json">
    <json:map>
      <json:string key="project"><xsl:value-of select="../../@project"/></json:string>
      <json:string key="type"><xsl:value-of select="../@type"/></json:string>
      <json:string key="source"><xsl:value-of select="../source/@type"/></json:string>
      <xsl:if test="../source/@extension"><json:string key="extension"><xsl:value-of select="../source/@extension"/></json:string></xsl:if>
      <xsl:if test="../source/@document-type"><json:string key="doctype"><xsl:value-of select="../source/@document-type"/></json:string></xsl:if>
      <xsl:if test="../source/@url-type"><json:string key="urltype"><xsl:value-of select="../source/@url-type"/></json:string></xsl:if>
      <json:string key="target"><xsl:value-of select="@name"/></json:string>
      <json:string key="previewname"><xsl:value-of select="@previewname"/></json:string>
      <json:string key="description"><xsl:value-of select="description"/></json:string>
      <json:boolean key="developer"><xsl:value-of select="@developer = 'true' or @perspective = 'developer'" /></json:boolean>
      <xsl:if test="@configfile"><json:string key="configfile"><xsl:value-of select="@configfile"/></json:string></xsl:if>
      <xsl:if test="../../@override = 'true'"><json:boolean key="override">true</json:boolean></xsl:if>
      <json:array key="parameters">
        <xsl:for-each select="param">
          <json:map>
            <xsl:for-each select="@*">
              <xsl:choose>
                <xsl:when test="name() = 'min' or name() = 'max' or (name() = 'default' and matches(., '^\d+$'))">
                  <json:number key="{name()}"><xsl:value-of select="." /></json:number>
                </xsl:when>
                <xsl:when test=". = 'true' or . = 'false'">
                  <json:boolean key="{name()}"><xsl:value-of select="."/></json:boolean>
                </xsl:when>
                <xsl:otherwise><json:string key="{name()}"><xsl:value-of select="."/></json:string></xsl:otherwise>
              </xsl:choose>
            </xsl:for-each>
            <xsl:if test="value">
              <json:array key="values">
                <xsl:for-each select="value"><json:string><xsl:value-of select="." /></json:string></xsl:for-each>
              </json:array>
            </xsl:if>
            <xsl:if test="autosuggest">
              <json:map key="autosuggest">
                <xsl:for-each select="autosuggest/@*"><json:string key="{name()}"><xsl:value-of select="." /></json:string></xsl:for-each>
              </json:map>
            </xsl:if>
          </json:map>
        </xsl:for-each>
      </json:array>
    </json:map>
  </xsl:template>

  <xsl:function name="f:camelify" as="xs:string">
    <xsl:param name="name"/>
    <xsl:value-of>
      <xsl:analyze-string select="$name" regex="-([a-z])">
        <xsl:matching-substring><xsl:value-of select="upper-case(regex-group(1))"/></xsl:matching-substring>
        <xsl:non-matching-substring><xsl:value-of select="."/></xsl:non-matching-substring>
      </xsl:analyze-string>
    </xsl:value-of>
  </xsl:function>

  <!--
    Generates a unique timestamp in micro-seconds for an event
  -->
  <xsl:function name="f:timestamp-micro" as="xs:string?">
    <xsl:param name="event" as="element(event)"/>
    <xsl:choose>
      <xsl:when test="$event/@timestamp">
        <xsl:value-of select="concat($event/@timestamp, format-number(count($event/preceding-sibling::event[@timestamp = $event/@timestamp]), '000'))"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="unix-time" select="(xs:dateTime($event/@datetime) - xs:dateTime('1970-01-01T00:00:00')) div xs:dayTimeDuration('PT0.001S')" as="xs:numeric"/>
        <xsl:value-of select="concat(floor($unix-time), format-number(count($event/preceding-sibling::event[@datetime = $event/@datetime]), '000'))"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>

  <!-- TODO We should use the function below, the JSON should be as close as possible to XML -->

  <!--
    Return a string representation of a duration
    @param time The time in seconds
  -->
  <xsl:function name="f:duration-short" as="xs:string">
    <xsl:param name="time"/>
    <xsl:value-of>
      <xsl:choose>
        <xsl:when test="not($time)">--</xsl:when>
        <xsl:when test="$time = 0"><i>eternal</i></xsl:when>
        <xsl:otherwise>
          <xsl:variable name="duration" select="xs:duration(concat('PT', $time, 'S'))"/>
          <xsl:variable name="days"     select="days-from-duration($duration)"/>
          <xsl:variable name="hours"    select="hours-from-duration($duration)"/>
          <xsl:variable name="minutes"  select="minutes-from-duration($duration)"/>
          <xsl:variable name="seconds"  select="seconds-from-duration($duration)"/>
          <xsl:choose>
            <xsl:when test="$days gt 1"><xsl:value-of select="$days"/>d </xsl:when>
            <xsl:when test="$hours gt 1"><xsl:value-of select="$hours"/>h </xsl:when>
            <xsl:when test="$minutes gt 1"><xsl:value-of select="$minutes"/>m </xsl:when>
            <xsl:otherwise><xsl:value-of select="$seconds"/>s </xsl:otherwise>
          </xsl:choose>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:value-of>
  </xsl:function>

</xsl:stylesheet>
