<?xml version="1.0"?>
<!--
  Define useful functions for the templates
-->
<xsl:stylesheet version="2.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:f="http://www.pageseeder.com/function"
                exclude-result-prefixes="#all">

<!--
  Return the formatted expiration as "x days, y hours, z minutes".

  @param expiration  the <token ... expiration="P1DT6H"/> element (optional)
  @param default     the default expiration (e.g. 'PT12H')
-->
<xsl:function name="f:get-expiration">
  <xsl:param name="expiration" />
  <xsl:param name="default" />

  <xsl:variable name="expires" select="if ($expiration) then $expiration/@expiration else xs:duration($default)" />
  <xsl:value-of select="f:format-duration($expires)" />
</xsl:function>

<!--
  Return a formatted duration as "x days, y hours, z minutes".

  @param d  the ISO 8601 duration
-->
<xsl:function name="f:format-duration">
  <xsl:param name="d" as="xs:duration"/>

  <xsl:variable name="days"    select="days-from-duration($d)" />
  <xsl:variable name="hours"   select="hours-from-duration($d)" />
  <xsl:variable name="minutes" select="minutes-from-duration($d)" />
  <xsl:if test="$days != 0">
    <xsl:value-of select="concat($days, ' day', if ($days != 1) then 's' else '')" />
  </xsl:if>
  <xsl:if test="$hours != 0">
    <xsl:if test="$days != 0">, </xsl:if>
    <xsl:value-of select="concat($hours, ' hour', if ($hours != 1) then 's' else '')" />
  </xsl:if>
  <xsl:if test="$minutes != 0">
    <xsl:if test="$days != 0 or $hours != 0">, </xsl:if>
    <xsl:value-of select="concat($minutes, ' minute', if ($minutes != 1) then 's' else '')" />
  </xsl:if>
</xsl:function>

<!--
  Return a formatted link

  @param url  The URL of the link
  @param text The text to display
-->
<xsl:function name="f:link" as="element(a)">
  <xsl:param name="url" />
  <xsl:param name="text" />
  <a href="{$url}"><xsl:value-of select="$text" /></a>
</xsl:function>

<!--
  Return the URL for an external URI

  @param uri  The external URI
-->
<xsl:function name="f:url">
  <xsl:param name="uri" />
  <xsl:value-of select="$uri/@scheme" />
  <xsl:text>://</xsl:text>
  <xsl:value-of select="$uri/@host" />
  <xsl:if test="not($uri/@scheme = 'http' and $uri/@port = '80') and not($uri/@scheme = 'https' and $uri/@port = '443')">
    <xsl:text>:</xsl:text>
    <xsl:value-of select="$uri/@port" />
  </xsl:if>
  <xsl:value-of select="$uri/@path" />
</xsl:function>

<!--
  Return a formatted button

  @param url  The URL of the link
  @param text The text to display
-->
<xsl:function name="f:button" as="element(div)">
  <xsl:param name="url" />
  <xsl:param name="text" />
  <div class="button-wrap"><!--[if mso]>
  <v:roundrect xmlns:v="urn:schemas-microsoft-com:vml" xmlns:w="urn:schemas-microsoft-com:office:word" href="http://" style="height:40px;v-text-anchor:middle;width:200px;" arcsize="10%" stroke="f" fillcolor="#016ebc">
    <w:anchorlock/>
    <center>
  <![endif]-->
    <a href="{$url}" class="button-link"><xsl:value-of select="$text" /></a>
    <!--[if mso]>
      </center>
    </v:roundrect>
  <![endif]--></div>
</xsl:function>


<!--
  Extracts the host from a URL

  @param url The full URL

  @return the host name
-->
<xsl:function name="f:hostname">
  <xsl:param name="url" />
  <xsl:analyze-string regex="https?://([\w\-\.]+)([:/].*)?" select="$url">
    <xsl:matching-substring>
      <xsl:value-of select="regex-group(1)"/>
    </xsl:matching-substring>
    <xsl:non-matching-substring>
      <xsl:value-of select="$url"/>
    </xsl:non-matching-substring>
  </xsl:analyze-string>
</xsl:function>

<!--
  Finds the specified strings in the given text.

  @param strings the strings to look for in the text
  @param text    The text to search

  @return the strings that where found in the texts.
-->
<xsl:function name="f:find-string">
  <xsl:param name="strings"/>
  <xsl:param name="text"/>
  <xsl:variable name="upper" select="upper-case($text)"/>
  <xsl:for-each select="tokenize($strings, ',')">
    <xsl:if test="contains($upper, normalize-space(upper-case(.)))">
      <xsl:value-of select="normalize-space(.)" />
    </xsl:if>
  </xsl:for-each>
</xsl:function>

<!--
  Build the name for the icon corresponding to the mediatype of the URI provided

  Note: This file should be kept in sync with 'mediatypes.xsl' used for the user interface.

  @param uri the uri

  @return the icon name
 -->
<xsl:function name="f:mediatype-icon">
  <xsl:param name="uri" />
  <xsl:choose>
    <!-- Specific PageSeeder -->
    <xsl:when test="$uri/@mediatype = 'application/vnd.pageseeder.psml+xml'">media-psml.png</xsl:when>
    <xsl:when test="$uri/@external = 'true'">media-url.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'application/vnd.pageseeder.url+xml'">media-url.png</xsl:when>
    <!-- Specific MIME -->
    <xsl:when test="$uri/@mediatype = 'application/pdf'">media-pdf.png</xsl:when>
    <xsl:when test="starts-with($uri/@mediatype, 'application/vnd.ms-word') or $uri/@mediatype = 'application/msword'">media-word.png</xsl:when>
    <xsl:when test="starts-with($uri/@mediatype, 'application/vnd.ms-excel')">media-excel.png</xsl:when>
    <xsl:when test="starts-with($uri/@mediatype, 'application/vnd.ms-powerpoint')">media-powerpoint.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'">media-word.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'">media-excel.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'application/vnd.openxmlformats-officedocument.presentationml.presentation'">media-powerpoint.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'text/properties'">media-properties.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'text/html'">media-code.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'text/css' or ends-with($uri/@path, '.scss')">media-code.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'application/javascript' or $uri/@mediatype = 'application/json'">media-code.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'application/zip'">media-compressed.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'application/x-tar'">media-compressed.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'application/x-gzip'">media-compressed.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'application/java-archive'">media-compressed.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'application/xml-dtd'">media-xml.png</xsl:when>
    <xsl:when test="ends-with($uri/@path, '.xsd')">media-xml.png</xsl:when>
    <!-- Main categories -->
    <xsl:when test="ends-with($uri/@mediatype, 'xml')">media-xml.png</xsl:when>
    <xsl:when test="starts-with($uri/@mediatype, 'text/')">media-text.png</xsl:when>
    <xsl:when test="starts-with($uri/@mediatype, 'image/')">media-image.png</xsl:when>
    <xsl:when test="starts-with($uri/@mediatype, 'audio/')">media-audio.png</xsl:when>
    <xsl:when test="starts-with($uri/@mediatype, 'video/')">media-video.png</xsl:when>
    <xsl:when test="$uri/@mediatype = 'folder'" >ico-folder.png</xsl:when>
    <xsl:otherwise>media-unknown.png</xsl:otherwise>
  </xsl:choose>
</xsl:function>

<!--
  Matches any part of the text which contains a string starting with
  'http://' or 'https://' and create a link.

  @param text the text node to work from.
  @return a sequence of the text nodes and links.
-->
<xsl:function name="f:linkify">
  <xsl:param name="text" />
  <xsl:for-each select="$text">
    <xsl:analyze-string select="." regex="http[s]?://[^\s]+">
      <xsl:matching-substring>
        <a href="{.}"><xsl:value-of select="."/></a>
      </xsl:matching-substring>
      <xsl:non-matching-substring>
        <xsl:value-of select="."/>
      </xsl:non-matching-substring>
    </xsl:analyze-string>
  </xsl:for-each>
</xsl:function>

<!--
  Replaces the newlines \n by <br/> tags

  @param text The text to process
-->
<xsl:function name="f:text-to-html">
  <xsl:param name="text"/>
  <xsl:variable name="lines" select="tokenize($text, '\n')"/>
  <xsl:for-each select="$lines">
    <xsl:copy-of select="."/><br/>
  </xsl:for-each>
</xsl:function>

<!--
  Returns the user interface notification values:
    Announcements only, Essential, Normal, Daily digest, Weekly digest

  @param notify The member's notification [none|immediate|essential|daily|weekly]
-->
<xsl:function name="f:notification">
  <xsl:param name="notify"/>
  <xsl:choose>
    <xsl:when test="$notify='none'">Announcements only</xsl:when>
    <xsl:when test="$notify='immediate'">Normal</xsl:when>
    <xsl:when test="$notify='essential'">Essential</xsl:when>
    <xsl:when test="$notify='daily'">Daily digest</xsl:when>
    <xsl:when test="$notify='weekly'">Weekly digest</xsl:when>
    <xsl:otherwise>Unknown</xsl:otherwise>
  </xsl:choose>
</xsl:function>

<!-- Returns HTML for labels with icons.

  @param images-url The URL to use for images
  @param labels The comma separated list of labels
-->
<xsl:function name="f:labels">
  <xsl:param name="images-url"/>
  <xsl:param name="labels"/>
  <table cellpadding="0" cellspacing="0" border="0"><tr><td class="labels_td">
    <xsl:for-each select="tokenize($labels, ',')">
      <img src="{$images-url}/ico-label.png" alt="Label" border="0" class="labels_icon"/>
      <xsl:text>&#xa0;</xsl:text>
      <span><xsl:value-of select="." /></span>
      <xsl:text>&#xa0;</xsl:text>
    </xsl:for-each>
  </td></tr></table>
</xsl:function>

<!--
  Returns title for a group/project or (name if no title)

  @param group The group or project element
-->
<xsl:function name="f:group-title">
  <xsl:param name="group"/>
  <xsl:choose>
    <xsl:when test="$group/@title">
      <xsl:value-of select="$group/@title" />
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$group/@name" />
    </xsl:otherwise>
  </xsl:choose>
</xsl:function>

</xsl:stylesheet>
