<!--
  Indexing transformations for SVG images.

  The source XML follows the structure below:

  ```
    <index-data mediatype="image/svg+xml">
      <document level="metadata"> ... </document>
      <workflow> ... </workflow>
      <content source="file"> ... </content>
    </index-data>
  ```

  @author Christophe Lauret
  @author Jean-Baptiste Reure

  @version 5.9900
-->
<xsl:transform  xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="2.0"
                xmlns:xhtml="http://www.w3.org/1999/xhtml"
                xmlns:psf="http://www.pageseeder.com/function"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:svg="http://www.w3.org/2000/svg"
                exclude-result-prefixes="#all">

<!-- We extend the default index module -->
<xsl:import href="index.xsl" />

<!-- indexXFields group property -->
<xsl:param name="ps-indexXFields" select="'false'" />

<!--
  Generate the fields specific to Webp from content extracted by Tika.

  ```
  <content source="file">
    <svg xmlns="http://www.w3.org/2000/svg">
        ...
    </svg>
  </content>
  ```
-->
<xsl:template match="index-data[@mediatype='image/svg+xml']/content" mode="ixml" priority="2">
  <xsl:for-each select="svg:svg">

    <!-- Image dimensions -->
    <xsl:variable name="viewport" select="psf:from-viewport(@width, @height)"/>
    <xsl:variable name="viewbox" select="psf:from-viewbox(@viewBox)"/>
    <xsl:choose>
      <xsl:when test="$viewport">
        <xsl:sequence select="psf:ps-dimension($viewport/@width, $viewport/@height)"/>
      </xsl:when>
      <xsl:when test="$viewbox">
        <xsl:sequence select="psf:ps-dimension($viewbox/@width, $viewbox/@height)"/>
      </xsl:when>
    </xsl:choose>

  </xsl:for-each>

</xsl:template>

<xsl:function name="psf:from-viewport" as="element()?">
  <xsl:param name="width"  as="xs:string?"/>
  <xsl:param name="height" as="xs:string?"/>
  <xsl:if test="$width and $height">
    <xsl:variable name="width-pixels" select="psf:to-pixels($width)"/>
    <xsl:variable name="height-pixels" select="psf:to-pixels($height)"/>
    <xsl:if test="$width-pixels and $height-pixels">
      <viewport width="{$width-pixels}" height="{$height-pixels}" />
    </xsl:if>
  </xsl:if>
</xsl:function>

<!--
  Width and height in SVG are length that may be expressed in different units.
-->
<xsl:function name="psf:to-pixels" as="xs:integer?">
  <xsl:param name="length" />
  <xsl:analyze-string select="$length" regex="^([0-9]+\.?[0-9]*)(em|ex|px|in|cm|mm|pt|pc|%)?$">
    <xsl:matching-substring>
      <xsl:variable name="value" select="number(regex-group(1))" />
      <xsl:variable name="unit" select="regex-group(2)" />
      <xsl:choose>
        <!-- length ::= number ("em" | "ex" | "px" | "in" | "cm" | "mm" | "pt" | "pc" | "%")? -->
        <xsl:when test="$unit = 'em'"><xsl:value-of select="round($value * 16)"/></xsl:when>
        <xsl:when test="$unit = 'ex'"><xsl:value-of select="round($value * 10)"/></xsl:when>
        <xsl:when test="$unit = 'px'"><xsl:value-of select="round($value)"/></xsl:when>
        <xsl:when test="$unit = 'in'"><xsl:value-of select="round($value * 96)"/></xsl:when>
        <xsl:when test="$unit = 'cm'"><xsl:value-of select="round($value * 96 div 2.54)"/></xsl:when>
        <xsl:when test="$unit = 'mm'"><xsl:value-of select="round($value * 96 div 254)"/></xsl:when>
        <xsl:when test="$unit = 'pt'"><xsl:value-of select="round($value * 4 div 3)"/></xsl:when>
        <xsl:when test="$unit = 'pc'"><xsl:value-of select="round($value * 16)"/></xsl:when>
        <xsl:when test="$unit = '%'"><xsl:value-of select="round($value * 8)"/></xsl:when>
        <xsl:when test="$unit = ''"><xsl:value-of select="round($value)"/></xsl:when>
      </xsl:choose>
    </xsl:matching-substring>
  </xsl:analyze-string>
</xsl:function>

<!--
  Width and height in SVG are length that may be expressed in different units.
-->
<xsl:function name="psf:from-viewbox" as="element()?">
  <xsl:param name="viewbox" />
  <xsl:analyze-string select="$viewbox" regex="^([0-9]+\.?[0-9]*)\s([0-9]+\.?[0-9]*)\s([0-9]+\.?[0-9]*)\s([0-9]+\.?[0-9]*)$">
    <xsl:matching-substring>
      <xsl:variable name="min-x" select="number(regex-group(1))" />
      <xsl:variable name="min-y" select="number(regex-group(2))" />
      <xsl:variable name="width" select="number(regex-group(3))" />
      <xsl:variable name="height" select="number(regex-group(4))" />
      <viewbox width="{round($width - $min-x)}" height="{round($height - $min-y)}" />
    </xsl:matching-substring>
  </xsl:analyze-string>
</xsl:function>

</xsl:transform>
