<!--
  Indexing transformations for PageSeeder URLs

  The source XML follows the structure below:

  ```
    <index-data mediatype="application/vnd.pageseeder.url+xml">
      <document level="metadata"> ... </document>
    </index-data>
  ```

  @author Christophe Lauret
  @version 5.9000
-->
<xsl:transform  xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="2.0"
                xmlns:psf="http://www.pageseeder.com/function"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                exclude-result-prefixes="#all">

<!-- Contains commons fields -->
<xsl:import href="index.xsl" />

<!--
  Fields generated for a `<document>` representing a URL
-->
<xsl:template match="index-data[@mediatype='application/vnd.pageseeder.url+xml']/document" mode="ixml" priority="2">

  <!-- Core fields -->
  <xsl:sequence select="psf:ps-id($psid-field-name, $psid-field-value)"/>
  <xsl:sequence select="psf:ps-title(documentinfo/uri/displaytitle)"/>
  <xsl:sequence select="psf:ps-type(if (@version != 'current') then 'urlversion' else 'url')"/>
  <xsl:sequence select="psf:ps-subtype(if (@version != 'current') then 'urlversion' else if (@folder = 'true') then 'folder' else 'url')"/>
  <xsl:sequence select="psf:ps-mediatype('application/vnd.pageseeder.url+xml')"/>
  <xsl:sequence select="psf:ps-modifieddate(xs:dateTime(if (documentinfo/uri/@modified) then documentinfo/uri/@modified else documentinfo/uri/@created))"/>

  <!-- URL fields (no filename or folder)-->
  <xsl:sequence select="psf:ps-size(documentinfo/uri/@size)"/>
  <xsl:sequence select="psf:ps-description(documentinfo/uri/description)"/>
  <xsl:sequence select="psf:ps-labels(documentinfo/uri/labels)"/>

  <!-- URL specific -->
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pshost" doc-values="sorted">
    <xsl:value-of select="documentinfo/uri/@host" />
  </field>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psurl" doc-values="sorted">
    <xsl:value-of select="psf:uri-to-url(documentinfo/uri)" />
  </field>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psurltype" doc-values="sorted">
    <xsl:value-of select="if (documentinfo/uri/@urltype) then documentinfo/uri/@urltype else 'default'" />
  </field>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psurlmediatype" doc-values="sorted">
    <xsl:value-of select="documentinfo/uri/@mediatype"/>
  </field>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pssource" doc-values="sorted">
    <xsl:value-of select="if (documentinfo/uri/@source) then documentinfo/uri/@source else 'external'" />
  </field>

  <!-- PageSeeder Metadata -->
  <xsl:if test="metadata/@checked">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" name="psmetadatachecked" date-format="yyyyMMddHHmmss"
           date-resolution="second" doc-values="sorted"><xsl:value-of select="psf:format-date-for-lucene(metadata/@checked)" /></field>
  </xsl:if>
  <xsl:for-each select="metadata//property">
    <xsl:sequence select="psf:ps-metadata(.)"/>
  </xsl:for-each>

  <!-- The content is the actual URL (with no word count) -->
  <field store="compress" index="docs-and-freqs-and-positions-and-offsets" tokenize="true" name="pscontent">
    <xsl:value-of select="psf:uri-to-decoded-url(documentinfo/uri)"/>
  </field>

  <!-- Versions info -->
  <xsl:variable name="latestversion" select="documentinfo/versions/version[@id = max(../version/xs:integer(@id))]" />
  <!-- Leave an empty field so documents with no version can be searched -->
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pslatestversion" doc-values="sorted"><xsl:value-of select="$latestversion/@name" /></field>
  <xsl:if test="$latestversion">
    <xsl:sequence select="psf:ps-latestversiondate($latestversion/@created)"/>
  </xsl:if>

  <!-- Created date, use version's if we're indexing one -->
  <xsl:choose>
    <xsl:when test="@version != 'current'">
      <xsl:variable name="version" select="documentinfo/versions/version[@name = current()/@version]"/>
      <xsl:sequence select="psf:ps-createddate($version/@created)"/>
      <!-- used by DeleteReleaseQuery.java -->
      <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psversionid" doc-values="sorted"><xsl:value-of select="$version/@id" /></field>
    </xsl:when>
    <xsl:when test="documentinfo/uri/@created">
      <xsl:sequence select="psf:ps-createddate(documentinfo/uri/@created)"/>
    </xsl:when>
  </xsl:choose>

  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psxrefcount" numeric-type="int" doc-values="sorted"><xsl:value-of select="count(//xref | //blockxref)"/></field>
  <!-- XRef labels -->
  <xsl:for-each select="distinct-values(tokenize(string-join(//xref/@labels|//blockxref/@labels, ','), ',')[. != ''])">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psxreflabel" doc-values="sorted-set"><xsl:value-of select="normalize-space(.)" /></field>
  </xsl:for-each>
  <xsl:for-each select="distinct-values(tokenize(string-join(//reversexref/@labels, ','), ',')[. != ''])">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psreversexreflabel" doc-values="sorted-set"><xsl:value-of select="normalize-space(.)" /></field>
  </xsl:for-each>
  <!-- XRef types -->
  <xsl:for-each select="distinct-values(//xref/@type|//blockxref/@type)">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psxreftype" doc-values="sorted-set"><xsl:value-of select="." /></field>
  </xsl:for-each>
  <xsl:for-each select="distinct-values(//reversexref/@forwardtype)">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psreversexreftype" doc-values="sorted-set"><xsl:value-of select="." /></field>
  </xsl:for-each>
  <!-- Link labels -->
  <xsl:for-each select="distinct-values(tokenize(string-join(//link/@labels, ','), ',')[. != ''])">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pslinklabel" doc-values="sorted-set"><xsl:value-of select="normalize-space(.)" /></field>
  </xsl:for-each>

  <!-- Workflow -->
  <xsl:apply-templates select="workflow" mode="ixml"/>

</xsl:template>

<xsl:template match="extra" mode="ixml" priority="2">
  <!-- reverse xrefs -->
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psreversexrefcount" numeric-type="int" doc-values="sorted"><xsl:value-of select="if (@reversexref-count) then @reversexref-count else 0"/></field>
  <!-- groups -->
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psgroupcount" numeric-type="int" doc-values="sorted"><xsl:value-of select="count(group)"/></field>
  <xsl:for-each select="group">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psgroupname"><xsl:value-of select="@name" /></field>
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psgroupid" numeric-type="long" doc-values="sorted"><xsl:value-of select="@id" /></field>
  </xsl:for-each>
</xsl:template>

</xsl:transform>
