<!--
  A set of XSLT functions commonly used for indexing.

  Use the functions in this class to ensure consistency in the representation
  of PageSeeder fields.

  @author Christophe Lauret
  @author Jean-Baptiste Reure

  @version 5.9000
-->
<xsl:transform  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:psf="http://www.pageseeder.com/function"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:xhtml="http://www.w3.org/1999/xhtml"
                exclude-result-prefixes="#all" version="2.0">

<xsl:param name="timezone.rawoffset" />
<!-- constants used in this class -->
<xsl:variable name="max-integer" select="2147483647" as="xs:integer"/>

<!-- CORE FIELDS ========================================================== -->

<!--
  Generates the `psid` field.

  @param name  The name of the PageSeeder ID field
  @param value The value for the field
-->
<xsl:function name="psf:ps-id">
  <xsl:param name="name"/>
  <xsl:param name="value"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="{$name}" doc-values="sorted"><xsl:value-of select="$value"/></field>
</xsl:function>

<!--
  Generates the `pstitle` field.

  Also creates a field for the title that isn't analysed or stored but used for sorting only.

  @param title The title of this indexed document
-->
<xsl:function name="psf:ps-title">
  <xsl:param name="title"/>
  <field store="true"   index="docs-and-freqs-and-positions-and-offsets" tokenize="true"  name="pstitle"><xsl:value-of select="$title"/></field>
  <field store="false"  index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pstitle-sort" doc-values="sorted"><xsl:value-of select="lower-case($title)"/></field>
</xsl:function>

<!--
  Generates the `psdescription` field.

  @param description The description of this indexed document
-->
<xsl:function name="psf:ps-description">
  <xsl:param name="description" as="xs:string?"/>
  <xsl:if test="$description and $description != ''">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" name="psdescription" tokenize="true"><xsl:value-of select="$description"/></field>
  </xsl:if>
</xsl:function>

<!--
  Generates the `pstype` field.

  @param type The main type of indexed document (document, comment, etc...)
-->
<xsl:function name="psf:ps-type">
  <xsl:param name="type"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pstype" doc-values="sorted"><xsl:value-of select="$type"/></field>
</xsl:function>

<!--
  Generates the `pssubtype` field.

  @param type  The main type of indexed document (document, comment, etc...)
  @param media The media type
-->
<xsl:function name="psf:ps-subtype">
  <xsl:param name="type"/>
  <xsl:param name="mediatype"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pssubtype" doc-values="sorted">
  <xsl:choose>
    <xsl:when test="$type = 'document'"><xsl:value-of select="psf:to-subtype($mediatype)"/></xsl:when>
    <xsl:otherwise><xsl:value-of select="$type"/></xsl:otherwise>
  </xsl:choose>
  </field>
</xsl:function>

<!--
  Generates the `pstype` field.

  @param subtype The subtype of indexed document (document, comment, image, audio, etc...)
-->
<xsl:function name="psf:ps-subtype">
  <xsl:param name="subtype"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pssubtype" doc-values="sorted"><xsl:value-of select="$subtype"/></field>
</xsl:function>

<!--
  Generates the `psmediatype` field.

  We need to filter the cases when the mediatype reported by the PSML does not
  quite match what we ought to use.

  @param mediatype The media type
-->
<xsl:function name="psf:ps-mediatype">
  <xsl:param name="mediatype"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psmediatype" doc-values="sorted">
    <xsl:value-of select="psf:clean-mediatype($mediatype)"/>
  </field>
</xsl:function>



<!--
  Generates the `psmodifieddate` field.

  @param modified The last modified date as a dateTime object.
-->
<xsl:function name="psf:ps-modifieddate">
  <xsl:param name="modified" as="xs:dateTime"/>
  <xsl:variable name="lastmodified" select="psf:format-date-for-lucene($modified)"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" name="psmodifieddate" date-format="yyyyMMddHHmmss"
         date-resolution="second" doc-values="sorted"><xsl:value-of select="$lastmodified" /></field>
</xsl:function>

<!--
  Generates the `pscreateddate` field.

  @param created The created date as a dateTime object.
-->
<xsl:function name="psf:ps-createddate">
  <xsl:param name="created" as="xs:dateTime"/>
  <xsl:variable name="date" select="psf:format-date-for-lucene($created)"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" name="pscreateddate" date-format="yyyyMMddHHmmss"
         date-resolution="second" doc-values="sorted"><xsl:value-of select="$date" /></field>
</xsl:function>

<!--
  Generates the `pslatestversiondate` field.

  @param created The latest version creation date as a dateTime object.
-->
<xsl:function name="psf:ps-latestversiondate">
  <xsl:param name="created" as="xs:dateTime"/>
  <xsl:variable name="date" select="psf:format-date-for-lucene($created)"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" name="pslatestversiondate" date-format="yyyyMMddHHmmss"
         date-resolution="second" doc-values="sorted"><xsl:value-of select="$date" /></field>
</xsl:function>

<!--
  Generates the `psmedia-modifieddate` field.

  @param modified The modified date as a string.
-->
<xsl:function name="psf:psmedia-modifieddate">
  <xsl:param name="modified" />
  <xsl:if test="$modified castable as xs:dateTime">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" name="psmedia-modifieddate" date-format="yyyyMMddHHmmss"
        date-resolution="second" doc-values="sorted">
      <xsl:value-of select="psf:format-date-for-lucene(xs:dateTime($modified))"/>
    </field>
  </xsl:if>
</xsl:function>

<!--
  Generates the `psmedia-createddate` field.

  @param created The created date as a string.
-->
<xsl:function name="psf:psmedia-createddate">
  <xsl:param name="created" />
  <xsl:if test="$created castable as xs:dateTime">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" name="psmedia-createddate" date-format="yyyyMMddHHmmss"
        date-resolution="second" doc-values="sorted">
      <xsl:value-of select="psf:format-date-for-lucene(xs:dateTime($created))"/>
    </field>
  </xsl:if>
</xsl:function>

<!--
  Generates the `psmedia-createddate` and `psmedia-modifieddate` fields from EXIF metadata.

  @param tika-head   The TIKA xhtml head element.
  @param uri-created The URI created date time (used for timezone).
-->
<xsl:function name="psf:exif-media-dates">
  <xsl:param name="tika-head" />
  <xsl:param name="uri-created" />
  <xsl:variable name="gps-date" select="$tika-head/xhtml:meta[@name='GPS Date Stamp']/@content" />
  <xsl:variable name="gps-time" select="$tika-head/xhtml:meta[@name='GPS Time-Stamp']/@content" />
  <xsl:variable name="date-time" select="$tika-head/xhtml:meta[@name='Date/Time']/@content" />
  <xsl:variable name="date-time-original" select="$tika-head/xhtml:meta[@name='Date/Time Original']/@content" />

  <xsl:variable name="created">
    <xsl:choose>
      <xsl:when test="$gps-date and $gps-time">
        <xsl:value-of select="concat(translate($gps-date,':','-'), 'T', substring($gps-time,1,8), 'Z')" />
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="concat(translate(substring-before($date-time-original,' '),':','-'), 'T', substring-after($date-time-original,' '), $timezone.rawoffset)" />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:sequence select="psf:psmedia-createddate($created)" />

  <!-- If there's a gps date and the exif datetime = exif datetime original, then use the gps date as it has the right timezone -->
  <xsl:variable name="modified">
    <xsl:choose>
      <xsl:when test="$gps-date and $gps-time and $date-time and $date-time = $date-time-original">
        <xsl:value-of select="concat(translate($gps-date,':','-'), 'T', substring($gps-time,1,8), 'Z')" />
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="concat(translate(substring-before($date-time,' '),':','-'), 'T', substring-after($date-time,' '), $timezone.rawoffset)" />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:sequence select="psf:psmedia-modifieddate($modified)" />

</xsl:function>

<!--
  Generates the `psprefixcontent` field.

  @param content The entire content of the item to index.
-->
<xsl:function name="psf:ps-prefixcontent">
  <xsl:param name="prefix"/>
  <xsl:param name="content"/>
  <field store="compress" index="docs-and-freqs-and-positions-and-offsets" tokenize="true" name="psprefixcontent">
    <xsl:if test="$prefix">
      <xsl:value-of select="$prefix" />
      <xsl:text>&#8203; </xsl:text><!-- zero width space + normal space for extract -->
    </xsl:if>
    <xsl:value-of select="$content"/>
  </field>
</xsl:function>

<!--
  Generates the `pscontent` field.

  @param content The entire content of the item to index.
-->
<xsl:function name="psf:ps-content">
  <xsl:param name="content"/>
  <field store="compress" index="docs-and-freqs-and-positions-and-offsets" tokenize="true" name="pscontent"><xsl:value-of select="$content"/></field>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pswordcount" numeric-type="int" doc-values="sorted">
    <xsl:value-of select="count(tokenize(normalize-space(string-join($content,'')), ' '))"/>
  </field>
</xsl:function>

<!--
  Generates the `pscontent` field.

  @param content The entire content of the item to index.
-->
<xsl:function name="psf:ps-content-xhtml">
  <xsl:param name="content"/>
  <xsl:variable name="words">
    <xsl:apply-templates select="$content" mode="ixml"/>
  </xsl:variable>
  <field store="compress" index="docs-and-freqs-and-positions-and-offsets" tokenize="true" name="pscontent">
    <xsl:copy-of select="$words" />
  </field>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pswordcount" numeric-type="int" doc-values="sorted">
    <xsl:value-of select="count(tokenize(normalize-space($words), ' '))"/>
  </field>
</xsl:function>

<!--
  Generates the `pslabel` fields from a string

  This is for flag labels (which do not have any content) for documents and fragments

  @param labels The labels collection.
-->
<xsl:function name="psf:ps-labels">
  <xsl:param name="labels" as="xs:string?"/>
  <xsl:if test="$labels and normalize-space($labels) != ''">
    <xsl:sequence select="psf:ps-label(distinct-values(tokenize($labels, ','))[normalize-space(.) != ''])"/>
  </xsl:if>
</xsl:function>

<!--
  Generates the `pslabel` fields.

  This is for flag labels (which do not have any content) for documents and fragments

  @param labels The labels collection.
-->
<xsl:function name="psf:ps-label">
  <xsl:param name="labels" as="xs:string*"/>
  <xsl:for-each select="$labels">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pslabel" doc-values="sorted-set"><xsl:value-of select="normalize-space(.)" /></field>
  </xsl:for-each>
</xsl:function>

<!-- OPTIONAL / COMMON FIELDS ============================================= -->

<!--
  Generates the `psowned` field.

  @param type whether the document is owned by the group (i.e. in the group's default folder)
-->
<xsl:function name="psf:ps-owned">
  <xsl:param name="owned" as="xs:boolean" />
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psowned" doc-values="sorted"><xsl:value-of select="$owned"/></field>
</xsl:function>

<!--
  Generates the `psdocid` field.

  @param docid The PageSeeder document ID.
-->
<xsl:function name="psf:ps-docid">
  <xsl:param name="docid" as="xs:string?"/>
  <xsl:if test="$docid and $docid != ''">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="true" name="psdocid"><xsl:value-of select="$docid" /></field>
  </xsl:if>
</xsl:function>

<!--
  Generates the `pssize` field.

  @param size The size of the indexed item in bytes.
-->
<xsl:function name="psf:ps-size">
  <xsl:param name="size" as="xs:integer?"/>
  <xsl:if test="$size and $size le $max-integer">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pssize" numeric-type="int" doc-values="sorted"><xsl:value-of select="$size"/></field>
  </xsl:if>
</xsl:function>

<!--
  Generates the `psfilename` field.

  @param filename The filename
-->
<xsl:function name="psf:ps-filename">
  <xsl:param name="filename"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="true" name="psfilename"><xsl:value-of select="$filename"/></field>
</xsl:function>

<!--
  Generates the `psfolder` and `psancestor` fields.

  @param folder The uri folder (usually starts with /ps/ )
-->
<xsl:function name="psf:ps-folder">
  <xsl:param name="folder"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psfolder" doc-values="sorted"><xsl:value-of select="$folder"/></field>
  <xsl:sequence select="psf:_ancestors($folder)"/>
</xsl:function>

<!--
  Generates the `width`, `height` and `pixelcount` fields - for images only.

  @param width  The graphic's width in pixels
  @param height The graphic's height in pixels
-->
<xsl:function name="psf:ps-dimension">
  <xsl:param name="width"  as="xs:integer"/>
  <xsl:param name="height" as="xs:integer"/>
  <field store="true"   index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pswidth"      numeric-type="int"  doc-values="sorted"><xsl:value-of select="$width" /></field>
  <field store="true"   index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psheight"     numeric-type="int"  doc-values="sorted"><xsl:value-of select="$height" /></field>
  <field store="false"  index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pspixelcount" numeric-type="long" doc-values="sorted"><xsl:value-of select="$width * $height" /></field>
</xsl:function>

<!--
  Generates the `taskdefinitionid` field - for tasks only.

  @param id  The id of the task definition (initiator) comment
-->
<xsl:function name="psf:ps-taskdefinitionid">
  <xsl:param name="id"  as="xs:integer"/>
  <field store="true"  index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pstaskdefinitionid" numeric-type="long" doc-values="sorted"><xsl:value-of select="$id" /></field>
</xsl:function>

<!--
  Generates the `discussionid` field - for comments only.

  @param id  The id of the discussion
-->
<xsl:function name="psf:ps-discussionid">
  <xsl:param name="id"  as="xs:integer"/>
  <field store="true"  index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psdiscussionid" numeric-type="long" doc-values="sorted"><xsl:value-of select="$id" /></field>
</xsl:function>

<!--
  Generates the `psauthor` fields - for tasks and comments only.

  @param name  The author's full name
-->
<xsl:function name="psf:ps-author">
  <xsl:param name="name" />
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psauthor" doc-values="sorted"><xsl:value-of select="$name"/></field>
</xsl:function>

<!--
  Generates the `psauthorid` field - for tasks and comments only.

  @param id  The id of the author member
-->
<xsl:function name="psf:ps-authorid">
  <xsl:param name="id"  as="xs:integer"/>
  <field store="true"  index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psauthorid" numeric-type="long" doc-values="sorted"><xsl:value-of select="$id" /></field>
</xsl:function>

<!--
  Generates the `pscontextexternal` field - for tasks and comments only.

  @param external  whether the context is an external URI (URL)
-->
<xsl:function name="psf:ps-contextexternal">
  <xsl:param name="external" as="xs:boolean" />
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pscontextexternal" doc-values="sorted"><xsl:value-of select="$external"/></field>
</xsl:function>

<!--
  Generates the `pscontexturiid` field - for tasks and comments only.

  @param id  The id of the context URI (0 for general context)
-->
<xsl:function name="psf:ps-contexturiid">
  <xsl:param name="id"  as="xs:integer"/>
  <field store="true"  index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pscontexturiid" numeric-type="long" doc-values="sorted"><xsl:value-of select="$id" /></field>
</xsl:function>

<!--
  Generates the `pscontexturiid` field - for tasks and comments only (0 if no context URI).
  If there is a context URI then also:
  Generates the `pscontexturititle` field - for tasks and comments only.
  Generates the `pscontexturimediatype` field - for tasks and comments only.
  Generates the `pscontexturidocumenttype` field - for tasks and comments on PSML documents only.
  Generates the `pscontexturiurltype` field - for tasks and comments on URLs only.

  @param uri  The context URI (optional)
-->
<xsl:function name="psf:ps-contexturi">
  <xsl:param name="uri"  as="element(uri)?"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pscontexturiid" numeric-type="long" doc-values="sorted">
    <xsl:value-of select="if ($uri) then $uri/@id else 0" />
  </field>
  <xsl:if test="$uri">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="true"  name="pscontexturititle"><xsl:value-of select="$uri/displaytitle"/></field>
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pscontexturimediatype" doc-values="sorted">
      <xsl:value-of select="psf:clean-mediatype($uri/@mediatype)"/>
    </field>
    <xsl:choose>
      <xsl:when test="$uri/@mediatype='application/vnd.pageseeder.psml+xml'">
        <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pscontexturidocumenttype" doc-values="sorted">
          <xsl:value-of select="if ($uri/@documenttype) then $uri/@documenttype else 'default'"/>
        </field>
      </xsl:when>
      <xsl:when test="$uri/@external='true'">
        <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pscontexturiurltype" doc-values="sorted">
          <xsl:value-of select="if ($uri/@urltype) then $uri/@urltype else 'default'" />
        </field>
      </xsl:when>
    </xsl:choose>
  </xsl:if>
</xsl:function>

<!--
  Generates the `pscontextfragment` field - for tasks and comments only.

  @param name  The context fragment
-->
<xsl:function name="psf:ps-contextfragment">
  <xsl:param name="name" />
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pscontextfragment" doc-values="sorted"><xsl:value-of select="$name"/></field>
</xsl:function>

<!--
  Generates the `psstatus` field.

  @param name  The document or task status
-->
<xsl:function name="psf:ps-status">
  <xsl:param name="status" />
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psstatus" doc-values="sorted"><xsl:value-of select="if ($status) then $status else 'None'"/></field>
</xsl:function>

<!--
  Generates the `psduedate` field.

  @param name  The document or task due date
-->
<xsl:function name="psf:ps-duedate">
  <xsl:param name="duedate" />
  <xsl:if test="$duedate">
    <xsl:variable name="date" select="psf:format-date-for-lucene($duedate)"/>
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" doc-values="sorted"
           name="psduedate" date-format="yyyyMMddHHmmss" date-resolution="second"><xsl:value-of select="$date" /></field>
  </xsl:if>
</xsl:function>

<!--
  Generates the `pspriority` field.

  @param name  The task priority
-->
<xsl:function name="psf:ps-priority">
  <xsl:param name="priority" />
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false"
         name="pspriority" doc-values="sorted"><xsl:value-of select="if (string($priority) != '') then $priority else 'None'"/></field>
</xsl:function>

<!--
  Generates the `psassignedto` field.

  @param name The name of the person the task/document is assigned to
-->
<xsl:function name="psf:ps-assignedto">
  <xsl:param name="name" />
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false"
         name="psassignedto" doc-values="sorted"><xsl:value-of select="if (string($name) != '') then $name else 'Nobody'"/></field>
</xsl:function>

<!--
  Generates the `psassignedtoid` field.

  @param id  The id of the assigned member
-->
<xsl:function name="psf:ps-assignedtoid">
  <xsl:param name="id"  as="xs:integer"/>
  <field store="true"  index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psassignedtoid" numeric-type="long" doc-values="sorted"><xsl:value-of select="$id" /></field>
</xsl:function>

<!--
  Generates the `pscomment-status` field.

  @param name  The comment status
-->
<xsl:function name="psf:ps-comment-status">
  <xsl:param name="status" />
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pscomment-status" doc-values="sorted"><xsl:value-of select="if ($status) then $status else 'None'"/></field>
</xsl:function>

<!--
  Generates the `pscomment-duedate` field.

  @param name  The comment due date
-->
<xsl:function name="psf:ps-comment-duedate">
  <xsl:param name="duedate" />
  <xsl:if test="$duedate">
    <xsl:variable name="date" select="psf:format-date-for-lucene($duedate)"/>
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" doc-values="sorted"
           name="pscomment-duedate" date-format="yyyyMMddHHmmss" date-resolution="second"><xsl:value-of select="$date" /></field>
  </xsl:if>
</xsl:function>

<!--
  Generates the `pscomment-priority` field.

  @param name  The comment priority
-->
<xsl:function name="psf:ps-comment-priority">
  <xsl:param name="priority" />
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false"
         name="pscomment-priority" doc-values="sorted"><xsl:value-of select="if (string($priority) != '') then $priority else 'None'"/></field>
</xsl:function>

<!--
  Generates the `psassignedto` field.

  @param name The name of the person the comment is assigned to
-->
<xsl:function name="psf:ps-comment-assignedto">
  <xsl:param name="name" />
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false"
         name="pscomment-assignedto" doc-values="sorted"><xsl:value-of select="if (string($name) != '') then $name else 'Nobody'"/></field>
</xsl:function>

<!--
  Generates the `pscomment-assignedtoid` field.

  @param id  The id of the member assigned to the comment
-->
<xsl:function name="psf:ps-comment-assignedtoid">
  <xsl:param name="id"  as="xs:integer"/>
  <field store="true"  index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pscomment-assignedtoid" numeric-type="long" doc-values="sorted"><xsl:value-of select="$id" /></field>
</xsl:function>

<!--
  Generates the `psstatuschangeddate` field.

  @param name  The document status changed date
-->
<xsl:function name="psf:ps-statuschangeddate">
  <xsl:param name="statuschangeddate" />
  <xsl:if test="$statuschangeddate">
    <xsl:variable name="date" select="psf:format-date-for-lucene($statuschangeddate)"/>
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" doc-values="sorted"
           name="psstatuschangeddate" date-format="yyyyMMddHHmmss" date-resolution="second"><xsl:value-of select="$date" /></field>
  </xsl:if>
</xsl:function>

<!--
  Generates a `psproperty-*` field

  Note: Any character in the name that isn't `[A-Za-z0-9_-]` will be replaced
  by an underscore '_'

  @param property The property element (required)
-->
<xsl:function name="psf:ps-property">
<xsl:param name="property" as="element(property)"/>
<xsl:choose>
  <xsl:when test="$property[link or xref or value]">
    <xsl:for-each select="$property/*">
      <xsl:sequence select="psf:_ps-property($property/@name, text(), $property/@datatype, 'property')"/>
    </xsl:for-each>
  </xsl:when>
  <xsl:when test="$property/*">
    <xsl:sequence select="psf:_ps-property($property/@name, string-join($property/*//text(),' '), $property/@datatype, 'property')"/>
  </xsl:when>
  <xsl:otherwise>
    <xsl:sequence select="psf:_ps-property($property/@name, $property/@value, $property/@datatype, 'property')"/>
  </xsl:otherwise>
</xsl:choose>
</xsl:function>

<!--
  Generates a `psmetadata-*` field

  Note: Any character in the name that isn't `[A-Za-z0-9_-]` will be replaced
  by an underscore '_'

  @param property The property element (required)
-->
<xsl:function name="psf:ps-metadata">
<xsl:param name="property" as="element(property)"/>
<xsl:choose>
  <xsl:when test="$property[xref or value]">
    <xsl:for-each select="$property/*">
      <xsl:sequence select="psf:_ps-property($property/@name, text(), $property/@datatype, 'metadata')"/>
    </xsl:for-each>
  </xsl:when>
  <xsl:when test="$property/*">
    <xsl:sequence select="psf:_ps-property($property/@name, string-join($property/*//text(),' '), $property/@datatype, 'metadata')"/>
  </xsl:when>
  <xsl:otherwise>
    <xsl:sequence select="psf:_ps-property($property/@name, $property/@value, $property/@datatype, 'metadata')"/>
  </xsl:otherwise>
</xsl:choose>
</xsl:function>

<!--
  Generates an extra field `x-*`.

  @param name  The name of the extra field
  @param value The value of the extra field
-->
<xsl:function name="psf:x-field">
  <xsl:param name="name" />
  <xsl:param name="value" />
  <xsl:variable name="x-name" select="lower-case(replace($name, '[^a-zA-Z0-9]', '-'))"/>
  <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="x-{$x-name}" doc-values="sorted-set"><xsl:value-of select="$value" /></field>
</xsl:function>


<!-- Utility functions ==================================================== -->

<!--
  Convert a uri element to a URL
 -->
<xsl:function name="psf:uri-to-url">
  <xsl:param name="uri" />
  <xsl:value-of select="$uri/@scheme" />
  <xsl:text>://</xsl:text>
  <xsl:value-of select="$uri/@host" />
  <xsl:if test="not($uri/@scheme = 'http' and $uri/@port = '80') and not($uri/@scheme = 'https' and $uri/@port = '443')">
    <xsl:text>:</xsl:text>
    <xsl:value-of select="$uri/@port" />
  </xsl:if>
  <xsl:value-of select="$uri/@path" />
</xsl:function>

<!--
  Convert a uri element to a URL
 -->
<xsl:function name="psf:uri-to-decoded-url">
  <xsl:param name="uri" />
  <xsl:value-of select="$uri/@scheme" />
  <xsl:text>://</xsl:text>
  <xsl:value-of select="$uri/@host" />
  <xsl:if test="not($uri/@scheme = 'http' and $uri/@port = '80') and not($uri/@scheme = 'https' and $uri/@port = '443')">
    <xsl:text>:</xsl:text>
    <xsl:value-of select="$uri/@port" />
  </xsl:if>
  <xsl:value-of select="$uri/@decodedpath" />
</xsl:function>

<!--
  Determines the subtype based on the media type.

  @param mediatype The mediatype of the item being indexed.

  @return the corresponding subtype.
-->
<xsl:function name="psf:to-subtype" as="xs:string">
  <xsl:param name="mediatype" as="xs:string?"/>
  <xsl:choose>
    <xsl:when test="starts-with($mediatype, 'image/')">image</xsl:when>
    <xsl:when test="starts-with($mediatype, 'video/')">video</xsl:when>
    <xsl:when test="starts-with($mediatype, 'audio/')">audio</xsl:when>
    <xsl:when test="starts-with($mediatype, 'application/vnd.openxmlformats')">office</xsl:when>
    <xsl:when test="ends-with($mediatype,   'word')">office</xsl:when>
    <xsl:when test="ends-with($mediatype,   'excel')">office</xsl:when>
    <xsl:when test="ends-with($mediatype,   'x-gzip')">archive</xsl:when>
    <xsl:when test="ends-with($mediatype,   'x-tar')">archive</xsl:when>
    <xsl:when test="ends-with($mediatype,   'zip')">archive</xsl:when>
    <xsl:when test="ends-with($mediatype,   'java-archive')">archive</xsl:when>
    <xsl:when test="$mediatype = 'application/mathml+xml'">math</xsl:when>
    <xsl:when test="$mediatype = 'application/x-tex'">math</xsl:when>
    <xsl:when test="ends-with($mediatype,   'folder')">folder</xsl:when><!-- Indexer and PSML use different mediatypes for folders -->
    <xsl:when test="$mediatype = 'application/msword'">office</xsl:when>
    <xsl:when test="$mediatype = 'application/pdf'">pdf</xsl:when>
    <xsl:when test="$mediatype = 'application/vnd.pageseeder.comment+xml'">comment</xsl:when>
    <xsl:when test="$mediatype = 'application/vnd.pageseeder.task+xml'">task</xsl:when>
    <xsl:when test="$mediatype = 'application/vnd.pageseeder.url+xml'">url</xsl:when><!-- XXX: Maybe folder??? -->
    <xsl:otherwise>other</xsl:otherwise>
  </xsl:choose>
</xsl:function>

<!--
  Remove the markdown syntax from the specified text

  @param md The markdown text

  @return the text with the markdown syntax removed.
-->
<xsl:function name="psf:clean-markdown">
  <xsl:param name="md"/>
  <xsl:value-of select="replace(replace($md,
                        '(`([^`]+)`)|(\*\*([^\*]+)\*\*)|(__([^_]+)__)|(\*([^\*]+)\*)', '$2$4$6$8', 'm'),
                        '(#+\s)|(\s#+)|(```)|(---+)|(===+)', '', 'm')"/>
</xsl:function>

<!-- Other utility functions - keep 'em private =========================== -->

<!--
  Clean media type by replacing:
  - "text/xml" with "application/xml"
  - "folder" with "application/vnd.pageseeder.folder"

  @param md The media type

  @return the clean media type.
-->
<xsl:function name="psf:clean-mediatype">
  <xsl:param name="mediatype"/>
  <xsl:choose>
    <xsl:when test="$mediatype = 'text/xml'">application/xml</xsl:when>
    <xsl:when test="$mediatype = 'folder'">application/vnd.pageseeder.folder</xsl:when>
    <xsl:otherwise><xsl:value-of select="$mediatype"/></xsl:otherwise>
  </xsl:choose>
</xsl:function>

<!--
  Formats the date component of the specified element for lucene.
  Assumes date is in the format yyyy/MM/dd HH:mm:ss+XX:yy
  where XX:yy is the timezone offset in hours:minutes

  @param date The date to format

  @return the formatted date as yyyyMMddHHmmss
-->
<xsl:function name="psf:format-date-for-lucene">
  <xsl:param name="theDateTime" as="xs:dateTime" />
  <!-- Dates in lucene are saved as GMT so we need to bring the date to the GMT timezone -->
  <xsl:variable name="dateTime" select="adjust-dateTime-to-timezone($theDateTime, xs:dayTimeDuration('PT0H'))" />
  <xsl:variable name="year"     select="format-number(year-from-dateTime($dateTime),    '0000')" />
  <xsl:variable name="month"    select="format-number(month-from-dateTime($dateTime),   '00')" />
  <xsl:variable name="day"      select="format-number(day-from-dateTime($dateTime),     '00')" />
  <xsl:variable name="hours"    select="format-number(hours-from-dateTime($dateTime),   '00')" />
  <xsl:variable name="minutes"  select="format-number(minutes-from-dateTime($dateTime), '00')" />
  <xsl:variable name="seconds"  select="format-number(seconds-from-dateTime($dateTime), '00')" />
  <xsl:value-of select="concat($year, $month, $day, $hours, $minutes, $seconds)"/>
</xsl:function>

<!--
  Computes all the ancestors from the specified path and generate the
  appropriate `psancestor` fields (recursive)

  @param path The path to decompose into ancestors
-->
<xsl:function name="psf:_ancestors">
  <xsl:param name="path"/>
  <xsl:if test="string-length($path) gt 0">
    <field store="false" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="psancestor"><xsl:value-of select="$path"/></field>
    <xsl:variable name="steps" select="tokenize($path, '/')"/>
    <xsl:if test="count($steps) gt 1">
      <xsl:sequence select="psf:_ancestors(string-join(subsequence($steps,1,count($steps) - 1), '/'))"/>
    </xsl:if>
  </xsl:if>
</xsl:function>

<!--
  Generates a `psproperty-*` field

  Note: Any character in the name that isn't `[A-Za-z0-9_-]` will be replaced
  by an underscore '_'

  @param name     The name of the property (required)
  @param value    The value of the property (required)
  @param datatype The data type (i.e. 'date', 'datetime')
  @param prefix   The prefix 'property' or 'metadata'
-->
<xsl:function name="psf:_ps-property">
<xsl:param name="name"     />
<xsl:param name="value"    />
<xsl:param name="datatype" />
<xsl:param name="prefix"   />
<xsl:variable name="fieldname" select="concat('ps', $prefix, '-', replace($name, '[^-\w]', '_'))"/>
<xsl:choose>
  <!-- We try to parse date properties as dates -->
  <xsl:when test="$datatype='date'">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false"
           name="{$fieldname}" date-format="yyyy-MM-dd" date-resolution="day" doc-values="sorted-set">
      <xsl:value-of select="normalize-space($value)" />
    </field>
  </xsl:when>
  <!-- We try to parse datetime properties as datetimes -->
  <xsl:when test="$datatype='datetime'">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false"
           name="{$fieldname}" date-format="yyyy-MM-dd'T'HH:mm:ssX" date-resolution="second" doc-values="sorted-set">
      <xsl:value-of select="normalize-space($value)" />
    </field>
  </xsl:when>
  <!-- Markdown: we store markdown but analyze after removing the markdown syntax -->
  <xsl:when test="$datatype='markdown'">
    <field store="true"   index="none" name="{$fieldname}"><xsl:value-of select="$value" /></field>
    <field store="false"  index="docs-and-freqs-and-positions-and-offsets" tokenize="true" name="{$fieldname}">
      <xsl:value-of select="normalize-space(psf:clean-markdown($value))"/>
    </field>
  </xsl:when>
  <!-- Analyze fields ending in '-text' -->
  <xsl:when test="ends-with($fieldname, '-text')">
    <field store="true"  index="docs-and-freqs-and-positions-and-offsets" tokenize="true" name="{$fieldname}">
      <xsl:value-of select="if ($datatype='markup') then normalize-space($value) else $value"/>
    </field>
  </xsl:when>
  <xsl:otherwise>
     <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="{$fieldname}"
            doc-values="sorted-set"><xsl:value-of select="normalize-space($value)" /></field>
  </xsl:otherwise>
</xsl:choose>
</xsl:function>

<!-- Handling XHTML ======================================================= -->

<xsl:template match="xhtml:body" mode="ixml">
  <xsl:apply-templates select="xhtml:*" mode="ixml"/>
</xsl:template>

<!-- Headings and paragraphs -->
<xsl:template match="xhtml:p|xhtml:h1|xhtml:h2|xhtml:h3|xhtml:h4|xhtml:h5|xhtml:h6|xhtml:table" mode="ixml">
  <xsl:apply-templates mode="ixml"/><xsl:text>&#xa;</xsl:text>
</xsl:template>

<!-- Table cells -->
<xsl:template match="xhtml:th|xhtml:td" mode="ixml">
  <xsl:apply-templates mode="ixml"/><xsl:text> | </xsl:text>
</xsl:template>

<!-- List items -->
<xsl:template match="xhtml:li" mode="ixml">
  <xsl:apply-templates mode="ixml"/><xsl:text>&#xa;</xsl:text>
</xsl:template>

<!--
  Generate the fields from a publication.

  ```
  <publication id="[publication id]"
               hostid="[host id]"
               rooturiid="[uri id]"
               title="[uri title]"
               defaultgroupid="[groupid]"
               type="[type]" />

  ```
-->
<xsl:template match="publication" mode="ixml">
    <field store="true" index="docs-and-freqs-and-positions-and-offsets" tokenize="false" name="pspublicationid" doc-values="sorted-set"><xsl:value-of select="@id" /></field>
</xsl:template>

</xsl:transform>
