/**
 * $Id: ListFriendlyValueURIHandlerTest.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.tests;

import  java.util.List;

import  junit.framework.TestSuite;

import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.ValueURIHandler;
import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.solo.StringList;
import  com.idaremedia.antx.valueuri.info.ListFriendlyValueURIHandler;

/**
 * Testsuite for {@linkplain ListFriendlyValueURIHandler}.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,test
 **/

public final class ListFriendlyValueURIHandlerTest extends ValueURIHandlerTestSkeleton
{
    /**
     * Initializes a new test case for named method.
     * @param methodname test case method's name (non-null)
     **/
    public ListFriendlyValueURIHandlerTest(String methodname)
    {
        super("ListFriendlyValueURIHandler::",methodname);
    }


    /**
     * Returns full test suite for ListFriendlyValueURIHandler.
     **/
    public static TestSuite suite()
    {
        return new TestSuite(ListFriendlyValueURIHandlerTest.class);
    }

//  ---------------------------------------------------------------------------------------------------------
//  --------------------------------------- [ Misc Factory Methods ] ----------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    protected String xmlTestFile()
    {
        return "listfriendly.xml";
    }

    protected ValueURIHandler newOUT()
    {
        return new ListFriendlyValueURIHandler();
    }
    
    protected void setUp() throws Exception
    {
        super.setUp();
        StringList sl = new StringList();
        sl.setId("abBCd");
        sl.addConfiguredString(new InnerString("a"));
        sl.addConfiguredString(new InnerString("bB"));
        sl.addConfiguredString(new InnerString("C"));
        sl.addConfiguredString(new InnerString("d"));
        getProject().addReference(sl.getId(),sl);
        getProject().addReference("1,2,3","ERROR");
    }

//  ---------------------------------------------------------------------------------------------------------
//  ------------------------------------------- [ The Test Cases ] ------------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    /**
     * Verify a blank uri fragment is evaluated as expected.
     * @since JWare/AntX 0.5
     **/
    public void testBaseline_AntX05()
    {
        checkBaseline();
        
        assertNotNil(getProject().getReference("abBCd"),"abBCd reference");
        ValueURIHandler out = newOUT();
        String result = out.valueFrom("a,b,c","$list:a,b,c",m_rqlink);
        assertEqual(result,"a,b,c","$list:a,b,c");
        result = out.valueFrom("abBCd","$list:abBCd",m_rqlink);
        assertEqual(result,"a,bB,C,d","$list:abBCd");
    }


    /**
     * Verify a blank or all whitespace uri fragment is evaluated as expected.
     * @since JWare/AntX 0.5
     **/
    public void testBlank_AntX05()
    {
        ValueURIHandler out = newOUT();
        String result;
        result = out.valueFrom("","$list:",m_rqlink);
        assertEqual(result,"","$list:");
        result = out.valueFrom("?size","$list:?size",m_rqlink);
        assertEqual(result,"0","$list:?size");
        result = out.valueFrom("   ?size","$list:   ?size",m_rqlink);
        assertEqual(result,"0","$list:   ?size");
        result = out.valueFrom("[]","$list:[]",m_rqlink);
        assertEqual(result,"","$list:[]");
        assertEqual("",out.valueFrom("?values","$list:?values",m_rqlink),"$list:?values");
        assertEqual("[]",out.valueFrom("?dump","$list:?dump",m_rqlink),"$list:?dump");
        result = out.valueFrom("?1","$list:?1",m_rqlink);
        assertNil(result,"$list:?1");
        result = out.valueFrom("?ickyblorf","$list:?ickyblorf",m_rqlink);
        assertNil(result,"$list:?ickyblorf");
    }
    
    
    /**
     * Verify defining a custom delimiter for existing lists.
     * @since JWare/AntX 0.5
     **/
    public void testCustomDelim_AntX05()
    {
        ValueURIHandler out = newOUT();
        String result;
        result = out.valueFrom("abBCd?values,,;","$list:abBCd?values,,;",m_rqlink);
        assertEqual(result,"a;bB;C;d","$list:abBCd?values,,;");
        result = out.valueFrom("x,y,z?values,,-","$list:x,y,z?values,,-",m_rqlink);
        assertEqual(result,"x-y-z","$list:x,y,z?values,,-");
        result = out.valueFrom("x,y,z?values,, ","$list:x,y,z?values,, ",m_rqlink);
        assertEqual(result,"x y z","$list:x,y,z?values,, ");
        result = out.valueFrom("[1,2,3]?,,|","$list:[1,2,3]?,,|",m_rqlink);
        assertEqual(result,"1|2|3","$list:[1,2,3]?,,|");
    }


    /**
     * Verify we can force handler to use query as an inlined string
     * even if a reference of same name exists.
     * @since JWare/AntX 0.5
     **/
    public void testSquareBracketsOverrideRefId_AntX05()
    {
        ValueURIHandler out = newOUT();
        String result;
        result = out.valueFrom("1,2,3","$list:1,2,3",m_rqlink);
        assertNil(result,"$list:1,2,3");
        result = out.valueFrom("[1,2,3]","$list:[1,2,3]",m_rqlink);
        assertEqual(result,"1,2,3","$list:[1,2,3]");
    }



    /**
     * Ensure the handler really is "List" friendly towards List classes
     * that are not AntX-based.
     * @since JWare/AntX 0.5
     **/
    public void testWorksForPlainListTypesToo_AntX05()
    {
        List l = AntXFixture.newList();
        l.add("larry");
        l.add("curly");
        l.add("moe");
        getProject().addReference("3stooges",l);

        ValueURIHandler out = newOUT();
        String result;
        result = out.valueFrom("3stooges","$list:3stooges",m_rqlink);
        assertEqual(result,"larry,curly,moe","$list:3stooges");
        result = out.valueFrom("3stooges?values,,!","$list:3stooges?values,,!",m_rqlink);
        assertEqual(result,"larry!curly!moe","$list:3stooges?values,,!");
        result = out.valueFrom("3stooges?2","$list:3stooges?2",m_rqlink);
        assertEqual(result,"moe","$list:3stooges?2");
        result = out.valueFrom("3stooges?3","$list:3stooges?3",m_rqlink);
        assertNil(result,"$list:3stooges?3");
    }



    /**
     * Verify some things from within an Ant-only environment.
     * @since JWare/AntX 0.5
     **/
    public void testListFriendlyHandlerScript_AntX05()
    {
        runTarget("testListFriendlyHandlerScript_AntX05");
    }
}

/* end-of-ListFriendlyValueURIHandlerTest.java */