/**
 * $Id: BasenameValueURIHandlerTest.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.tests;

import  java.io.File;

import  org.apache.tools.ant.taskdefs.condition.Os;

import  junit.framework.TestSuite;

import  com.idaremedia.antx.ValueURIHandler;
import  com.idaremedia.antx.valueuri.BasenameValueURIHandler;

/**
 * Testsuite for {@linkplain XDefCheckValueURIHandler}.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,test
 **/

public final class BasenameValueURIHandlerTest extends ValueURIHandlerTestSkeleton
{
    /**
     * Initializes a new test case for named method.
     * @param methodname test case method's name (non-null)
     **/
    public BasenameValueURIHandlerTest(String methodname)
    {
        super("BasenameValueURIHandler::",methodname);
        isWindoze = Os.isFamily("windows");
    }


    /**
     * Returns full test suite for BasenameValueURIHandler.
     **/
    public static TestSuite suite()
    {
        return new TestSuite(BasenameValueURIHandlerTest.class);
    }

//  ---------------------------------------------------------------------------------------------------------
//  --------------------------------------- [ Misc Factory Methods ] ----------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    protected ValueURIHandler newOUT()
    {
        return new BasenameValueURIHandler();
    }

//  ---------------------------------------------------------------------------------------------------------
//  ------------------------------------------- [ The Test Cases ] ------------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    /**
     * Verify an empty path fragment is evaluated as project's basedir.
     * @since JWare/AntX 0.5
     **/
    public void testBaseline_AntX05()
    {
        checkBaseline();

        File basedir = getProject().getBaseDir();
        assertNotNil(basedir,"project basedir");
        
        ValueURIHandler out = newOUT();
        String blankresult = out.valueFrom("","$basename:",m_rqlink);
        String basepath = basedir.getPath();
        String baseresult = out.valueFrom(basepath,"$basename:"+basepath,m_rqlink);
        assertEqual(blankresult,baseresult,"$basename:");
        blankresult = out.valueFrom("?.foo","$basename:?.foo",m_rqlink);
        assertEqual(blankresult,baseresult,"$basename:?.foo");
    }


    /**
     * Verify the basename of "/" is an empty string.
     * @since JWare/AntX 0.5
     **/
    public void testRootPathIsBlankBasename_AntX05()
    {
        ValueURIHandler out = newOUT();
        String slash = isWindoze ? "\\" : "/";
        String result = out.valueFrom(slash,"$basename:"+slash,m_rqlink);
        assertEqual(result,"","$basename:"+slash);
    }


    /**
     * Verify basename of /-terminated path works as expected.
     * @since JWare/AntX 0.5
     **/
    public void testSlashTerminatedPath_AntX05()
    {
        ValueURIHandler out = newOUT();
        String result= out.valueFrom("/fu/bar/","$basename:/fu/bar/",m_rqlink);
        assertEqual(result,"bar","$basename:/fu/bar/");
        if (isWindoze) {
            result = out.valueFrom("\\fu\\bar\\","$basename:\\fu\\bar\\",m_rqlink);
            assertEqual(result,"bar","$basename:\\fu\\bar\\");
        }
    }


    /**
     * Verify can strip the extension of a file name.
     * @since JWare/AntX 0.5
     */
    public void testCanStripSuffix_AntX05()
    {
        ValueURIHandler out = newOUT();
        String result = out.valueFrom("/xml/file.xml?xml","$basename:/xml/file.xml?xml",m_rqlink);
        assertEqual(result,"file","$basename:/xml/file.xml?xml");
        result = out.valueFrom("/xml/file.xml?.xml","$basename:/xml/file.xml?.xml",m_rqlink);
        assertEqual(result,"file","$basename:/xml/file.xml?.xml");
        result = out.valueFrom("/xml/file.xml?.XML","$basename:/xml/file.xml?.XML",m_rqlink);
        assertEqual(result,"file.xml","$basename:/xml/file.xml?.XML");
        result = out.valueFrom("/a/dir/fubar-libs?-libs","$basename:/a/dir/fubar-libs?-libs",m_rqlink);
        assertEqual(result,"fubar","$basename:/a/dir/fubar-libs?-libs");
    }


    /**
     * Verify that handler dereferences properties as expected.
     * @since JWare/AntX 0.5
     **/
    public void testEmbeddedPropertyRefsHandled_AntX05()
    {
        ValueURIHandler out = newOUT();
        String result = out.valueFrom("@(basedir)","$basename:@(basedir)",m_rqlink);
        String basepath = getProject().getBaseDir().getPath();
        String baseresult = out.valueFrom(basepath,"$basename:"+basepath,m_rqlink);
        assertEqual(result,baseresult,"$basename:@(basedir)");
        
        getProject().setProperty("ext","-lib");
        result = out.valueFrom("/fu/bar-lib/?@(ext)","$basename:/fu/bar-lib/?@(ext)",m_rqlink);
        assertEqual(result,"bar","$basename:/fu/bar-lib/?@(ext)");
    }


    private final boolean isWindoze;
}

/* end-of-BasenameValueURIHandlerTest.java */