/**
 * $Id: TruncateStringURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.ValueURIHandlerSkeleton;

/**
 * Value URI handler that will truncate a long string to some prescribed max length.
 * The general form of the URI: 
 * <span class="src"><nobr>$truncate:string[?[maxlength][,,[left|right]]]</nobr>
 * </span>.
 * <p>
 * <b>Example Usage:</b><pre>
 *    &lt;foreach i="file" files="thefiles"&gt;
 *       &lt;property name="next" value="${$basename:@(file)|<b>$truncate:?21</b>|$lowercase:}.dat"/&gt;
 *       &lt;emit msgid="progress.next" msgarg1="${next}"/&gt;
 *       ...
 * 
 *   -- To Install --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="truncate"
 *             value="com.idaremedia.antx.valueuri.TruncateStringURIHandler"/&gt;
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 * @see       ShortenStringURIHandler
 **/

public final class TruncateStringURIHandler extends ValueURIHandlerSkeleton
{

    /**
     * Default max threshold for truncated strings ({@value}).
     */
    public static final int MAXLEN= ShortenStringURIHandler.MAXLEN;


    /**
     * Default minimum length for truncated strings (<i>1</i>).
     */
    public static final int MINLEN= 1;


    /**
     * Tries to truncate the incoming string as requested. If string does
     * not need shortening it is returned unchanged (property references
     * are resolved).
     **/
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        final Project P = clnt.getProject();
        int maxlength = MAXLEN;
        int dropped = ShortenStringURIHandler.RIGHT;
        String longstring = uriFragment;

        int i = longstring.indexOf("?");
        if (i>=0) {
            longstring = longstring.substring(0,i++);
            if (i<uriFragment.length()) {
                String s;
                int j = uriFragment.indexOf(",,",i);
                if (j>=0) {
                    s = Tk.resolveString(P,uriFragment.substring(i,j),true);
                    maxlength = Tk.integerFrom(s,MAXLEN);
                    s = uriFragment.substring(j+2);
                    dropped = ShortenStringURIHandler.leftOrRight(s,dropped,P);
                } else {
                    s = Tk.resolveString(P,uriFragment.substring(i),true);
                    maxlength = Tk.integerFrom(s,MAXLEN);
                }
            }
        }

        if (maxlength<MINLEN) {
            maxlength=MINLEN;
        }
        longstring = Tk.resolveString(P,longstring,true);
        final int strlen = longstring.length();
        if (strlen>maxlength) {
            switch (dropped) {
                case ShortenStringURIHandler.LEFT: {
                    int from = strlen-maxlength;
                    longstring= longstring.substring(from);
                    break;
                }
                default: {
                    longstring = longstring.substring(0,maxlength);
                    break;
                }
            }
        }
        return longstring;
    }
}

/* end-of-TruncateStringURIHandler.java */