/**
 * $Id: UtilitiesTest.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.tests;

import  java.util.Collections;
import  java.util.List;
import  java.util.Map;

import  junit.framework.TestSuite;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.ut.HTC;
import  com.idaremedia.antx.ut.HTCUtils;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Tests to ensure the various AntX helper utility classes work as expected.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,test
 **/

public final class UtilitiesTest extends HTC
{
    /** <i>PET</i> Test Category. **/
    public static final String TEST_CATEGORY="CLASS";


    /**
     * Create new testcase.
     **/
    public UtilitiesTest(String methodName)
    {
        super("Utilities::",methodName);
    }


    /**
     * Create full test suite.
     **/
    public static TestSuite suite()
    {
        return new TestSuite(UtilitiesTest.class);
    }


    /**
     * Create baseline test suite.
     **/
    public static TestSuite baseline()
    {
        return suite();
    }


    /**
     * Make this test (standalone) self-running.
     **/
    public static void main(String[] argv)
    {
        HTCUtils.quickCheck(suite());
    }

// ---------------------------------------------------------------------------------------------------------
// ---------------------------------------- [ Misc Setup Methods ] -----------------------------------------
// ---------------------------------------------------------------------------------------------------------

    protected void setUp() throws Exception
    {
        configureProjectFromResource("empty.xml");
        Project P = getProject();
        assertNotNil(P,"test's project");
    }

// ---------------------------------------------------------------------------------------------------------
// ------------------------------------------- [ The Test Cases ] ------------------------------------------
// ---------------------------------------------------------------------------------------------------------

    public void checkBaseline()
    {
        //--Ensures setUp() works and can find our xml file!
    }


    public void testBaseline()
    {
        checkBaseline();
    }


    /**
     * Verifies we can access internal AntX ui strings as expected.
     **/
    public void testAntXStringsAccessible_AntX04()
    {
        String s1 = AntX.uistrs().get("antx_.debug.helloworld");
        assertNotWhitespace(s1,"AntX debug UIString");
        assertTrue(s1.indexOf("DO NOT REMOVE")>=0,"Expected AntX debug string");

        //Commonly used...verify something there
        assertNotWhitespace(AntX.uistrs().get(AntX.CLONE_BROKEN_MSGID));
        assertNotWhitespace(AntX.uistrs().get("cv.require"));
        assertNotWhitespace(AntX.uistrs().get("task.warn.property.exists"));
        assertNotWhitespace(AntX.uistrs().get("task.cant.flip.unknown"));
    }


    /**
     * Verifies we can avoid overhead for readonly maps as expected.
     **/
    public void testReadonlyMapWrapperUtil_AntX04()
    {
        Map mp;
        Map mutableMap = AntXFixture.newMap();

        assertNil(Tk.readonlyFrom(null,false));
        mp = Tk.readonlyFrom(null,true);
        assertNotNil(mp);
        assertIdent(mp,Tk.readonlyFrom(mp,false));

        mp= Collections.EMPTY_MAP;
        assertIdent(mp,Tk.readonlyFrom(mp,false));

        mp= Collections.unmodifiableMap(mutableMap);
        assertIdent(mp,Tk.readonlyFrom(mp,false));
        assertNotIdent(mp,Tk.readonlyFrom(mutableMap,false));
    }


    /**
     * Verifies we can avoid overhead for strings with no variables.
     **/
    public void testResolveStringOverheadUtil_AntX04()
    {
        Project P= getProject();

        assertNil(Tk.resolveString(P,null));

        String s = "no variables here";
        assertIdent(s,Tk.resolveString(P,s),"no variables in string");

        assertNotNil(P.getProperty("p.true"),"p.true definition");
        s = "This is a test: ${p.true}";
        assertNotEqual(s,Tk.resolveString(P,s),"variables in string");
    }



    /**
     * Verifies we can switch from macro-isque attributes to
     * standard project properties.
     **/
    public void testResolveLocalVariables_AntX04()
    {
        Project P= getProject();

        assertNil(Tk.resolveString(P,null));

        String s = "no variables here";
        assertIdent(s,Tk.resolveString(P,s,true),"no variables in string");

        assertNotNil(P.getProperty("p.true"),"p.true definition");

        s = "This is a test: @{p.true}. Happy ? @{p.false}";
        String xpected = "This is a test: true. Happy ? false";
        String out = Tk.resolveString(P,s,true);

        assertEqual(out,xpected,"local variables in string(a)");

        s = "This is a test: ${p.true}. Happy ? @{p.false}";
        out = Tk.resolveString(P,s,true);
        assertEqual(out,xpected,"local variables in string(b)");
    }


    /**
     * Verifies we can switch from value-URI-friendly attributes
     * to standard project attributes.
     * @since JWare/AntX 0.5
     */
    public void testResolveValueURIFriendlyRefs_AntX05()
    {
        Project P= getProject();
        String s;

        s = "This is a test: @(p.true). Happy ? @(p.false)";
        String xpected = "This is a test: true. Happy ? false";
        String out = Tk.resolveString(P,s,true);

        assertEqual(out,xpected,"local variables in string(a)");

        s = "This is a test: ${p.true}. Happy ? @(p.false)";
        out = Tk.resolveString(P,s,true);
        assertEqual(out,xpected,"local variables in string(b)");
    }


    /**
     * Verifies that an empty string is interpreted as an empty 
     * list by Tk.splitList.
     * @since JWare/AntX 0.5
     **/
    public void testSplitListEmptyStringIsZeroLengthList_AntX05()
    {
        List l = Tk.splitList("");
        assertNotNil(l,"empty string's list");
        assertEqual(0,l.size(),"list's size");
    }
}

/* end-of-UtilitiesTest.java */
