/**
 * $Id: StringItemList.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2003-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.starters;

import  java.io.BufferedReader;
import  java.io.ByteArrayInputStream;
import  java.io.InputStream;
import  java.io.InputStreamReader;
import  java.io.IOException;
import  java.util.ArrayList;
import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableDataType;
import  com.idaremedia.antx.FixtureComponent;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.helpers.ArrayIterator;
import  com.idaremedia.antx.helpers.Empties;
import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.helpers.InputFileLoader;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.LocalTk;

/**
 * A structured collection of strings or string-representations as a formal Ant type.
 * <p>
 * StringItemLists bring formal structure to a sequence of items that are (initially)
 * represented as strings in a build script. StringItemLists can be subclassed to
 * create live (non-string) entities from the strings; for example, a list of URLs
 * or Files can start life in a build script as strings.  The StringItemList allows
 * the real types to be created and verified when the data is declared instead of the
 * first time the declaration is used (or referenced).
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public abstract class StringItemList extends AssertableDataType
    implements Cloneable, FixtureComponent
{
    /** Flag passed to {@linkplain #copyOfStrings copyOfStrings}. **/
    public final static Boolean PROCESSED=Boolean.FALSE;
    
    /** Flag passed to {@linkplain #copyOfStrings copyOfStrings}. **/
    public final static Boolean RAW= Boolean.TRUE;



    /**
     * Initializes a string item list.
     **/
    protected StringItemList()
    {
        super(AntX.starters);
        m_strings= new ArrayList(10);
    }



    /**
     * Initializes a CV-labeled string item list.
     * @param iam CV-label (non-null)
     **/
    protected StringItemList(String iam)
    {
        super(iam);
        m_strings= new ArrayList(10);
    }



    /**
     * Initializes a new pre-initialized string item list.
     * @param iam CV-label (non-null)
     * @param list comma-delimited list of strings (non-null)
     **/
    protected StringItemList(String iam, String list)
    {
        super();
        List l = Tk.splitList(list,AntX.DEFAULT_DELIMITER);
        if (l instanceof ArrayList) {
            m_strings = (ArrayList)l;
        } else {
            m_strings= new ArrayList(l);
            l= null;
        }
    }



    /**
     * Returns a deep-clone of this string item list. Underlying
     * elements are also cloned. If this item is a reference to
     * another object, that object's clone method is invoked.
     **/
    public Object clone()
    {
        if (isReference()) {
            return getOtherItemList(getClass()).clone();
        }
        try {
            StringItemList copy = (StringItemList)super.clone();
            copy.m_strings = (ArrayList)m_strings.clone();
            cloneInternals(copy);
            return copy;
        } catch(CloneNotSupportedException clnx) {
            throw new Error(uistrs().get(AntX.CLONE_BROKEN_MSGID));
        }
    }



    /**
     * Capture our identifier for feedback since types don't always
     * get correct location information.
     * @param id our identifier (via build script)
     **/
    public void setId(String id)
    {
        m_Id= id;
    }



    /**
     * Tries to return an unique identifier for this type. Looks
     * for a build-script identifier; if none found creates a
     * unique string based on type's class name and instance
     * reference.
     **/
    public final String getId()
    {
        if (m_Id!=null) {
            return m_Id;
        }
        if (isReference()) {
            return getOtherItemList(getClass()).getId();
        }
        return super.getId();
    }



    /**
     * Returns <i>true</i> if this string item list is empty.
     **/
    public final boolean isEmpty()
    {
        if (isReference()) {
            return getOtherItemList(getClass()).isEmpty();
        }
        return m_strings.isEmpty();
    }



    /**
     * Returns the current size of this string item list.
     **/
    public final int size()
    {
        if (isReference()) {
            return getOtherItemList(getClass()).size();
        }
        return m_strings.size();
    }



    /**
     * Returns an iterator for a <em>snapshot</em> of this string
     * item list's current converted contents. Modifications via
     * the returned iterator are <em>not</em> reflected in this
     * list.
     * @see #copyOfItems copyOfItems
     **/
    public final Iterator readonlyIterator()
    {
        if (isReference()) {
            return getOtherItemList(getClass()).readonlyIterator();
        }
        return copyOfItems(getProject()).iterator();
    }



    /**
     * Returns an iterator for an <em>snapshot</em> of this item
     * list's current strings. Modifications via the returned iterator
     * are <em>not</em> reflected in this list.
     * @see #copyOfStrings copyOfStrings
     **/
    public final Iterator readonlyStringIterator()
    {
        if (isReference()) {
            return getOtherItemList(getClass()).readonlyStringIterator();
        }
        return new ArrayIterator(copyOfStrings(getProject(),PROCESSED));
    }



    /**
     * Returns an iterator for an <em>snapshot</em> of this item
     * list's current strings. Modifications via the returned iterator
     * are <em>not</em> reflected in this list.
     * @param p [optional] project against which property references resolved.
     * @see #copyOfStrings copyOfStrings
     * @since JWare/AntX 0.5
     **/
    public final Iterator readonlyStringIterator(Project p)
    {
        if (isReference()) {
            return getOtherItemList(getClass()).readonlyStringIterator(p);
        }
        return new ArrayIterator(copyOfStrings(p,PROCESSED));
    }



    /**
     * Returns an iterator for an <em>snapshot</em> of this item
     * list's current strings. Modifications via the returned iterator
     * are <em>not</em> reflected in this list.
     * @param p [optional] project against which property references resolved.
     * @param how flag for whether or not prefix/suffix applied
     * @see #copyOfStrings copyOfStrings
     * @see #RAW
     * @see #PROCESSED
     * @since JWare/AntX 0.5
     **/
    public final Iterator readonlyStringIterator(Project p, Boolean how)
    {
        if (isReference()) {
            return getOtherItemList(getClass()).readonlyStringIterator(p,how);
        }
        require_(how!=null,"readonlyItr- nonzro how flag");
        return new ArrayIterator(copyOfStrings(p,how));
    }



    /**
     * Appends given item's string (without property substitution).
     * This method should be called by the subclasses' script-facing
     * equivalent.
     * @param item new item
     * @throws BuildException if this item is a reference
     **/
    protected void addItem(InnerString item)
    {
        require_(item!=null,"addItm- nonzro itm");
        if (isReference()) {
            throw tooManyAttributes();
        }
        addItemFinal(item.toString());
        edited();
    }



    /**
     * Appends given item's string with property substitution
     * using given project as property value source.
     * This method should be called by the subclasses' script-facing
     * equivalent.
     * @param item new item
     * @param theProject source project for replaced properties
     * @throws BuildException if this item is a reference
     **/
    protected void addItem(InnerString item, Project theProject)
    {
        require_(item!=null,"addItm- nonzro itm");
        if (isReference()) {
            throw tooManyAttributes();
        }
        addItemFinal(item.toString(theProject));
        edited();
    }



    /**
     * Appends a flex string's determined value.
     * Caller must configure flex string's project information
     * before calling this method. This method should be
     * called by the subclasses' script-facing equivalent.
     * @param item new item
     * @throws BuildException if this item is a reference
     **/
    protected void addItem(FlexString item)
    {
        require_(item!=null,"addItm- nonzro itm");
        if (isReference()) {
            throw tooManyAttributes();
        }
        addItemFinal(item.getValue());
        edited();
    }



    /**
     * Adds the contents of an byte-encoded character buffer to this list.
     * The stream must contain newline-delimited strings in the current
     * runtime's character encoding. This method acts as a no-op if
     * the stream is empty.
     * @param allStrings the characters to be read (non-null)
     * @return <i>true</i> if at least one item added to this list
     * @throws BuildException if unable to read file once opened
     * @since JWare/AntX 0.4
     **/
    private boolean addBuffer(byte[] allStrings)
    {
        require_(allStrings!=null,"addBfr- nonzro byt[]");
        BufferedReader r= new BufferedReader
                (new InputStreamReader(new ByteArrayInputStream(allStrings)));
        int N=0;
        try {
            String ln;
            while ((ln=r.readLine())!=null) {
                addItemFinal(ln);
                N++;
            }
        } catch(IOException iox) {
            throw new BuildException(iox);
        } finally {
            if (N!=0) {
                edited();
            }
        }
        return N!=0;
    }



    /**
     * Adds the contents of the given file to this list. The file
     * must contain newline-delimited strings. This method acts as
     * a no-op if the file does not exist or is empty.
     * @param f file path or URL string to be read (non-null)
     * @return <i>true</i> if at least one item added to this list
     * @throws BuildException if unable to read file once opened
     * @since JWare/AntX 0.4
     **/
    protected boolean addFileOrURL(String f)
    {
        require_(f!=null,"addFil- nonzro filnam");
        if (isReference()) {
            throw tooManyAttributes();
        }
        byte[] bytes=null;
        try {
            bytes = new InputFileLoader().loadFile(f);
        } catch(IOException iox) {
            String warning = uistrs().get
                ("solo.list.filenotfound", getId(), f);
            log(warning,Project.MSG_WARN);
            return false;
        }
        return addBuffer(bytes);
    }



    /**
     * Adds the contents of the given resource file to this list.
     * The resource file must contain newline-delimited strings.
     * This method acts as a no-op if the no such resource exists.
     * @param rn (subpath) name of resoure (non-null)
     * @throws BuildException if unable to read resoure once opened
     * @since JWare/AntX 0.4
     **/
    protected boolean addResource(String rn)
    {
        require_(rn!=null,"addRsrc- nonzro name");
        if (isReference()) {
            throw tooManyAttributes();
        }
        InputStream is = LocalTk.getSystemResourceAsStream(rn, getProject());
        if (is==null) {
            return false;
        }
        byte[] bytes=null;
        try {
            bytes = new InputFileLoader().load(is);
        } catch(IOException iox) {
            String error = uistrs().get
                ("solo.list.reznotfound", getId(), rn);
            log(error,Project.MSG_ERR);
            throw new BuildException(error,iox);
        }
        return addBuffer(bytes);
    }



    /**
     * Appends given string (as-is) to this string item list.
     * @param string string to append (non-null)
     **/
    protected final void addItemFinal(String string)
    {
        if (!includeItem(string)) {
            String error = uistrs().get("solo.list.invalid.item",getId(),
                                        String.valueOf(string));
            log(error,Project.MSG_ERR);
            throw new BuildException(error);
        }
        m_strings.add(string);
    }



    /**
     * Filter to prevent illegal string forms from making it into
     * this list. Called by {@linkplain #addItemFinal addItemFinal()}
     * just before appending to underlying list. By default, allows
     * any non-null string (even empty ones).
     **/
    protected boolean includeItem(String candidate)
    {
        return candidate!=null;
    }



    /**
     * Returns a copy of this string item list's underlying
     * strings. Caller assumes ownership of returned array.
     * Never returns <i>null</i>. Returned strings have had
     * any prefix and/or suffix  modifications applied.
     * @param p [optional] project from which property references read.
     * @param how [optional] flag for whether or not prefix/suffix applied
     * @see #isCustom
     * @see #PROCESSED
     * @see #RAW
     **/
    protected String[] copyOfStrings(Project p, Boolean how)
    {
        verify_(!isReference(),"copyOfStrs- standalone");//?
        String[] sl=null;
        if (how==PROCESSED && isCustom()) {
            sl = new String[m_strings.size()];
            StringBuffer sb = new StringBuffer(150);
            for (int i=0;i<sl.length;i++) {
                sb.append(getPrefixString());
                sb.append(Tk.resolveString(p,m_strings.get(i).toString(),true));
                sb.append(getSuffixString());
                sl[i]= sb.substring(0);
                sb.delete(0,sb.length());
            }
        } else if (p!=null) {//NB:resolve any @(...) references! AntX-0.5
            sl = new String[m_strings.size()];
            StringBuffer sb = new StringBuffer(150);
            for (int i=0;i<sl.length;i++) {
                sb.append(Tk.resolveString(p,m_strings.get(i).toString(),true));
                sl[i]= sb.substring(0);
                sb.delete(0,sb.length());
            }
        } else {
            sl = (String[])m_strings.toArray(Empties.EMPTY_STRING_ARRAY);
        }
        return sl;
    }



    /**
     * Returns a copy of this string item list's underlying
     * strings as converted matching objects. Caller assumes
     * ownership of returned array. Never returns <i>null</i>.
     * Returned items have had  any prefix and/or suffix
     * modifications applied.
     * @param p [optional] project from which property references read.
     * @see #isCustom
     **/
    protected List copyOfItems(Project p)
    {
        verify_(!isReference(),"copyOfItms- standalone");//?
        String[] sl = copyOfStrings(p,PROCESSED);
        ArrayList copy = new ArrayList(sl.length);
        for (int i=0;i<sl.length;i++) {
            copy.add(sl[i]);
            sl[i]= null;
        }
        sl=null;
        return copy;
    }



    /**
     * Return the raw underlying list of strings. Should never
     * be exposed via any public interface. Never returns
     * <i>null</i>.
     **/
    protected final List rawStringsList()
    {
        return m_strings;
    }



    /**
     * Sets a prefix that will be prepended to all strings
     * returned by {@linkplain #copyOfStrings copyOfStrings} and
     * {@linkplain #copyOfItems copyOfItems}. This
     * method should be called by the subclasses' script-facing
     * equivalent.
     * @param prefix the prefix (non-null)
     * @throws BuildException if this item is a reference
     **/
    protected void setPrefixString(String prefix)
    {
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_prefix = prefix==null ? "" : prefix;
        edited();
    }



    /**
     * Returns this string item list's prefix or the empty
     * string if never set (or reset). Never returns <i>null</i>.
     **/
    protected final String getPrefixString()
    {
        return m_prefix;
    }



    /**
     * Sets a suffix that will be appended to all strings
     * returned by {@linkplain #copyOfStrings copyOfStrings} and
     * {@linkplain #copyOfItems copyOfItems}. This
     * method should be called by the subclasses' script-facing
     * equivalent.
     * @param suffix the suffix (non-null)
     * @throws BuildException if this item is a reference
     **/
    protected void setSuffixString(String suffix)
    {
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_suffix = suffix==null ? "" : suffix;
        edited();
    }



    /**
     * Returns this string item list's suffix or the empty
     * string if never set (or reset). Never returns <i>null</i>.
     **/
    protected final String getSuffixString()
    {
        return m_suffix;
    }



    /**
     * Called to by {@linkplain #clone clone()} to copy converted
     * string information. The calling method has already cloned
     * the raw underlying string list and other fields. A hook
     * for subclasses; by default does nothing.
     * @param cloned the cloned item (non-null)
     **/
    protected void cloneInternals(StringItemList cloned)
    {
        //nothing for strings.
    }



    /**
     * Returns the StringItemList referenced by this instance.
     * @param c required class of referred-to thingy (non-null)
     **/
    protected final StringItemList getOtherItemList(Class c)
    {
        return (StringItemList)getCheckedRef(c,getTypicalName(c));
    }



    /**
     * Returns <i>true</i> if cannot ignore modifier string.
     **/
    private boolean willModify(String it)
    {
        return it!=null && it.length()>0;
    }



    /**
     * Returns <i>true</i> if this list should modify raw strings
     * with prefix and/or suffix additions.
     * @see #rawStringsList
     **/
    protected final boolean shouldModifyStrings()
    {
        return willModify(getPrefixString()) || willModify(getSuffixString());
    }



    /**
     * Returns <i>true</i> if this item list's is not empty
     * and {@linkplain #shouldModifyStrings shouldModifyStrings} has
     * returned <i>true</i>.
     **/
    protected final boolean isCustom()
    {
        return (!m_strings.isEmpty()) && shouldModifyStrings();
    }



    /**
     * Returns a user-readable name for this type. By default
     * returns the lowercased leaf name of this item's class. Never
     * returns <i>null</i>.
     **/
    protected String getTypicalName(Class c)
    {
        if (c==null) { c = getClass(); }
        return Tk.lowercaseFrom(Tk.leafNameFrom(c));
    }



    private String m_Id;
    private ArrayList m_strings= new ArrayList(10);
    private String m_prefix="", m_suffix="";
}

/* end-of-StringItemList.java */
