/**
 * $Id: StrictAntLib.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.starters;

import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.UnknownElement;
import  org.apache.tools.ant.taskdefs.Antlib;
import  org.apache.tools.ant.taskdefs.AntlibDefinition;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.helpers.TaskHandle;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;

/**
 * Starting implementation for an AntX-based Antlib that restricts its contents in some
 * way. For example, an AntXtras ACM definition file will limit its contents to a single
 * <span class="src">&lt;macrodef&gt;</span>. The filtering process this task uses to
 * limit its contents is like that used by the AntX {@linkplain TaskSet} class. 
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,infra
 **/

public abstract class StrictAntLib extends Antlib implements StrictOuterTask
{
    /**
     * Initializes a new strict antlib instance.
     * @since JWare/AntX 0.5
     */
    protected StrictAntLib()
    {
        super();
        Iam_ = AntX.fixture+"AntLib:";
    }



    /**
     * Initializes a new labeled strict antlib instance.
     * @param iam CV-label (non-null)
     * @since JWare/AntX 0.5
     */
    protected StrictAntLib(String iam)
    {
        super();
        Iam_= Tk.cvlabelFrom(iam);
    }



    /**
     * Initializes the default URI for all definitions loaded by
     * this library.
     * @param uri the XML-NS URI identifier
     **/
    public void setURI(String uri)
    {
        super.setURI(uri);
        m_uriLocal = uri;
    }



    /**
     * Returns the default URI used for this library's definition
     * items. Will return <i>null</i> if never set explicitly.
     */
    public final String getDefaultItemURI()
    {
        return m_uriLocal;
    }



    /**
     * Initializes the default classloader of all definitions
     * loaded by this library.
     * @param cl the class loader
     */
    public void setClassLoader(ClassLoader cl)
    {
        super.setClassLoader(cl);
        m_clLocal = cl;
    }



    /**
     * Returns the default class loader used for this library's
     * definition items. Will return <i>null</i> if never set.
     */
    public final ClassLoader getDefaultItemClassLoader()
    {
        if (m_clLocal==null) {
            m_clLocal= Antlib.class.getClassLoader();
        }
        return m_clLocal;
    }



    /**
     * Adds another library definition task to this library if
     * allowed. Definition is performed when this library executed.
     * @param task task to be added (non-null)
     * @throws BuildException if task cannot be included in library
     * @since JWare/AntX 0.5
     **/
    public void addTask(Task task)
    {
        TaskHandle taskH = new TaskHandle(task);

        if (includeTask(taskH)) {
            getTasksList().add(taskH.getTask());
        } else {
            String ermsg = AntX.uistrs().get("taskset.nested.task.disallowed",
                                    task.getTaskName(), this.getTaskName());
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }



    /**
     * Returns this library's list of nested definitions. Each element
     * is list has been verified by {@linkplain #includeTask includeTask}.
     * @since JWare/AntX 0.5
     **/
    protected final List getTasksList()
    {
        return m_nestedTasks;
    }



    /**
     * Returns <i>true</i> if task inside handle is allowed to nest
     * within this library. By default returns <i>true</i> if the task 
     * is an instance of the <span class="src">AntlibDefinition</span>.
     * This method can also switch the task to be included in this 
     * library by updating the incoming handle. Warning: the task should
     * never be set to <i>null</i> if this method returns <i>true</i>.
     * @param taskH the task to be nested (as handle)
     * @since JWare/AntX 0.5
     * @see #checkTaskCount checkTaskCount
     * @see #checkTaskType checkTaskType
     **/
    protected boolean includeTask(TaskHandle taskH)
    {
        boolean ok= checkTaskCount(taskH);
        if (ok) {
            Class c = TaskExaminer.trueClass(taskH.getTask());
            ok = checkTaskType(c,taskH);
        }
        return ok;
    }



    /**
     * Template method that should verify whether another task can be
     * added to this library. Called by {@linkplain #includeTask includeTask}.
     * Many strict library limit the number of elements that can be
     * defined per library file.
     * @param taskH the task to be nested (as handle)
     * @return <i>false</i> if another task cannot be added
     **/
    protected boolean checkTaskCount(TaskHandle taskH)
    {
        return true;
    }



    /**
     * Template method that should verify whether a task of a particular
     * type can be added to this library.
     * @param c the true class of the task under test (non-null)
     * @param taskH the task to be nested (as handle)
     * @return <i>false</i> if task of named type cannot be added
     **/
    protected boolean checkTaskType(Class c, TaskHandle taskH)
    {
        return c!=null && AntlibDefinition.class.isAssignableFrom(c);
    }



    /**
     * Verifies we're in a live project (created from build process).
     * @throws IllegalStateException if not in project
     **/
    protected final void verifyInProject_(String calr)
    {
        if (getProject()==null) {
            String ermsg = AntX.uistrs().get("cv.verifyInP",Iam_,calr);
            log(ermsg, Project.MSG_ERR);
            throw new IllegalStateException(ermsg);
        }
    }



    /**
     * Ensures this library can proceed with task installation. By
     * default just checks to make sure our project reference has 
     * been defined. Subclasses will often check to make sure at least
     * one library item has been defined.
     * @param calr calling scope's identifier (non-null)
     **/
    protected void verifyCanExecute_(String calr)
    {
        verifyInProject_(calr);
    }



    /**
     * Executes a single nested library task. The incoming library
     * task must be already configured (not a placeholder) and
     * initialized.
     * @param nestedTask the library task (non-null)
     **/
    protected void performLibraryTask(Task nestedTask)
    {
        AntlibDefinition libTask = (AntlibDefinition)nestedTask;
        libTask.setURI(getDefaultItemURI());
        libTask.setAntlibClassLoader(getDefaultItemClassLoader());
        libTask.execute();//@.impl Use execute and *not* perform!
    }



    /**
     * Method called before this antlib performs its nested
     * library tasks. Does nothing by default.
     **/
    protected void aboutToInstall()
    {
        //nothing
    }
    


    /**
     * Method called after this antlib has tried to perform 
     * its nested tasks. The incoming uncaught exception reference
     * will be non-null if this task aborted prematurely due to
     * an unhandled exception; otherwise, it is <i>null</i>. Does
     * nothing by default.
     * @param uncaught [optional] reason for termination
     **/
    protected void installCompleted(RuntimeException uncaught)
    {
        //nothing
    }



    /**
     * Performs each of this library's nested elements. Does nothing
     * if this library is empty. Implemented as a template method with
     * several predefined hook methods for subclasses.
     * @see #aboutToInstall
     * @see #performLibraryTask performLibraryTask(&#8230;)
     * @see #installCompleted installCompleted(&#8230;)
     * @.pattern GoF.TemplateMethod
     **/
    public void execute()
    {
        verifyCanExecute_("exec");
        
        List tl = getTasksList();
        
        RuntimeException failure=null;
        aboutToInstall();
        try {
            if (!tl.isEmpty()) {
                Location libLocation = getLocation();

                for (Iterator itr= tl.iterator();itr.hasNext();) {
                    Task libitem = (Task)itr.next();
                    setLocation(libitem.getLocation());

                    if (libitem instanceof UnknownElement) {
                        UnknownElement ue= (UnknownElement)libitem;
                        ue.maybeConfigure();
                        libitem = ue.getTask();
                    }
                    if (libitem!=null) {
                        performLibraryTask(libitem);
                    }
                }

                setLocation(libLocation);
            }            
        } catch(RuntimeException rtX) {
            failure = rtX;
            throw rtX;
        } finally {
            installCompleted(failure);
        }
    }


    protected final String Iam_;
    private List m_nestedTasks= AntXFixture.newList(10);
    private String m_uriLocal;
    private ClassLoader m_clLocal;
}


/* end-of-StrictAntLib.java */