/**
 * $Id: MsgTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.starters;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.MsgGetter;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.go.Iff;
import  com.idaremedia.antx.go.Unless;
import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.Conditional;

/**
 * Simple task used to generate <i>msgid</i>-based messages for subclasses. Also useful
 * to test that resource bundles are where they're supposed to be. By default behaves
 * like the standard Ant '<i>echo</i>' task only using property-based resource bundles.
 * If undefined, the messages noise-level is whatever the AntX's runtime default is
 * defined to be.
 * <p>
 * Some examples of how to use MsgTask (assume MsgTask is mapped to the
 * <code>&lt;show&gt;</code> element):<pre>
 *   &lt;show msgid="msg.helloworld"/&gt;
 *   &lt;show msgid="msg.byeworld" level="verbose"/&gt;
 *   &lt;show msgid="msg.youarehere" msgarg1="Compiling" level="debug"/&gt;
 *   &lt;show msgid="err.no.jdk15" unless="jdk15.present"&gt;
 *      &lt;defaultmsg&gt;Need J2SE 1.5 or later&lt;/defaultmsg&gt;
 *   &lt;/show&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public class MsgTask extends AssertableTask implements Conditional
{
    /**
     * Initializes a new MsgTask.
     **/
    public MsgTask()
    {
        super(AntX.starters);
    }


    /**
     * Initializes a new CV-tagged MsgTask.
     * @param iam CV-label (non-null)
     **/
    public MsgTask(String iam)
    {
        super(iam);
    }


    /**
     * Initializes a new CV-tagged MsgTask with a predefined
     * default message.
     * @param iam CV-label (non-null)
     * @param defaultMsg default message (non-null)
     **/
    public MsgTask(String iam, String defaultMsg)
    {
        super(iam);
        require_(defaultMsg!=null,"ctor- nonzro dfltMsg");
        m_defaultMsgStr = defaultMsg;
    }


    /**
     * Adds an if-condition to this echo task.
     * @since JWare/AntX 0.2
     **/
    public void setIf(String property)
    {
        m_ifProperty = (property==null) ? "" : property;
    }


    /**
     * Returns this task's (raw) if-condition if any. Returns
     * empty string if condition never set.
     * @since JWare/AntX 0.2
     **/
    public String getIfProperty()
    {
        return m_ifProperty;
    }


    /**
     * Adds an unless-condition to this echo task.
     * @since JWare/AntX 0.2
     **/
    public void setUnless(String property)
    {
        m_unlessProperty = (property==null) ? "" : property;
    }


    /**
     * Returns this task's (raw) unless-condition if any. Returns
     * empty string if condition never set.
     * @since JWare/AntX 0.2
     **/
    public String getUnlessProperty()
    {
        return m_unlessProperty;
    }


    /**
     * Sets this task's preferred message noise level.
     * @param level new level (non-null)
     * @see #getPreferredMsgLevel
     **/
    public void setMsgLevel(NoiseLevel level)
    {
        require_(level!=null,"setLvl- nonzro lvl");
        m_preferredMsgLevel = level;
    }


    /**
     * Synonym for {@linkplain #setMsgLevel setMsgLevel} that matches
     * the standard '<i>echo</i>' task's parameter.
     * @param nl the noise level (non-null)
     * @since JWare/AntX 0.2
     **/
    public final void setLevel(NoiseLevel nl)
    {
        setMsgLevel(nl);
    }


    /**
     * Returns this task's preferred message noise level. Returns
     * <i>null</i> if never set. For the message level used when
     * showing message, see {@linkplain #getMsgLevel}.
     **/
    public final NoiseLevel getPreferredMsgLevel()
    {
        return m_preferredMsgLevel;
    }


    /**
     * Returns this task's <em>effective</em> message noise level. If
     * this task's preferred msg level never set, this method returns
     * the default message level set at either the project, system, or
     * AntX-defaults levels. Never returns <i>null</i>.
     * @see AssertableTask#getDefaultMsgLevel
     **/
    public final NoiseLevel getMsgLevel()
    {
        NoiseLevel nl= getPreferredMsgLevel();
        if (nl==null) {
            nl= getDefaultMsgLevel();
        }
        return nl;
    }


    /**
     * <em>Replaces</em> the default message string associated with
     * this task. Usually only specified if it is acceptable for resource
     * bundles to be missing at build-time.
     * @see #getDefaultMsg
     **/
    public void addConfiguredDefaultMsg(InnerString defaultMsgStr)
    {
        require_(defaultMsgStr!=null,"addDfltMsg- nonzro str");
        m_defaultMsgStr= defaultMsgStr.toString(getProject());
    }


    /**
     * Returns this task's default message string if any. Never
     * returns <i>null</i> but will return the empty string if this
     * element never defined.
     **/
    public final String getDefaultMsg()
    {
        return m_defaultMsgStr;
    }


    /**
     * Sets the inlined string message emitted by this task. Same
     * as nesting a default message.
     * @param string the message (non-null)
     **/
    public void setMessage(String string)
    {
        require_(string!=null,"setMsg- nonzro string");
        addConfiguredDefaultMsg(new InnerString(string));
    }


    /**
     * Shortcut to dynamically define a single additional substition
     * value in final message. This argument is mapped to the
     * <i>{2}</i> template variable.
     **/
    public final void setMsgArg1(String arg1)
    {
        m_arg1= arg1;
    }


    /**
     * Returns optional first argument to msgid's template string.
     * This argument is mapped to the <i>{2}</i> template variable.
     **/
    public final String getMsgArg1()
    {
        return m_arg1;
    }


    /**
     * Shortcut to dynamically define a second additional substition
     * value in final message.  This argument is mapped to the
     * <i>{3}</i> template variable.
     **/
    public final void setMsgArg2(String arg2)
    {
        m_arg2= arg2;
    }


    /**
     * Returns optional second argument to msgid's template string. This
     * argument is mapped to the <i>{3}</i> template variable.
     **/
    public final String getMsgArg2()
    {
        return m_arg2;
    }


    /**
     * Extension of inherited '<i>getMsg</i>' that will return this
     * task's default message string if its msgid is undefined.
     * @see #setMsgArg1
     * @see #setMsgArg2
     **/
    public String getMsg()
    {
        String  msg = null;
        boolean chk = false;

        if (m_arg1!=null) {
            if (m_arg2!=null) {
                msg= getMsg(newMsgGetter(m_arg1,m_arg2));
            } else {
                msg= getMsg(newMsgGetter(m_arg1));
            }
        } else if (m_arg2!=null) {
            msg = getMsg(newMsgGetter("",m_arg2));
        } else {
            msg = super.getMsg();
            chk = true;
        }

        if (chk) {
            String dfltMsg = getDefaultMsg();
            if (Tk.isWhitespace(msg)) {
                msg = dfltMsg;
            } else if (msg.equals(getMsgId()) && !Tk.isWhitespace(dfltMsg)) {
                msg = dfltMsg;
            }
        }

        return msg;
    }


    /**
     * Extension of inherited '<i>getMsg(MsgGetter)</i>' that will
     * return this task's default message string if its msgid is
     * undefined.
     **/
    public String getMsg(MsgGetter getr)
    {
        String msg = super.getMsg(getr);

        if (Tk.isWhitespace(msg)) {
            msg = getDefaultMsg();
        }

        return msg;
    }


    /**
     * Tests whether or not this task's "if" condition resolves
     * to a an existing property.
     * @return <i>true</i>if the "if" property is defined
     * @since JWare/AntX 0.2
     **/
    protected final boolean testIfCondition()
    {
        return Iff.allowed(getIfProperty(), getProject());
    }


    /**
     * Tests whether or not this task's "unless" condition resolves
     * to an existing property.
     * @return <i>true</i> if the "unless" property is not defined
     * @since JWare/AntX 0.2
     **/
    protected final boolean testUnlessCondition()
    {
        return Unless.allowed(getUnlessProperty(),getProject());
    }


    /**
     * Work like the default '<i>echo</i>' Ant task except using
     * UIStringManager underneath if can.
     **/
    public void execute()
    {
        verifyCanExecute_("execute");

        if (testIfCondition() && testUnlessCondition()) {
            log(getMsg(), getMsgLevel().getNativeIndex());
        }
    }


    private NoiseLevel m_preferredMsgLevel;
    private String m_defaultMsgStr="";
    private String m_arg1, m_arg2;//NB: additonal args to MessageFormat
    private String m_ifProperty="";
    private String m_unlessProperty="";
}


/* end-of-MsgTask.java */
