/**
 * $Id: IterationConfigurator.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.starters;

import  java.lang.reflect.Field;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AssertableLibDefinition;
import  com.idaremedia.antx.FixtureInitializer;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Fixture administrator that initializes an (internal) iteration field. Iteration
 * configurators make it possible to support a variety of extensions to iteration w/o
 * polluting the class with public setters and other implementation-specific class
 * signatures.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   special (object setters are single, injection is guarded)
 * @.group    impl,infra
 * @.pattern  Fowler.DependencyInjection
 * @see       Iteration
 **/

public abstract class IterationConfigurator extends AssertableLibDefinition
    implements FixtureInitializer
{
    /**
     * Initializes a new iteration configurator.
     * @param iam CV-label (non-null)
     **/
    protected IterationConfigurator(String iam)
    {
        super(iam);
    }



    /**
     * Initializes a new iteration configurator to setup the
     * named field.
     * @param iam CV-label (non-null)
     * @param fieldName iteration field name (non-null)
     **/
    protected IterationConfigurator(String iam, String fieldName)
    {
        super(iam);
        setInjectionFieldName(fieldName);
    }



    /**
     * Set the name of the field this configuration initializes.
     * @param fieldName iteration field name (non-null)
     **/
    protected void setInjectionFieldName(String fieldName)
    {
        require_(fieldName!=null,"setField- nonzro field");
        m_fieldName = fieldName;
    }



    /**
     * Returns the name of the field this configuration will
     * try to initialize. Will return <i>null</i> if never set
     * explicitly.
     **/
    protected final String getInjectionFieldName()
    {
        return m_fieldName;
    }



    /**
     * Factory method for this configurator's iteration field
     * value. The returned object is usually configured via normal
     * Ant reflection mechanisms. The configurator subclass must
     * ensure that the returned field object and the declared
     * field's type are compatible.
     * @param currentValue the current value of the context field
     * @return the field value to be injected
     * @see com.idaremedia.antx.Defaults#mergeBehavior()
     **/
    protected abstract Object newInjectionField(Object currentValue);



    /**
     * Ensures this configurator has been assigned a valid field
     * name. Subclasses will usually check that the current installed
     * Iteration is of expected class.
     * @throws BuildException if field name not initialized
     **/
    protected void verifyCanExecute_(String calr)
    {
        verifyInProject_(calr);

        if (m_fieldName==null) {
            String e = uistrs().get("fixture.ioc.unknown.field",
                                   Tk.leafNameFrom(getClass()));
            log(e,Project.MSG_ERR);
            throw new BuildException(e,getLocation());
        }
    }



    /**
     * Filter method to block automatic injection mechanism.
     * By default returns <i>true</i> (always update), but subclasses
     * may have "opt-out" mechanisms to prevent (re)injection.
     * This method is called <em>before</em> the standard verification
     * method, {@linkplain #verifyCanExecute_ verifyCanExecute(&#8230;)}.
     **/
    protected boolean doInjection()
    {
        return true;
    }



    /**
     * Called before the (current) iteration's field is set by
     * the injection process. Does nothing by default.
     * @param context the current iteration object (non-null)
     * @param currentValue the current value of the context field
     **/
    protected void verifyInitialFieldState(Iteration context,
        Object currentValue)
    {
        //nothing
    }



    /**
     * Called after the (current) iteration's field has been
     * initialized. Subclasses can do specified usage checks on
     * installed field (via Iteration public interface). Does 
     * nothing by default.
     **/
    protected void verifyAfterInjection()
    {
        //nothing
    }



    /**
     * If indicated, tries to create and initialize our custom
     * iteration field. If the condition method {@linkplain #doInjection}
     * returns <i>false</i> this method will <em>not</em> try to alter
     * the iteration's field(s).
     * @throws BuildException if unable to initialize field for 
     *         any reason.
     **/
    public void execute()
    {
        if (doInjection()) {
            verifyCanExecute_("exec");
            
            Iteration context = Iteration.get();
            try {
                Field f = Tk.findField(context.getClass(),m_fieldName);
                boolean open = openField(f);
                try {
                    //- guard against concurrent modification to our field,
                    //- particularly if subclass can do merging if needed!!
                    
                    synchronized(context.getConfigLock(m_fieldName)) {
                        Object current = f.get(context);
                        verifyInitialFieldState(context,current);
                        f.set(context,newInjectionField(current));
                    }
                    
                } finally {
                    closeField(f,open);
                }
                
            } catch(NoSuchFieldException nfX) {
                String e = uistrs().get("fixture.ioc.bad.field",
                            Tk.leafNameFrom(context.getClass()),m_fieldName);
                log(e,Project.MSG_ERR);
                throw new BuildException(e,nfX,getLocation());
            } catch(IllegalAccessException iaX) {
                throw new BuildException(iaX,getLocation());
            } catch(IllegalArgumentException baX) {
                throw new BuildException(baX,getLocation());
            }
    
            verifyAfterInjection();
        }
    }



    /**
     * Make the given iteration field accessible for update.
     * @param f the field (non-null)
     * @return <i>true</i> if the field was already accessible
     **/
    private boolean openField(Field f)
    {
        boolean is = f.isAccessible();
        if (!is) {
            f.setAccessible(true);
        }
        return is;
    }



    /**
     * Reset the given field back to its accessibility state
     * when {@linkplain #openField openField} was called.
     * @param f the field (non-null)
     * @param was <i>true</i> if the field was initially accessible
     **/
    private void closeField(Field f, boolean was)
    {
        if (!was) {
            f.setAccessible(false);
        }
    }


    private String m_fieldName;
}


/* end-of-IterationConfigurator.java */