/**
 * $Id: UrlList.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  java.io.File;
import  java.net.MalformedURLException;
import  java.net.URL;
import  java.util.ArrayList;
import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.ListFriendly;
import  com.idaremedia.antx.starters.StringItemList;

/**
 * A list of URLs as a structured Ant type. Useful to verify URLs as they are declared
 * instead of when they're used for the first time. A URL list's iterator returns URL
 * objects not URL strings.
 * <p>
 * <b>Examples:</b><pre>
 *     &lt;<b>urls</b> id="default.webservers" prefix="http://www." suffix=".info/"&gt;
 *       &lt;url value="idaremedia"/&gt;
 *       &lt;url value="jware"/&gt;
 *       &lt;url value="antxtras"/&gt;
 *     &lt;/urls&gt;
 *
 * -OR- (for use in AntX looping construct)
 *
 *     &lt;<b>urls</b> id="test.urls" delim="||;||"&gt;
 *       &lt;url value="http://localhost:9090/builds/"/&gt;
 *       &lt;url value="http://dev.mycompany.com/builds/"/&gt;
 *     &lt;/urls&gt;
 *     &#46;&#46;&#46;
 *
 *    <i>-WHERE-</i>
 *     &lt;callforeach items="test.urls" &#46;&#46;&#46;/&gt;
 *
 *    <i>-IS-SAME-AS-</i>
 *     &lt;copyproperty name="testurls" reference="test.urls"/&gt;
 *     &lt;callforeach list="${testurls}" delim="||;||" &#46;&#46;&#46;/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple (after fully configured)
 * @.group   api,helper
 * @.caveat  URL lists cannot handle AntX property references because they 
 *           are verified as they are built.
 **/

public final class UrlList extends StringItemList
    implements ListFriendly, AntLibFriendly
{
    /** Default delimiter used by all URL lists. **/
    public static final String DEFAULT_DELIMITER = ";";


    /**
     * Initializes a new UrlList instance.
     **/
    public UrlList()
    {
        super(AntX.fixture+"UrlList:");
    }


    /**
     * Sets a custom delimiter for this URL list. This
     * delimiter will be used by all stringification(TM) methods.
     * @param delimiter new delimiter (non-null)
     * @throws BuildException if this list is a reference
     **/
    public void setDelim(String delimiter)
    {
        require_(delimiter!=null,"setDelim- nonzro delim");
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_delim = delimiter;
        edited();
    }


    /**
     * Returns the delimiter this URL list will use for
     * any stringification(TM) of its contents. Defaults
     * to the semi-colon. Never returns <i>null</i>.
     **/
    public String getDelim()
    {
        if (isReference()) {
            return getOtherList().getDelim();
        }
        return m_delim;
    }


    /**
     * Sets a prefix that will be prepended to all URLs
     * strings processed by this list. Usually something like:
     * "<code>http://www.</code>".
     * @param prefix the prefix (non-null)
     * @throws BuildException if this item is a reference
     **/
    public void setPrefix(String prefix)
    {
        setPrefixString(prefix);
    }


    /**
     * Returns this URL list's prefix or the empty
     * string if never set. Never returns <i>null</i>.
     **/
    public String getPrefix()
    {
        if (isReference()) {
            return getOtherList().getPrefix();
        }
        return getPrefixString();
    }


    /**
     * Set a suffix that will be appended to all URLs
     * strings processed by this list. Usually something like:
     * "<code>.com/index.html</code>".
     * @param suffix the suffix (non-null)
     * @throws BuildException if this item is a reference
     **/
    public void setSuffix(String suffix)
    {
        setSuffixString(suffix);
    }


    /**
     * Returns this string list's suffix or the empty
     * string if never set. Never returns <i>null</i>.
     **/
    public String getSuffix()
    {
        if (isReference()) {
            return getOtherList().getSuffix();
        }
        return getSuffixString();
    }


    /**
     * Appends the contents of a newline-delimited file to
     * this string list. No-op if file is missing or empty.
     * @param f the file (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setFile(File f)
    {
        require_(f!=null,"setfil- nonzro fil");
        addFileOrURL(f.getPath());
    }


    /**
     * Appends the contents of a newline-delimited resource
     * file to this string list. No-op if file is missing or
     * empty.
     * @param rn the resource path (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setResource(String rn)
    {
        require_(rn!=null,"setRez- nonzro reznam");
        addResource(rn);
    }


    /**
     * Appends given URL string (with property substitution)
     * to this URL list. URL string is immediately converted
     * to a URL (so errors will fail quick during parse).
     * @param urlstring new URL string
     * @throws BuildException if invalid URL string or this list
     *         is a reference
     **/
    public void addConfiguredURL(InnerString urlstring)
    {
        require_(urlstring!=null,"addURL- nonzro urlstr");
        addItem(urlstring,getProject());
        addedURLstring();
    }


    /**
     * Returns a delimited list of this URL list's contents.
     * Stringifies the original declared strings not the URL
     * external forms.
     * @see #setDelim
     **/
    public String stringFrom(Project theProject)
    {
        if (isReference()) {
            return getOtherList().stringFrom(theProject);
        }
        StringBuffer sb = new StringBuffer(150);
        final String theDelimiter = getDelim();

        int N=0;
        Iterator itr= readonlyStringIterator(null);

        while (itr.hasNext()) {
            if (N>0) {
                sb.append(theDelimiter);
            }
            sb.append(itr.next().toString());
            N++;
        }
        itr=null;
        return sb.substring(0);
    }


    /**
     * Synonym for {@linkplain #stringFrom stringFrom()}.
     **/
    public String toString()
    {
        return stringFrom(getProject());
    }


    /**
     * Filter <i>null</i> strings and any string that
     * resolves as empty or all whitespace.
     * @param string candidate string
     **/
    protected boolean includeItem(String string)
    {
        return !Tk.isWhitespace(string);
    }


    /**
     * Returns a copy of this URL list's underlying URLs. Each
     * URL is copied since we don't know how URLs work internally
     * so it's not safe to assume they can be shared. Caller
     * assumes ownership of returned array and contents. Never
     * returns <i>null</i>.
     **/
    protected List copyOfItems(Project p)
    {
        ArrayList l = (ArrayList)m_URLs.clone();
        if (!l.isEmpty()) {
            try {
                for (int i=0,N=l.size();i<N;i++) {
                    l.set(i, new URL(l.get(i).toString()));
                }
            } catch(MalformedURLException mfx) {
                throw new BuildException(mfx);
            }
        }
        return l;
    }


    /**
     * Ensures clone object has its own copy of any matching
     * URLs.
     * @param cloned the newly cloned object (non-null)
     * @see #copyOfItems
     **/
    protected void cloneInternals(StringItemList cloned)
    {
        super.cloneInternals(cloned);
        ((UrlList)cloned).m_URLs = (ArrayList)copyOfItems(null);
    }


    /**
     * Converts a URL string to a URL object triggering protocol
     * handler verification by JVM.
     * @return URL based on string
     * @throws BuildException if unable to parse url string
     **/
    private URL convertURL(String urlstr)
    {
        try {
            if (shouldModifyStrings()) {
                StringBuffer sb = new StringBuffer(urlstr.length()+20);
                sb.append(getPrefixString());
                sb.append(urlstr);
                sb.append(getSuffixString());
                urlstr = sb.substring(0);
            }
            return new URL(urlstr);
        } catch (MalformedURLException mfx) {
            String error = uistrs().get("solo.urls.malformed",getId(),urlstr);
            log(error,Project.MSG_ERR);
            throw new BuildException(error);
        }
    }


    /**
     * Converts the last URL string to a live URL object.
     **/
    private void addedURLstring()
    {
        List sl= rawStringsList();
        String urlstr = (String)sl.get(sl.size()-1);
        m_URLs.add(convertURL(urlstr));
        ensure_(sl.size()==m_URLs.size(),"addedURL- synced lists");
    }


    private final UrlList getOtherList()
    {
        return (UrlList)getOtherItemList(UrlList.class);
    }


    private String m_delim = DEFAULT_DELIMITER;
    private ArrayList m_URLs = new ArrayList(5);
}

/* end-of-UrlList.java */
