/**
 * $Id: UnassignTask.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.PropertyHelper;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.parameters.FeedbackLevel;
import  com.idaremedia.antx.parameters.FlexValueSupport;
import  com.idaremedia.antx.parameters.IsA;

/**
 * Helper task that removes fixture elements like variables, references, and even
 * properties. You cannot remove command properties (like system and command-line
 * directives).
 * <p/>
 * <b>Example Usage:</b><pre>
 *  &lt;unassign variable="__bn"/&gt;
 *  &lt;unassign reference="ant.PropertyHelper" feedback="quiet"/&gt;
 *  &lt;unassign property="loopcount"/&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 * @see       ExportTask
 **/

public class UnassignTask extends AssertableTask
    implements FlexValueSupport, AntLibFriendly
{
    private static final int _NOTYPE= -1;//NB: must be invalid array index!


    /**
     * Initializes a new unassign task instance. This task's
     * source must be defined before it is executed.
     **/
    public UnassignTask()
    {
        super(AntX.fixture+"UnassignTask:");
    }


//  ---------------------------------------------------------------------------------------
//  Script-facing Parameters:
//  ---------------------------------------------------------------------------------------

    /**
     * Sets the property which this task will unassign. The property is
     * checked against the project's current list of user properties. If
     * there is a match, a build exception is thrown.
     * @param property the property to unassign (non-null)
     * @throws BuildException if property is an immutable property.
     */
    public void setProperty(String property)
    {
        require_(property!=null,"setProperty- nonzro name");
        verifyNotNamed();
        if (getProject().getUserProperty(property)!=null) {
            String error = getAntXMsg("task.cant.repl.userprop",property);
            log(error,Project.MSG_ERR);
            throw new BuildException(error, getLocation());
        }
        m_name = property;
        m_type = IsA.PROPERTY_INDEX;
    }



    /**
     * Sets the variable which this task will unassign. The variable
     * is removed unconditionally.
     * @param variable name of variable to unassign (non-null)
     */
    public void setVariable(String variable)
    {
        require_(variable!=null,"setVariable- nonzro name");
        verifyNotNamed();
        m_name = variable;
        m_type = IsA.VARIABLE_INDEX;
    }



    /**
     * Synonym for {@linkplain #setVariable setVariable}.
     **/
    public final void setVar(String variable)
    {
        setVariable(variable);
    }



    /**
     * Sets the reference which this task will unassign. The reference
     * is removed unconditionally; however, references that begin with
     * "<span class="src">ant.</span>" will cause a warning to be
     * registered unless the {@linkplain #setFeedback quiet option} has
     * been set.
     * @param refid name of reference to unassign (non-null)
     */
    public void setReference(String refid)
    {
        require_(refid!=null,"setReference- nonzro name");
        verifyNotNamed();
        m_name = refid;
        m_type = IsA.REFERENCE_INDEX;
    }



    /**
     * Instructs this unassigner whether to issue warnings when
     * told to unassign seemingly reserved fixture bits.
     * @param level feedback level (non-null).
     * @throws BuildException if unrecognized feedback level.
     **/
    public void setFeedback(String level)
    {
        require_(level!=null,"setFeedback- nonzro level");
        FeedbackLevel fb = FeedbackLevel.from(level);
        if (fb==null) {
            String e = getAntXMsg("task.illegal.param.value",
                            getTaskName(), level,"feedbacklevel");
            log(e, Project.MSG_ERR);
            throw new BuildException(e, getLocation());
        }
        m_fbLevel = fb;
    }


//  ---------------------------------------------------------------------------------------
//  Execution:
//  ---------------------------------------------------------------------------------------

    /**
     * Verifies that a source element has been specified.
     * @param calr calling method (non-null)
     * @throws BuildException if neither a variable, reference,
     *  nor property has been specified.
     */
    protected void verifyCanExecute(String calr)
    {
        super.verifyCanExecute_(calr);
        if (m_name==null) {
            String error = getAntXMsg("task.needs.this.attr",
                getTaskName(), "property|reference|variable");
            log(error, Project.MSG_ERR);
            throw new BuildException(error, getLocation());
        }
    }



    /**
     * Removes the indicated fixture element. Note that properties
     * cannot be completely removed (no way via APIs); instead, their
     * values are (re)set to the empty string.
     */
    public void execute()
    {
        verifyCanExecute_("exec");

        final Project myproject = getProject();
        switch(m_type) {
            case IsA.PROPERTY_INDEX: {
                PropertyHelper ph = PropertyHelper.getPropertyHelper(myproject);
                synchronized(ph) {
                    if (ph.getProperty(null,m_name)!=null) {
                        ph.setProperty(null,m_name,"",false);
                    }
                }
                break;
            }
            case IsA.VARIABLE_INDEX: {
                ExportedProperties.delete(m_name);
                break;
            }
            case IsA.REFERENCE_INDEX: {
                if (m_name.startsWith("ant.") &&
                    !FeedbackLevel.isQuietish(m_fbLevel,true)) {
                    String warning = getAntXMsg("unassign.warn.ant.runtime.ref",m_name);
                    log(warning, Project.MSG_VERBOSE);
                }
                myproject.getReferences().remove(m_name);
                break;
            }
        }
    }



    /**
     * Ensures this task has not been assigned a fixture data
     * source already.
     * @param errid [optiona] id of error message to use
     **/
    protected final void verifyNotNamed(String errid)
    {
        if (m_name!=null) {
            if (errid==null) {
                errid = "task.too.many.flex.attrs";
            }
            String error = getAntXMsg(errid);
            log(error, Project.MSG_ERR);
            throw new BuildException(error, getLocation());
        }
    }


    private void verifyNotNamed()
    {
        verifyNotNamed(null);
    }


    private String m_name;
    private int m_type=_NOTYPE;
    private FeedbackLevel m_fbLevel=FeedbackLevel.NORMAL;
}

/* end-of-UnassignTask.java */