/**
 * $Id: StringList.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2003-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  java.io.File;
import  java.util.Iterator;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.starters.ListFriendly;
import  com.idaremedia.antx.starters.StringItemList;
import  com.idaremedia.antx.starters.StringItemListHandle;

/**
 * A list of strings as a formal Ant type. Useful to minimize those "comma-delimited"
 * lists of string, filenames, etc. that seem to sprout up everywhere.
 * <p>
 * <b>Examples:</b><pre>
 *     &lt;<b>strings</b> id="default.extensions" prefix="*."&gt;
 *       &lt;string value="java"/&gt;
 *       &lt;string value="xml"/&gt;
 *       &lt;string value="ent"/&gt;
 *       &lt;string value="properties"/&gt;
 *     &lt;/strings&gt;
 *
 *     &lt;<b>strings</b> id="my.extensions" prefix="*." file="${user.home}/.buildext"/&gt;
 * 
 *     &lt;<b>strings</b> id="my.extensions" prefix="*." file="${user.home}/.buildext"&gt;
 *        &lt;string value="xsl"/&gt;
 *        &lt;string value="php"/&gt;
 *        &lt;strings listref="defaults.extensions" raw="yes"/&gt;
 *     &lt;/strings&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple (after fully configured)
 * @.group   api,helper
 **/

public final class StringList extends StringItemList
    implements ListFriendly, AntLibFriendly
{
    /**
     * Initializes a new StringList instance.
     **/
    public StringList()
    {
        super(AntX.fixture+"StringList:");
    }


    /**
     * Initializes a new pre-initialized StringList instance.
     **/
    public StringList(String list)
    {
        super(AntX.fixture+"StringList:",list);

    }


    /**
     * Sets a custom delimiter for this string list. This
     * delimiter is used by stringification(TM) methods.
     * @param delimiter new delimiter
     **/
    public void setDelim(String delimiter)
    {
        require_(delimiter!=null,"setDelim- nonzro delim");
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_delim = delimiter;
        edited();
    }


    /**
     * Returns the delimiter this strings list will use for
     * any stringification(TM) of its contents. Defaults
     * to a comma. Never returns <i>null</i>.
     * @see AntX#DEFAULT_DELIMITER
     **/
    public String getDelim()
    {
        if (isReference()) {
            return getOtherList().getDelim();
        }
        return m_delim;
    }


    /**
     * Sets a prefix that will be prepended to all strings
     * returned by this type.
     * @param prefix the prefix (non-null)
     * @throws BuildException if this item is a reference
     **/
    public void setPrefix(String prefix)
    {
        setPrefixString(prefix);
    }


    /**
     * Returns this string list's prefix or the empty
     * string if never set. Never returns <i>null</i>.
     **/
    public String getPrefix()
    {
        if (isReference()) {
            return getOtherList().getPrefix();
        }
        return getPrefixString();
    }


    /**
     * Set a suffix that will be appended to all strings
     * returned by this type.
     * @param suffix the suffix (non-null)
     * @throws BuildException if this item is a reference
     **/
    public void setSuffix(String suffix)
    {
        setSuffixString(suffix);
    }


    /**
     * Returns this string list's suffix or the empty
     * string if never set. Never returns <i>null</i>.
     **/
    public String getSuffix()
    {
        if (isReference()) {
            return getOtherList().getSuffix();
        }
        return getSuffixString();
    }


    /**
     * Appends the contents of a newline-delimited file to
     * this string list. No-op if file is missing or empty.
     * @param f the file (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setFile(File f)
    {
        require_(f!=null,"setfil- nonzro fil");
        addFileOrURL(f.getPath());
    }


    /**
     * Appends the contents of a newline-delimited resource
     * file to this string list. No-op if file is missing or
     * empty.
     * @param rn the resource path (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setResource(String rn)
    {
        require_(rn!=null,"setRez- nonzro reznam");
        addResource(rn);
    }


    /**
     * Appends given string (with property substitution) to
     * this string list.
     * @param string new string
     **/
    public void addConfiguredString(InnerString string)
    {
        require_(string!=null,"addStr- nonzro itm");
        addItem(string,getProject());
    }



    /**
     * Appends (in place) the strings referred to be the 
     * given handle. The strings are immediately inserted!
     * @param listH reference to another list-friendly data object.
     * @since JWare/AntX 0.5
     **/
    public void addConfiguredStrings(StringItemListHandle listH)
    {
        require_(listH!=null,"addStrings- nonzro listref");
        if (isReference()) {
            throw tooManyAttributes();
        }        
        Iterator itr = listH.readonlyStringIterator(getProject());
        while (itr.hasNext()) {
            String value = itr.next().toString();
            addItemFinal(value);
            edited();
        }
    }



    /**
     * Returns a comma-delimited list of this list's contents.
     **/
    public String stringFrom(Project theProject)
    {
        if (isReference()) {
            return getOtherList().stringFrom(theProject);
        }
        final String theDelimiter = getDelim();
        StringBuffer sb = new StringBuffer(150);

        int N=0;
        Iterator itr= readonlyStringIterator(theProject);
        while (itr.hasNext()) {
            if (N>0) {
                sb.append(theDelimiter);
            }
            sb.append(itr.next());
            N++;
        }
        itr=null;
        return sb.substring(0);
    }


    /**
     * Synonym for {@linkplain #stringFrom stringFrom()}.
     **/
    public String toString()
    {
        return stringFrom(getProject());
    }


    private final StringList getOtherList()
    {
        return (StringList)getOtherItemList(StringList.class);
    }

    private String m_delim = AntX.DEFAULT_DELIMITER;
}

/* end-of-StringList.java */
