/**
 * $Id: LocalFixtureTaskSet.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2004,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  java.io.File;
import  java.io.IOException;
import  java.net.MalformedURLException;
import  java.net.URL;
import  java.util.Iterator;
import  java.util.List;
import  java.util.Map;

import  org.apache.tools.ant.DynamicAttribute;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.FixtureOverlay;
import  com.idaremedia.antx.apis.Responses;
import  com.idaremedia.antx.helpers.InputFileLoader;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.LocalTk;
import  com.idaremedia.antx.ownhelpers.ScopedProperties;
import  com.idaremedia.antx.parameters.FlexSourceSupport;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * A taskset that overlays a set of localized properties for its nested elements.
 * You can also annotate this block of tasks using the <span class="src">meta</span>
 * attribute so that top-level selectors can examine the meta values w/o having to
 * fully form the usual Ant placeholder <span class="src">UnknownElement</span>.
 * You define local properties using custom attributes which are then installed as
 * supplemental (overriding) project properties for the duration of the taskset's
 * execution (effective for all nested, same-project, tasks). Note that all tasks
 * inherit a standard <span class="src">description</span> attribute so you should
 * not use meta tags as a simple description (just use the standard tag).
 * <p>
 * Because of the way <span class="src">UnknownElement</span> and
 * <span class="src">DynamicConfigurator</span> interact, it is not currently
 * possible to have data types nested inside a local fixture taskset. The
 * "DynamicConfigurator"-ism overrides the "TaskContainer"-isms.
 * <p>
 * This task will automatically convert variable and reference "URLs" to the
 * current value of the named item <em>whenever the property is read</em>. For
 * example using the variable URL "<span class="src">$var:varABC</span>" as a
 * value for a property <span class="src">ABC</span> will cause the variable
 * <span class="src">varABC</span> to be read everytime the property
 * <span class="src">ABC</span> is read.
 * <p>
 * <b>Example Usage:</b><pre>
 *     &lt;overlay category="normalization" todo="bug1234"&gt;
 *         ... <i>[Your regular tasks here]</i>
 *     &lt;/overlay&gt;
 *
 *     &lt;overlay meta="category=documentation,audience=public"&gt;
 *         ... <i>[Your regular tasks here]</i>
 *     &lt;/overlay&gt;
 *
 *     &lt;overlay file="${user.name}/.builds/etl.properties"&gt;
 *         ... <i>[Your regular tasks here]</i>
 *     &lt;/overlay&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.4
 * @author    ssmc, &copy;2004,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5.1
 * @.safety   single
 * @.group    api,helper
 * @.expects  Ant 1.6.5+
 **/

public class LocalFixtureTaskSet extends TaskSet
    implements DynamicAttribute, FixtureOverlay, FlexSourceSupport
{
    /**
     * Initializes a new local fixture taskset. This taskset behaves
     * exactly like a standard AntX taskset unless you install
     * local properties.
     * @see #setDynamicAttribute setDynamicAttribute(&#8230;)
     **/
    public LocalFixtureTaskSet()
    {
        super(AntX.fixture+"Overlay:");
    }



    /**
     * Initialized an embedded or enclosed fixture taskset.
     * @param iam caller's CV-label
     **/
    public LocalFixtureTaskSet(String iam)
    {
        super(iam);
    }



    /**
     * Sets this taskset's primary annotation note text.
     * @param notes the notes (non-null)
     **/
    public final void setMeta(String notes)
    {
       m_metaList = notes==null ? "" : notes;
    }



    /**
     * Returns this taskset's primary annotate note. Never
     * returns <i>null</i> but can return the empty string.
     **/
    public final String getMeta()
    {
        return m_metaList;
    }



    /**
     * Returns a map of name-value pairs represented by
     * this task's <span class="src">meta</span> attribute.
     * Never returns <i>null</i> but can return an
     * empty map.
     **/
    public final Map getMetaMap()
    {
        Map map = AntXFixture.newMap();
        List l = Tk.splitList(getMeta());
        if (!l.isEmpty()) {
            for (Iterator itr=l.iterator();itr.hasNext();) {
                String s = itr.next().toString();
                int i= s.indexOf('=');
                if (i>0) {
                    map.put(s.substring(0,i),s.substring(i+1));
                } else {
                    map.put(s,"");
                }
            }
        }
        l=null;
        return map;
    }



    /**
     * Adds the given attribute to this task's local fixture
     * properties. Will be installed when this task is executed.
     **/
    public final void setDynamicAttribute(String name, String value)
    {
        require_(name!=null,"setAttr- nonzro name");
        getOverridePropertiesNoNull().put(name,value);
    }



    /**
     * Common extraction code for a <span class="src">Properties</span>
     * stored at a URL once the URL has been determined.
     * @since JWare/AntX 0.4
     * @throws BuildException if unable to load definition from URL
     **/
    private void setFromPropertiesURL(URL url, String from)
    {
        Map p=null;
        try {
            p= InputFileLoader.loadProperties(url,null);
        } catch(IOException iox) {
            String warning = uistrs().get("task.bad.configfile",
                                          from,iox.getMessage());
            log(warning,Project.MSG_WARN);
        }
        if (p!=null) {
            getOverridePropertiesNoNull().putAll(p);
            p.clear();
        }
    }



    /* @inherit-doc. Load properties from file.
     */
    public void setFile(String filepath)
    {
        require_(filepath!=null,"setFile- nonzro path");

        File file = getProject().resolveFile(filepath);
        try {
            setFromPropertiesURL(AntXFixture.fileUtils().getFileURL(file),filepath);
        } catch(MalformedURLException mux) {
            String warning = uistrs().get("task.bad.configfile",
                                          filepath,mux.getMessage());
            log(warning,Project.MSG_WARN);
        }
    }



    /* @inherit-doc. Load properties from resource.
     */
    public void setResource(String resource)
    {
        require_(resource!=null,"setRez- nonzro resource name");

        URL url = LocalTk.getSystemResource(resource, getProject());
        if (url!=null) {
            setFromPropertiesURL(url,resource);
        } else {
            String warning = uistrs().get
                ("task.bad.configfile", resource,"File Not Found");
            log(warning,Project.MSG_WARN);
        }
    }


    /* @inherit-doc. Load properties from URL.
     */
    public void setURL(String urlstr)
    {
        require_(urlstr!=null,"setURL- nonzro URL");
        URL url = null;
        try {
            url = new URL(urlstr);
        } catch(MalformedURLException mux) {
            String warning = uistrs().get("task.bad.configfile",
                                          urlstr,mux.getMessage());
            log(warning,Project.MSG_WARN);
        }
        if (url!=null) {
            setFromPropertiesURL(url,urlstr);
        }
    }



    /**
     *  Installs local fixture properties if any exist.
     **/
    private void installLocalFixture()
    {
        verifyInProject_("install");
        if (m_localProperties!=null) {
            m_localProperties.install();
        }
    }



    /**
     * Uninstalls our local fixture properties (if was installed).
     **/
    private void uninstallLocalFixture()
    {
        if (m_localProperties!=null) {
            m_localProperties.uninstall(new Responses.LogUsing(this));
        }
    }




    /**
     * Installs this taskset's custom attributes as project
     * properties while nested tasks are executed. The local
     * properties will be removed before taskset exits (failure
     * or not).
     **/
    protected void performNestedTasks()
    {
        installLocalFixture();
        try {
            super.performNestedTasks();
        } finally {
            uninstallLocalFixture();
        }
    }


    /**
     * Returns this taskset's dynamic attribute name-value pairs.
     * This taskset will reflect any changes to the map. Will
     * return <i>null</i> if no dynamic attributes added to this
     * taskset.
     **/
    protected final Map getOverrideProperties()
    {
        return m_localProperties;
    }



    /**
     * Returns this taskset's dynamic attribute name-value pairs
     * allocating a new map if necessary.
     **/
    protected final Map getOverridePropertiesNoNull()
    {
        if (m_localProperties==null) {
            m_localProperties = new ScopedProperties(getProject(),false);
        }
        return m_localProperties;
    }



    private ScopedProperties m_localProperties;//NB: lazy-inited
    private String m_metaList="";
}

/* end-of-LocalFixtureTaskSet.java */
