/**
 * $Id: ImportTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  java.util.Iterator;
import  java.util.Map;
import  java.util.Properties;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Mapper;
import  org.apache.tools.ant.util.FileNameMapper;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.ExportedProperties;

/**
 * Copies exported variables into a project's properties space. Useful after running
 * sub-project, sub-targets or steps and want to bring generated variables back into the
 * project where standard Ant tasks can get at them. The import task does not overwrite
 * existing properties.
 * <p>
 * <b>Example Usage</b>:<pre>
 *   &lt;assignimport name="my.variable"/&gt;
 *   &lt;assignimport name="global.name" property="local.name"/&gt;
 *   &lt;assignimport name="my.variable" haltifmissing="yes"/&gt;
 *   &lt;assignimport prefix="defaults."/&gt;
 *   &lt;assignimport prefix="defaults." strip="yes"/&gt;
 *
 *   &lt;assignimport prefix="outputs." ismove="yes"&gt;
 *       &lt;mapper type="glob" from="outputs.*" to="defaults.*"/&gt;
 *   &lt;/assignimport&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      ExportTask
 * @see      ExportedProperties
 **/

public final class ImportTask extends AssertableTask
{
    /**
     * Initializes a new ImportTask instance.
     **/
    public ImportTask()
    {
        super(AntX.nopackage);
    }


    /**
     * Initializes a new CV-labeled ImportTask instance.
     * @param iam CV-label (non-null)
     **/
    public ImportTask(String iam)
    {
        super(iam);
    }


    /**
     * Sets the name of the single variable to import. Either this
     * parameter or a prefix must be defined before this task is
     * executed.
     * @param name the variable's name (non-null)
     * @throws BuildException if the 'prefix' parameter has already been defined
     **/
    public void setName(String name)
    {
        require_(name!=null,"setNam- nonzro nam");
        if (getPrefix()!=null) {
            String ermsg = getAntXMsg("import.one.from.attr");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        m_name = name;
    }


    /**
     * Returns the variable name to import. Returns <i>null</i>
     * if never set.
     **/
    public final String getName()
    {
        return m_name;
    }


    /**
     * Setup this task to import all thread-based variables that begin
     * with the given prefix.
     * @throws BuildException if the 'name' parameter has already been defined
     **/
    public void setPrefix(String prefix)
    {
        require_(prefix!=null && prefix.length()>0,"setPfx- nonzro pfx");
        if (getName()!=null) {
            String ermsg = getAntXMsg("import.one.from.attr");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        m_prefix = prefix;
    }


    /**
     * Returns the prefix of all variables to be imported by this task.
     * Will return <i>null</i> if never set.
     **/
    public final String getPrefix()
    {
        return m_prefix;
    }


    /**
     * Tells this task to remove the prefix from imported variables. Only
     * used if a prefix is defined.
     **/
    public void setStrip(boolean remove)
    {
        m_stripPrefix = remove;
    }


    /**
     * Returns <i>true</i> if prefix will be stripped from imported
     * variables. Defaults <i>false</i>.
     **/
    public final boolean willStrip()//oooouuuee...
    {
        return m_stripPrefix;
    }



    /**
     * Returns a new name mapper for this import task ensuring that
     * only one is defined for task.
     * @since JWare/AntX 0.4
     **/
    public Mapper createMapper()
    {
        if (m_nameMapper!=null) {
            String error = uistrs().get("taskset.only.one.specialtask",
                                        "mapper","mapper");
            log(error,Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
        m_nameMapper = new Mapper(getProject());
        return m_nameMapper;
    }



    /**
     * Tells this tasks whether to throw a build exception if named
     * variable doesn't exist within thread's scope. Defaults to
     * <i>false</i>. If a prefix is specified, at least one matching
     * variable must be found.
     * @param stop <i>true</i> if variable must exist.
     **/
    public void setHaltIfMissing(boolean stop)
    {
        m_haltIfNone = stop;
    }


    /**
     * Returns <i>true</i> if this task will stop if it cannot import
     * at least one variable. Defaults <i>false</i>.
     **/
    public final boolean isHaltIfMissing()
    {
        return m_haltIfNone;
    }


    /**
     * Sets the project-local name of the imported variable. If never
     * defined the property is named same as the variable.
     * @param property the local property's name (non-null)
     **/
    public void setProperty(String property)
    {
        require_(property!=null,"setProp- nonzro nam");
        m_updateProperty = property;
    }


    /**
     * Returns the project-local name set for this import task. Will
     * return <i>null</i> if never set (the variable's name will be
     * used).
     **/
    public final String getUpdateProperty()
    {
        return m_updateProperty;
    }


    /**
     * Tells this task to unset source variables once they've been
     * successfully copied to properties. This is a house-cleaning
     * parameter used when steps/sub-targets create variables expressly
     * to pass back to calling target.
     * @param is <i>true</i> if import is really a move
     **/
    public void setIsMove(boolean is)
    {
        m_likeMove = is;
    }


    /**
     * Returns <i>true</i> if this import should function like a
     * move operation. Defaults to <i>false</i>; the originating
     * variable is not removed after the copy operation.
     **/
    public final boolean isMove()
    {
        return m_likeMove;
    }


    /**
     * Imports one or all specified variables into this task's project's
     * properties. If the property already exists, it is not overwritten.
     * @throws BuildException if incomplete definition or variable not
     *         defined and 'haltIfMissing' flag is set <i>true</i>
     **/
    public void execute() throws BuildException
    {
        verifyCanExecute_("execute");

        Project P= getProject();
        int N=0;
        boolean zapVar= isMove();

        if (getPrefix()!=null) {
            final String pfx = getPrefix();
            final boolean strip = willStrip();
            final int PFXLEN= pfx.length();

            FileNameMapper mapper= null;
            if (m_nameMapper!=null) {
                mapper = m_nameMapper.getImplementation();
            }

            Properties allP = ExportedProperties.copy(null);
            Iterator itr= allP.entrySet().iterator();

            log("Import trying to convert properties starting with '"+pfx+
                "' to project; "+allP.size()+" candidates", Project.MSG_DEBUG);

            while (itr.hasNext()) {
                Map.Entry mE = (Map.Entry)itr.next();
                String key = mE.getKey().toString();
                if (key.startsWith(pfx)) {
                    if (strip) {
                        key = key.substring(PFXLEN);
                        verify_(key.length()>0,"import valid key aftr strip");
                    }
                    if (mapper!=null) {
                        String[] rslt = mapper.mapFileName(key);
                        if (rslt!=null) {
                            key = rslt[0];
                        }
                    }
                    P.setNewProperty(key, mE.getValue().toString());
                    N++;
                    if (zapVar) {
                        ExportedProperties.unset(mE.getKey().toString());
                    }
                }
            }
            log("Import tried to suck in ("+N+") variables into project",
                Project.MSG_DEBUG);
            allP.clear();

        } else {
            String varkey = getName();
            String value = ExportedProperties.readstring(varkey);
            if (value!=null) {
                String property = getUpdateProperty();
                if (property==null) {
                    property = varkey;
                }
                if (m_nameMapper!=null) {
                    FileNameMapper mapper = m_nameMapper.getImplementation();
                    String[] rslt = mapper.mapFileName(property);
                    if (rslt!=null) {
                        property = rslt[0];
                    }
                }
                log("Import trying to convert variable '"+varkey+
                    "' to property '"+property+"'",Project.MSG_DEBUG);

                P.setNewProperty(property, value);
                N++;
                if (zapVar) {
                    ExportedProperties.unset(varkey);
                }
            }
        }

        if (N==0 && isHaltIfMissing()) {
            String ermsg;
            if (getMsgId()!=null) {
                ermsg = getMsg();
            } else {
                ermsg= getAntXMsg("import.none",getName());
            }
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }



    /**
     * Verifies that this task is in a valid project and has either the
     * variable's (single) name or a prefix defined.
     * @throws BuildException if not in project or neither the prefix nor
     *         the name has been defined
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (getPrefix()==null && getName()==null) {
            String ermsg = getAntXMsg("task.needs.this.attr",getTaskName(),"name|prefix");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg, getLocation());
        }
    }


    private boolean m_haltIfNone;//NB:nope,keep goin'
    private String  m_prefix;
    private String  m_name;
    private String  m_updateProperty;
    private boolean m_stripPrefix, m_likeMove;
    private Mapper  m_nameMapper;
}

/* end-of-ImportTask.java */
