/**
 * $Id: PrinterRegistryContext.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.print;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.FixtureAdministrator;
import  com.idaremedia.antx.FixtureOverlays;
import  com.idaremedia.antx.FixtureIds;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.apis.ProblemHandler;

/**
 * Manages <nobr>{@linkplain FixtureOverlays FixtureOverlays}-based</nobr> PrinterRegistrys 
 * for the active thread. PrintTasks and friends should use this class instead of directly
 * accessing FixtureOverlays stacks. Unless {@linkplain ManagePrintersTask explicitly
 * initialized}, the default printer registry is an empty (readonly) registry.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,infra
 * @see      ManagePrintersTask
 **/

public final class PrinterRegistryContext implements FixtureAdministrator
{
    /**
     * The iteration category for all PrinterRegistryContext's information.
     **/
    public static final String FXID= FixtureIds.PRINTER_REGISTRY;



    /**
     * Returns the current thread's PrinterRegistry instance if one exists.
     * Returns <i>null</i> if no registry installed for executing thread.
     **/
    public static PrinterRegistry getPrinterRegistry()
    {
        return (PrinterRegistry)FixtureOverlays.getContextInstance().nearest(FXID);
    }



    /**
     * Helper to {@linkplain #getPrinterRegistry getPrinterRegistry} that returns
     * the {@linkplain #getDefaultPrinterRegistry default printer registry} if the
     * current thread has no explicit configuration. Never returns <i>null</i>.
     **/
    public static PrinterRegistry getPrinterRegistryNoNull()
    {
        PrinterRegistry pr = getPrinterRegistry();
        return (pr!=null) ? pr : getDefaultPrinterRegistry();
    }



    /**
     * Installs a new PrinterRegistry instance for the current thread. This
     * printer registry becomes the active (and only visible) printer registry
     * until it is unwound or another printer registry is installed.
     * @param pr the new printer registry (non-null)
     * @param noInstallHandler [optional] callback if cannot install registry
     * @return previous string manager if any (can be null)
     * @throws BuildException if incoming configuration already on iteration stack
     **/
    public static PrinterRegistry installPrinterRegistry(PrinterRegistry pr,
                                                         ProblemHandler noInstallHandler)
    {
        if (pr==null) {
            throw new IllegalArgumentException
                ("PrinterRegistryContext- Cannot install NULL");
        }

        String whoami = AntX.uistrs().dget("task.print.whoami","PrinterRegistry");
        return (PrinterRegistry)FixtureOverlays.installIfNot
                                    (FXID, pr, noInstallHandler, whoami);
    }



    /**
     * Removes the most recently installed PrinterRegistry for the current
     * thread. The previous installation is reactivated, or if this was
     * the only printer registry, the current thread's printer registry is
     * undefined (should use default if necessary).
     * @param noUninstallHandler [optional] handler used to notify in event of
     *                   already unwound stack
     **/
    public static void unwindPrinterRegistry(ProblemHandler noUninstallHandler)
    {
        String whoami = AntX.uistrs().dget("task.print.whoami","PrinterRegistry");
        FixtureOverlays.uninstallIfIs(FXID,noUninstallHandler,whoami);
    }



    /**
     * Returns the default PrinterRegistry used by this context. Never returns
     * <i>null</i>.
     * @see #setDefaultPrinterRegistry
     * @see #getEmptyPrinterRegistry
     **/
    public static PrinterRegistry getDefaultPrinterRegistry()
    {
        PrinterRegistry pr = (PrinterRegistry)Iteration.getProperty(FXID);
        return (pr==null) ? sm_EmptyRegistry : pr;
    }



    /**
     * Initializes the default PrinterRegistry returned by this context. Note
     * that your application does have to call this method to setup an empty
     * default registry. Contexts will automatically us an empty registry if
     * no default registry was installed. The new registry should not
     * be set to a printer registry that already exists on a thread's stack.
     * This method <em>replaces</em> any pre-existing default registry-- the
     * current setting is lost unless the caller maintains it.
     * @param pr new default registry
     * @return existing default registry or <i>null</i> if none set
     * @.safety guarded
     **/
    public static PrinterRegistry setDefaultPrinterRegistry(PrinterRegistry pr)
    {
        if (pr==null) {
            throw new IllegalArgumentException
                (AntX.uistrs().get("print.null.registry"));
        }
        return (PrinterRegistry)Iteration.setProperty(FXID,pr);
    }



    /**
     * Resets the default PrinterRegistry to the empty placeholder.
     * @.safety guarded
     * @since JWare/AntX 0.5
     **/
    public static final PrinterRegistry unsetDefaultPrinterRegistry()
    {
        return (PrinterRegistry)Iteration.removeProperty(FXID);
    }



    /**
     * Returns this context's immutable empty printer registry. Never
     * returns <i>null</i>.
     **/
    public static PrinterRegistry getEmptyPrinterRegistry()
    {
        return sm_EmptyRegistry;
    }



    /** Disallow; only static helpers. **/
    private PrinterRegistryContext()
    { }


    /** Controlling the default printer registry. **/
    private static final PrinterRegistry sm_EmptyRegistry = new PrinterRegistry(true);
}

/* end-of-PrinterRegistryContext.java */
