/**
 * $Id: PrinterRegistry.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2002-2005,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.print;

import  java.io.File;
import  java.util.Collection;
import  java.util.Collections;
import  java.util.Iterator;
import  java.util.Vector;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.DirectoryScanner;
import  org.apache.tools.ant.types.FileSet;
import  org.apache.tools.ant.types.Reference;
import  org.apache.tools.ant.util.FileUtils;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableDataType;
import  com.idaremedia.antx.FixtureComponent;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.parameters.FreeFormEnabled;
import  com.idaremedia.antx.parameters.RecoveryEnabled;

/**
 * A collection of {@linkplain com.idaremedia.antx.print.DisplayStrategy display
 * strategy} to object class mappings. Mappings are searched (for matches) in the
 * order they are inserted. Build scripts don't have to define an explicit default
 * print strategy for every registry; a {@linkplain AnyPrinter default} is always
 * defined. However, an explicit default strategy must be specified if the
 * builder wants to use a custom
 * {@linkplain com.idaremedia.antx.print.DisplayStrategy display strategy}
 * implementation as the registry's default.
 * <p>
 * PrinterRegistries are useful for clumping all printer mapping under a single collection
 * and then installing them all at once using a &lt;{@linkplain ManagePrintersTask
 * manageprinters}&gt; declaration. As of Ant 1.6 registries will accept your custom
 * printer mappings by typedef'd name so you no longer need to use default &lt;printer&gt;
 * shell declaration.
 * <p>
 * <b>Examples:</b><pre>
 *   &lt;typedef name="myprinter" classname="mycompany.MyPrinter"/&gt;
 *   ...
 *   &lt;printer-registry id="jware.antx.printers"&gt;
 *      &lt;defaultprinter id="jware.antx.printer.DEFAULT"/&gt;
 *      &lt;printer id="bundle.printer" classname="jware.antx.init.BundlePrinter"&gt;
 *          &lt;forclass name="jware.antx.strings.UISMBundle"/&gt;
 *      &lt;/printer&gt;
 *      &lt;printer id="errors.printer" classname="jware.antx.print.ErrorPrinter"&gt;
 *          &lt;forclass name="jware.antx.ErrorSnapshot"/&gt;
 *          &lt;forclass name="org.apache.tools.ant.BuildException"/&gt;
 *      &lt;/printer&gt;
 *      &lt;printer id="rules.printer" classname="jware.antx.print.MappingPrinter"&gt;
 *          &lt;forclass name="jware.antx.print.PrinterMapping"/&gt;
 *      &lt;/printer&gt;
 *      <i>&lt;-- Ant 1.6 or later only --&gt</i>
 *      &lt;myprinter id="mytypes.printer"/&gt;
 *   &lt;/printer-registry&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2005,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5.1
 * @.safety  multiple (once <em>fully</em> configured)
 * @.group   api,helper
 * @see      PrinterMapping
 **/

public class PrinterRegistry extends AssertableDataType
    implements Cloneable, FixtureComponent, FreeFormEnabled, AntLibFriendly, RecoveryEnabled
{
    /**
     * Initializes a new printer registry instance.
     **/
    public PrinterRegistry()
    {
        super(AntX.print);
        m_registry = new Vector(10,7);
    }


    /**
     * Initializes a new optionally locked printer registry.
     **/
    PrinterRegistry(boolean locked)
    {
        super(AntX.print);
        if (locked) {
            m_registry = Collections.EMPTY_LIST;
        } else {
            m_registry = new Vector(10,7);
        }
    }


    /**
     * Returns a clone of this printer registry. Supports copying
     * registries between parent and child projects.
     * @since JWare/AntX 0.3
     **/
    public Object clone()
    {
        if (isReference()) {
            return getRegistryRef().clone();
        }
        try {
            PrinterRegistry copy = (PrinterRegistry)super.clone();
            if (m_registry!=Collections.EMPTY_LIST) {
                Vector mappings = (Vector)((Vector)m_registry).clone();
                for (int i=0,n=mappings.size();i<n;i++) {
                    mappings.set(i, ((PrinterMapping)mappings.get(i)).clone());
                }
                copy.m_registry = mappings;
            }
            return copy;
        } catch(CloneNotSupportedException clnx) {
            throw new Error(uistrs().get(AntX.CLONE_BROKEN_MSGID));
        }
    }


    /**
     * Capture our identifier for feedback since types don't always
     * get correct location information.
     **/
    public void setId(String id)
    {
        m_Id= id;
    }


    /**
     * Returns a unique identifier for this registry.
     **/
    public final String getId()
    {
        if (m_Id!=null) {
            return m_Id;
        }
        if (isReference()) {
            return getRegistryRef().getId();
        }
        return super.getId();
    }


    /**
     * Tells this registry whether to propagate a build error if
     * unable to load a file-based printer mapping.
     * @param b <i>false</i> to ignored missing and malformed files
     * @since JWare/AntX 0.4
     * @see #addConfiguredFileSet
     **/
    public void setHaltIfError(boolean b)
    {
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_haltIfError = b;
    }


    /**
     * Returns <i>true</i> if this registry will generate a build
     * error if unable to load a printer mapping from a file.
     * @since JWare/AntX 0.4
     **/
    public boolean isHaltIfError()
    {
        if (isReference()) {
            return getRegistryRef().isHaltIfError();
        }
        return m_haltIfError;
    }


    /**
     * Adds a set of mappings to this registry via a standard Ant
     * fileset. Each file in the given file set must point to a
     * printer mapping definition <span class="src">Properties</span>
     * file. A build exception is thrown if any file is malformed
     * unless the {@linkplain #setHaltIfError} option is turned off.
     * @param fs configured fileset (non-null)
     * @since JWare/AntX 0.4
     * @throws BuildException if any file is missing or malformed
     **/
    public void addConfiguredFileSet(FileSet fs)
    {
        require_(fs!=null,"addFileset- nonzro fileset");
        if (isReference()) {
            throw tooManyAttributes();
        }

        DirectoryScanner dirscan = fs.getDirectoryScanner(getProject());
        String[] files = dirscan.getIncludedFiles();

        if (files.length>0) {
            FileUtils fileUtils = FileUtils.getFileUtils();
            File wrt = dirscan.getBasedir();
            int Nadded=0;
            for (int i=0;i<files.length;i++) {
                File rf= fileUtils.resolveFile(wrt,files[i]);
                try {
                    PrinterMapping pm = new PrinterMapping();
                    pm.setProjectFile(rf);
                    m_registry.add(pm);
                    Nadded++;
                } catch(BuildException bX) {
                    if (isHaltIfError()) {
                        throw bX;
                    }
                }
            }//for

            if (Nadded>0) {
                edited("addFileSet");
            }
            fileUtils= null;
        }

        dirscan= null;
    }


    /**
     * Shorthand version of {@linkplain #addConfiguredFileSet
     * addConfiguredFileSet}.
     * @param r reference id for fileset
     * @since JWare/AntX 0.4
     **/
    public final void setFileSetId(Reference r)
    {
        require_(r!=null,"setFileset- nonzro ref");
        if (isReference()) {
            throw tooManyAttributes();
        }
        FileSet fs = new FileSet();
        fs.setProject(getProject());
        fs.setRefid(r);
        addConfiguredFileSet(fs);
    }



    /**
     * Adds a new mapping to this registry. The inserted mapping and
     * this registry's ClassLoader must be compatible.
     * @param pm new mapping (non-null)
     * @throws UnsupportedOperationException if this registry is locked
     **/
    public void addConfiguredPrinter(PrinterMapping pm)
    {
        require_(pm!=null,"addMapin- nonzro");
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_registry.add(pm);
        edited("addPrinter");
    }



    /**
     * Adds a new (arbitrary) mapping subclass to this registry. The
     * inserted mapping and this registry's ClassLoader must be
     * compatible.
     * @param pm new mapping (non-null)
     * @.expects Ant 1.6 or later
     * @since JWare/AntX 0.4
     **/
    public final void addConfigured(PrinterMapping pm)
    {
        addConfiguredPrinter(pm);
    }


    /**
     * Adds a new default mapping to this registry. The inserted
     * mapping and this registry's ClassLoader must be compatible.
     * @throws UnsupportedOperationException if this registry is locked
     **/
    public void addConfiguredDefaultPrinter(AnyPrinter ap)
    {
        require_(ap!=null,"addDfltPrntr- nonzro prntr");
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_defaultPrinter = ap;
        edited("addDefaultPrinter");
    }


    /**
     * Adds a new (arbitrary) default mapping to this registry.
     * The inserted mapping and this registry's ClassLoader must be
     * compatible.
     * @param ap new default mapping (non-null)
     * @.expects Ant 1.6 or later
     * @since JWare/AntX 0.4
     **/
    public final void addConfigured(AnyPrinter ap)
    {
        addConfiguredDefaultPrinter(ap);
    }


    /**
     * Returns this registry's default printer. Never returns <i>null</i>
     * @see AnyPrinter
     **/
    public DisplayStrategy getDefaultPrinter()
    {
        if (isReference()) {
            return getRegistryRef().getDefaultPrinter();
        }
        return m_defaultPrinter;
    }


    /**
     * Returns the first mapping that can display the given object.
     * Returns <i>null</i> if unable to find a suitable mapping.
     * @param instance object to be displayed (can be <i>null</i>)
     **/
    public PrinterMapping findMappingFor(Object instance)
    {
        if (isReference()) {
            return getRegistryRef().findMappingFor(instance);
        }
        if (instance!=null) {
            synchronized(m_registry) {//this doesn't happen often enuf to copy here
                if (!m_registry.isEmpty()) {
                    Iterator itr= m_registry.iterator();
                    while (itr.hasNext()) {
                        PrinterMapping pm = (PrinterMapping)itr.next();
                        if (pm.isMatch(instance)) {
                            return pm;
                        }
                    }
                    itr=null;
                }
            }//lock
        }
        return null;
    }


    /**
     * Returns a best-fit display stratgy for the given object.
     * If a this registry doesn't contain an explicit mapping,
     * this method returns this registry's default strategy.
     **/
    public final DisplayStrategy getPrinterForNoNull(Object instance)
    {
        PrinterMapping pm = findMappingFor(instance);
        return (pm!=null) ? pm.getPrinter() : getDefaultPrinter();
    }


    /**
     * Returns this registry's reference strongly typed.
     **/
    protected final PrinterRegistry getRegistryRef()
    {
        return (PrinterRegistry)getCheckedRef(PrinterRegistry.class,"printer-registry");
    }


    private String m_Id;
    private Collection m_registry;//NB:cannot be final to support 'clone' (ssmc)
    private AnyPrinter m_defaultPrinter = new AnyPrinter();
    private boolean m_haltIfError = true; //NB:barfs by default
}

/* end-of-PrinterRegistry.java */
