/**
 * $Id: PrinterMapping.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.print;

import  java.io.File;
import  java.io.IOException;
import  java.net.MalformedURLException;
import  java.net.URL;
import  java.util.Iterator;
import  java.util.List;
import  java.util.HashSet;
import  java.util.Properties;
import  java.util.Set;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableDataType;
import  com.idaremedia.antx.FixtureComponent;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.helpers.InputFileLoader;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.LocalTk;
import  com.idaremedia.antx.parameters.FlexSourceSupport;

/**
 * Mapping between one or more object classes and a custom
 * {@linkplain com.idaremedia.antx.print.DisplayStrategy display strategy}
 * implementation. Usually defined as &lt;printer&gt;. An incomplete
 * printer mapping is either missing a strategy implementation or has no target
 * classnames defined (at least one must be defined). Only useful as part of a
 * {@linkplain PrinterRegistry} or referenced from a &lt;{@linkplain PrintTask print}&gt;
 * task.
 * <p>
 * <b>Examples:</b><pre>
 *   &lt;printer id="my.printer" classname="mycompany.ant.MyPrinterClass"&gt;
 *      &lt;forclass name="mycompany.ant.types.MyType"/&gt;
 *      &lt;forclass name="mycompany.ant.types.MyOtherType"/&gt;
 *   &lt;/printer&gt;
 *
 *   &lt;printer id="my.printer" resource="builds/myprinters.properties"/&gt;
 *
 *   &lt;printer id="my.printer" classname="mycompany.ant.MyPrinterClass"&gt;
 *      &lt;fortype name="mytype"/&gt; <i>&lt;!-- use typedef names --&gt;</i>
 *      &lt;fortype name="myothertype"/&gt;
 *   &lt;/printer&gt;
 *
 *   &lt;printer id="my.printer" property="myprinterclass"&gt;
 *      &lt;fortype name="mytype"/&gt; <i>&lt;!-- use typedef names --&gt;</i>
 *      &lt;fortype name="myothertype"/&gt;
 *   &lt;/printer&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple (once <em>fully</em> configured)
 * @.group   api,helper
 * @see      PrinterRegistry
 * @see      PrintTask
 **/

public class PrinterMapping extends AssertableDataType
    implements Cloneable, FixtureComponent, FlexSourceSupport, AntLibFriendly
{
    /**
     * Property key that holds the mapping's display strategy class
     * name when a <span class="src">Properties</span> resource is used
     * to initialize a mapping. Defaults to <span class="src">{@value}</span>.
     **/
    public static final String RESOURCE_KEY= "CLASSNAME";



    /**
     * Initializes a new printer mapping instance.
     **/
    public PrinterMapping()
    {
        super(AntX.print);
    }


    /**
     * Returns a clone of this printer mapping. Supports copying
     * of printer mappings between parent and forked child projects.
     * @since JWare/AntX 0.3
     **/
    public Object clone()
    {
        if (isReference()) {
            return getPrinterRef().clone();
        }
        try {
            PrinterMapping copy = (PrinterMapping)super.clone();
            copy.m_targetClasses = (HashSet)m_targetClasses.clone();
            return copy;
        } catch(CloneNotSupportedException clnx) {
            throw new Error(uistrs().get(AntX.CLONE_BROKEN_MSGID));
        }
    }


    /**
     * Capture our identifier for feedback since types don't always
     * get correct location information.
     **/
    public void setId(String id)
    {
        m_Id= id;
    }


    /**
     * Return a unique identifier for this mapping.
     **/
    public final String getId()
    {
        if (m_Id!=null) {
            return m_Id;
        }
        if (isReference()) {
            return getPrinterRef().getId();
        }
        return super.getId();
    }


    /**
     * Returns <i>true</i> if this mapping if fully defined. A mapping
     * requires at least its display strategy and one filter class to
     * be defined before it can be used.
     * @see #addConfiguredForClass addConfiguredForClass(...)
     * @see #setClassName
     **/
    public boolean isFullyDefined()
    {
        if (isReference()) {
            return getPrinterRef().isFullyDefined();
        }
        return m_printWorker!=null && !m_targetClasses.isEmpty();
    }



    /**
     * Sets this mappings display strategy implementation class. Class must
     * implement the {@linkplain com.idaremedia.antx.print.DisplayStrategy
     * display strategy} interface.
     * This parameter is required before this mapping can be used.
     * @param clas the printer's implementation class (non-null)
     * @throws BuildException if unable to create strategy instance from class
     **/
    public void setClassName(Class clas)
    {
        require_(clas!=null,"setClaz- nonzro clsid");
        if (isReference()) {
            throw tooManyAttributes();
        }
        try {
            m_printWorker = (DisplayStrategy)clas.newInstance();
        } catch(Exception anyX) {
            String ermsg = uistrs().get("printer.bad.impl.class",
                                        getId(), clas.getName());
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,anyX);
        }
        edited("setClassName");
    }


    /**
     * Sets this mappings display strategy to the class referred to by
     * the given property.
     * @since JWare/AntX 0.4
     **/
    public void setProperty(String property)
    {
        require_(property!=null,"setProp- nonzro prop nam");
        if (isReference()) {
            throw tooManyAttributes();
        }
        String classname = getProject().getProperty(property);
        if (classname!=null) {
            try {
                setClassName(Class.forName(classname));
            } catch(Exception anyX) {
                String ermsg = uistrs().get("printer.bad.property",
                                            getId(), property, classname);
                log(ermsg,Project.MSG_ERR);
                throw new BuildException(ermsg,anyX);
            }
        } else {
            String ermsg = uistrs().get("task.undef.property",property);
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg);
        }
    }



    /**
     * Returns the name of this mapping's display strategy implementation.
     * Returns <i>null</i> if no strategy implementation defined.
     * @see #setClassName
     **/
    public final String getPrinterClassName()
    {
        if (isReference()) {
            return getPrinterRef().getPrinterClassName();
        }
        return m_printWorker!=null ?
            m_printWorker.getClass().getName() :
            null;
    }



    /**
     * Adds a new class that this mapping's printer can handle.
     * @param clsinfo the name of the class (embedded in a &lt;forclass&gt; element)
     * @throws BuildException if unable to load object class from name
     **/
    public void addConfiguredForClass(InnerString clsinfo)
    {
        require_(clsinfo!=null,"addForClaz- nonzro valu");
        if (isReference()) {
            throw tooManyAttributes();
        }
        String classname = clsinfo.getText();
        try {
            addTargetClass(Class.forName(classname));
        } catch(Exception anyX) {
            String ermsg = uistrs().get("printer.bad.item.class",
                                        getId(), classname);
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,anyX);
        }
        edited("addForClass");
    }



    /**
     * Adds a new typedef'ed class that this mapping's printer can
     * handle. The type must already be registered.
     * @since JWare/AntX 0.4
     **/
    public void addConfiguredForType(InnerString typedefItem)
    {
        require_(typedefItem!=null,"addForTyp- nonzro typdef");
        if (isReference()) {
            throw tooManyAttributes();
        }
        String defname = typedefItem.getText();
        Class claz = (Class)getProject().getDataTypeDefinitions().get(defname);
        if (claz==null) {
            String ermsg = uistrs().get("printer.bad.typedef", getId(), defname);
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg);
        }
        addTargetClass(claz);
        edited("addForTypedef");
    }


    /**
     * Common extraction code for a <span class="src">Properties</span>
     * stored at a URL once the URL has been determined.
     * @since JWare/AntX 0.4
     * @throws BuildException if unable to load definition from URL
     **/
    private void setFromPropertiesURL(URL url, String from)
    {
        Properties p;
        try {
            p= InputFileLoader.loadProperties(url,null);
        } catch(IOException iox) {
            String error = uistrs().get("printer.bad.source",getId(),from);
            log(error,Project.MSG_ERR);
            throw new BuildException(error,iox);
        }
        String clsid = p.getProperty(RESOURCE_KEY);
        if (clsid!=null) {
            try {
                setClassName(Class.forName(clsid));
            } catch (Exception anyX) {
                String error = uistrs().get("printer.bad.impl.class", getId(),clsid);
                log(error,Project.MSG_ERR);
                throw new BuildException(error,anyX);
            }
            p.remove(RESOURCE_KEY);
            if (!p.isEmpty()) {
                Iterator itr= p.values().iterator();
                while (itr.hasNext()) {
                    String classname = itr.next().toString();
                    try {
                        addTargetClass(Class.forName(classname));
                    } catch(Exception anyX) {
                        String error = uistrs().get("printer.bad.item.class",
                                                    getId(), classname);
                        log(error,Project.MSG_ERR);
                        throw new BuildException(error,anyX);
                    }
                }//while
            }
        } else {
            String error = uistrs().get("printer.bad.source.class",getId(),from);
            log(error,Project.MSG_ERR);
            throw new BuildException(error);
        }
        p.clear();
        p=null;
    }


    /**
     * Completely defines this printer mapping using the contents of
     * a single resource (<span class="src">Properties</span>) file. The
     * display strategy class name must be stored under the
     * "<span class="src">CLASSNAME</span>" entry. All other entries
     * are assumed to contain target class names as values (the keys
     * are ignored). Properly handles the output of the standard
     * {@linkplain MappingPrinter Mapping printer}.
     * @throws BuildException if unable to locate/load display class
     * @throws BuildException if unable to load any target class
     * @since JWare/AntX 0.4
     **/
    public void setResource(String rn)
    {
        require_(rn!=null,"setrez- nonzro nam");
        if (isReference()) {
            throw tooManyAttributes();
        }
        URL url = LocalTk.getSystemResource(rn, getProject());
        if (url!=null) {
            setFromPropertiesURL(url,rn);
        } else {
            String warning = uistrs().get("printer.bad.source",getId(),rn);
            log(warning,Project.MSG_WARN);
        }
    }



    /**
     * Like {@linkplain #setResource setResource} except the Properties
     * file is defined with a <span class="src">File</span> object.
     * @param filepath the (Properties) file to load (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setFile(String filepath)
    {
        require_(filepath!=null,"setfile- nonzro filepath");
        if (isReference()) {
            throw tooManyAttributes();
        }
        try {
            File file = getProject().resolveFile(filepath);
            setFromPropertiesURL(AntXFixture.fileUtils().getFileURL(file),filepath);
        } catch(MalformedURLException mux) {
            String warning = uistrs().get("printer.bad.source",getId(),filepath);
            log(warning,Project.MSG_WARN);
        }
    }



    /**
     * Like {@linkplain #setFile(String) setFile(path)} but with a
     * resolved file object.
     * @param file the (Properties) file to load (non-null)
     * @since JWare/AntX 0.4
     **/
    public final void setProjectFile(File file)
    {
        require_(file!=null,"setFile- nonzro file");
        if (isReference()) {
            throw tooManyAttributes();
        }
        try {
            setFromPropertiesURL(AntXFixture.fileUtils().getFileURL(file),file.getPath());
        } catch(MalformedURLException mux) {
            String warning = uistrs().get("printer.bad.source",getId(),file.getPath());
            log(warning,Project.MSG_WARN);
        }
    }



    /**
     * Like {@linkplain #setResource setResource} except the Properties
     * file is defined with a <span class="src">URL</span> string.
     * @param urlstr the (Properties) file to load (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setURL(String urlstr)
    {
        require_(urlstr!=null,"setURL- nonzro URL");
        if (isReference()) {
            throw tooManyAttributes();
        }
        URL url = null;
        try {
            url = new URL(urlstr);
        } catch(MalformedURLException mux) {/*fallthru*/}
        if (url!=null) {
            setFromPropertiesURL(url,urlstr);
        } else {
            String warning = uistrs().get("printer.bad.source",getId(),urlstr);
            log(warning,Project.MSG_WARN);
        }
    }



    /**
     * Shortcut for adding a single or a comma-delimited list of
     * classnames to this mapping.
     * @param classlist comma-delimited list of class names (non-null)
     * @see #addConfiguredForClass addConfiguredForClass(...)
     * @throws BuildException if unable to load object class from name
     **/
    public final void setForClass(String classlist)
    {
        require_(classlist!=null,"setforclaz- nonzro list");
        if (isReference()) {
            throw tooManyAttributes();
        }
        List l = Tk.splitList(classlist);
        if (!l.isEmpty()) {
            InnerString helper= new InnerString();
            for (int i=0,N=l.size();i<N;i++) {
                helper.setValue((String)l.get(i));
                addConfiguredForClass(helper);
            }
            helper=null;
        }
        l=null;
    }


    /**
     * Shortcut for adding a single or a comma-delimited list of
     * script-declared type names to this mapping.
     * @param typelist comma-delimited list of typdef'ed names (non-null)
     * @see #addConfiguredForClass addConfiguredForClass(...)
     * @throws BuildException if unable to find typedef match
     * @since JWare/AntX 0.4
     **/
    public final void setForType(String typelist)
    {
        require_(typelist!=null,"setfortype- nonzro list");
        if (isReference()) {
            throw tooManyAttributes();
        }
        List l = Tk.splitList(typelist);
        if (!l.isEmpty()) {
            InnerString helper= new InnerString();
            for (int i=0,N=l.size();i<N;i++) {
                helper.setValue((String)l.get(i));
                addConfiguredForType(helper);
            }
            helper=null;
        }
        l=null;
    }



    /**
     * Returns this mapping's display strategy implementation.
     * @throws BuildException if no strategy defined for this mapping
     **/
    public DisplayStrategy getPrinter()
    {
        if (isReference()) {
            return getPrinterRef().getPrinter();
        }
        if (m_printWorker==null) {
            String ermsg = uistrs().get("printer.undefined",getId());
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg);
        }
        return m_printWorker;
    }



    /**
     * Returns <i>true</i> if this mapping applies to the given
     * object's class. Always returns <i>false</i> if this mapping's
     * definition is incomplete. Never matches to <i>null</i>.
     * @param instance the object to be displayed
     **/
    public boolean isMatch(Object instance)
    {
        if (isReference()) {
            return getPrinterRef().isMatch(instance);
        }
        if (instance!=null && isFullyDefined()) {
            Iterator itr= m_targetClasses.iterator();
            while (itr.hasNext()) {
                Class clas = (Class)itr.next();
                if (clas.isInstance(instance)) {//NB:solves CL issues (ssmc)
                    return true;
                }
            }
        }
        return false;
    }


    /**
     * Returns this mapping's reference strongly typed.
     **/
    protected final PrinterMapping getPrinterRef()
    {
        return (PrinterMapping)getCheckedRef(PrinterMapping.class,"printer");
    }



    /**
     * Returns a live reference to this mappings underlying target
     * classes. Used by display strategy (only). Never returns <i>null</i>.
     * @since JWare/AntX 0.4
     **/
    Set getTargetClasses()
    {
        return m_targetClasses;
    }



    /**
     * Adds handle class reference; emits warning if class already
     * known.
     **/
    private void addTargetClass(Class claz)
    {
        if (!m_targetClasses.add(claz)) {
            String warning = uistrs().get("printer.class.mapping.exists",
                                          getId(), claz.getName());
            log(warning,Project.MSG_WARN);
        }
    }

    private String  m_Id;
    private HashSet m_targetClasses= new HashSet(11);
    private DisplayStrategy m_printWorker;
}


/* end-of-PrinterMapping.java */
