/**
 * $Id: ManagePrintersTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2003,2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.print;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableLibDefinition;
import  com.idaremedia.antx.FixtureInitializer;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Helper task to (un)install a default printer registry. If a specific action is not
 * specified, will default to installing the default printer registry. Usually defined
 * &lt;manageprinters&gt; and used in an early build initialization target; for
 * example:<pre>
 *    &lt;printer-registry id="jware.antx.printers"&gt;
 *        &lt;printer .../&gt;
 *        ...
 *    &lt;/printer-registry&gt;
 *
 *    &lt;target name="--init-all"&gt;
 *      ...
 *      &lt;<b>manageprinters</b> registryid="jware.antx.printers"/&gt;
 *      ...
 *    &lt;/target&gt;
 *
 * -To remove a previously installed PrinterRegistry use either:-
 *
 *    &lt;manageprinters action="uninstall" registryid="..."/&gt;  -OR-
 *    &lt;manageprinters action="uninstall"/&gt;
 * </pre>
 * Only <em>one</em> default printer registry can be active at any one time within
 * the build-iteration. If &lt;manageprinters&gt; is used to install more than one
 * registry, the last registry installed wins; the previously installed registries
 * are overwritten.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2003,2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      PrinterRegistry
 * @see      PrintTask
 **/

public final class ManagePrintersTask extends AssertableLibDefinition
    implements FixtureInitializer
{
    /**
     * Initializes an ManagePrintersTask instance.
     **/
    public ManagePrintersTask()
    {
        super(AntX.print+"ManagePrintersTask:");
    }



    /**
     * Sets the registry which will be made the default by this task.
     **/
    public void setRegistryId(Reference ref)
    {
        require_(ref!=null,"setRegId- nonzro ref");
        m_registryReference = ref;
    }



    /**
     * Instructs this task how to manage the target registry.
     * @param op the operation (one of "install" or "uninstall")
     **/
    public void setAction(String op)
    {
        m_Op = Tk.lowercaseFrom(op);
    }



    /**
     * Tells this tasks whether the default printer registry should be
     * stopped on if it already exists.
     * @param overwrite <i>true</i> to unconditionally replace default registry.
     * @since JWare/AntX 0.5
     **/
    public void setOverwrite(boolean overwrite)
    {
        m_willSkip = overwrite ? Boolean.FALSE : Boolean.TRUE;
    }



    /**
     * Returns this task's registry reference. Will return <i>null</i>
     * if never set. Required for task to execute.
     **/
    public final Reference getRegistryReference()
    {
        return m_registryReference;
    }



    /**
     * Returns the printer registry this task will try to install
     * as the build-iteration default. To be called <em>after</em>
     * this task's configuration has been verified.
     **/
    protected final PrinterRegistry getPrinterRegistry()
    {
        return (PrinterRegistry)getProject().getReference(getRegistryReference().getRefId());
    }



    /**
     * Returns <i>true</i> if is an installation action (default).
     **/
    private boolean isInstall()
    {
        return m_Op==null || Strings.INSTALL.equals(m_Op);
    }



    /**
     * Verifies that this task has been completely defined. Ensures the
     * reference has been specified and exists for an installation operation.
     * @throws BuildException if verification fails
     **/
    public void verifyCanExecute_(String calr) throws BuildException
    {
        super.verifyCanExecute_(calr);
        Reference r = getRegistryReference();

        String ermsg = null;
        if (r==null || r.getRefId()==null) {
            if (isInstall()) {
                ermsg = uistrs().get("task.needs.this.attr",getTaskName(),"registryid");
            }
        } else {
            Object object = getProject().getReference(r.getRefId());
            if (!(object instanceof PrinterRegistry)) {
                ermsg = uistrs().get("initprint.bad.registryid",r.getRefId());
            }
        }
        if (ermsg!=null) {
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }



    /**
     * Tries to install or uninstall a {@linkplain PrinterRegistry} as the
     * default registry for all threads within a single build-iteration.
     * @throws BuildException if incompletely defined (no reference)
     **/
    public void execute() throws BuildException
    {
        verifyCanExecute_("execute");

        //Install
        if (isInstall()) {
            PrinterRegistry newpr = getPrinterRegistry();
            PrinterRegistry oldpr = PrinterRegistryContext.setDefaultPrinterRegistry(newpr);

            if (oldpr!=DFLT_PR && oldpr!=null && oldpr!=newpr) {
                String warning = uistrs().get("initprint.err.too.many");
                log(warning,Project.MSG_WARN);
            } else {
                log("Installed default PrinterRegistry with id=\""+
                    getRegistryReference().getRefId()+"\"", Project.MSG_DEBUG);
            }
        }
        //Uninstall
        else if (Strings.UNINSTALL.equals(m_Op)) {
            Reference r = getRegistryReference();

            if (r!=null && r.getRefId()!=null) {
                PrinterRegistry expectedpr = getPrinterRegistry();
                if (PrinterRegistryContext.getDefaultPrinterRegistry()!=expectedpr) {
                    String warning = uistrs().get("initprint.mismatch.installed",r.getRefId());
                    log(warning,Project.MSG_WARN);
                } else {
                    PrinterRegistryContext.unsetDefaultPrinterRegistry();
                    log("Uninstalled previously installed default PrinterRegistry with id=\""+
                        expectedpr.getId()+"\"", Project.MSG_DEBUG);
                }
            } else {
                PrinterRegistry oldpr = PrinterRegistryContext.unsetDefaultPrinterRegistry();
                if (oldpr!=DFLT_PR && oldpr!=null) {
                    log("Uninstalled default PrinterRegistry with id=\""+oldpr.getId()+"\"",
                        Project.MSG_DEBUG);
                }
            }
        }
        //?Unknown action
        else {
            String warning = uistrs().get("task.manager.err.unknown.operation",m_Op);
            log(warning,Project.MSG_WARN);
        }
    }



    /** Default printer registry isn't null it is a null placehoder. **/
    private static final PrinterRegistry DFLT_PR=
        PrinterRegistryContext.getEmptyPrinterRegistry();

    private Reference m_registryReference;
    private String m_Op;//NB: install,uninstall
    private Boolean m_willSkip;
}

/* end-of-ManagePrintersTask.java */
