/**
 * $Id: AnyPrinter.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.print;

import  java.io.BufferedOutputStream;
import  java.io.IOException;
import  java.io.OutputStream;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableDataType;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Basically a printer that spews whatever 'String.valueOf(&#46;&#46;&#46;)' returns.
 * Also acts as a handle to a default printer specified in the build script. Usually
 * defined in the context of a &lt;printer-registry&gt; as the default printer like:<pre>
 *     &lt;printer-registry id="antx.default.printers"&gt;
 *       &lt;defaultprinter id="antx.default.printer"/&gt; <b>&lt;-- ME --&gt;</b>
 *       &lt;printer id="printer.0" .../&gt;
 *        ...
 *     &lt;/printer-registry&gt;
 * -OR-
 *     &lt;printer-registry id="antx.default.printers"&gt;
 *       &lt;defaultprinter id="antx.default.printer"      <b>&lt;-- ME too --&gt;</b>
 *              classname="mycompany.custom.PrinterImpl"/&gt;
 *       &lt;printer id="printer.0" .../&gt;
 *        ...
 *     &lt;/printer-registry&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  n/a
 * @.group   api,helper
 * @see      PrinterRegistry
 **/

public class AnyPrinter extends AssertableDataType implements DisplayStrategy
{
    /**
     * Creates a new AnyPrinter instance.
     **/
    public AnyPrinter()
    {
        super(AntX.print);
    }


    /**
     * Capture our identifier for feedback since types don't always
     * get correct location information.
     **/
    public void setId(String id)
    {
        m_Id= id;
    }


    /**
     * Tries to return an identifier for this printer.
     **/
    public final String getId()
    {
        if (m_Id!=null) {
            return m_Id;
        }
        if (isReference()) {
            return getPrinterRef().getId();
        }
        return super.getId();
    }


    /**
     * Tells this printer to exclude forced newlines between each item
     * written. Newlines are written by default.
     * @param splitEm <i>true</i> to separate messages entries
     * @since JWare/AntX 0.4
     **/
    public void setSplitEntries(boolean splitEm)
    {
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_forceNL = splitEm;
        edited("splitEntries");
    }


    /**
     * Returns <i>true</i> if this task will insert a platform-specific
     * newline sequence between each item printed. Is on by default.
     * @since JWare/AntX 0.4
     **/
    public boolean willSplitEntries()
    {
        if (isReference()) {
            return getPrinterRef().willSplitEntries();
        }
        return m_forceNL;
    }



    /**
     * Force this printer to delegate to another
     * {@linkplain com.idaremedia.antx.print.DisplayStrategy display}
     * implementation.
     * @param classname name of DisplayStrategy class (non-null)
     * @throws BuildException if unable to create a printer instance from classname
     **/
    public void setClassName(String classname)
    {
        require_(classname!=null,"setClaz- nonzro name");
        if (isReference()) {
            throw tooManyAttributes();
        }
        if (!AnyPrinter.class.getName().equals(classname)) {//don't point to self
            try {
                Class strategyClass = Class.forName(classname);
                m_printWorker = (DisplayStrategy)strategyClass.newInstance();

            } catch (Exception anyX) {
                String ermsg = uistrs().get("printer.bad.impl.class",
                                            getId(), classname);
                log(ermsg, Project.MSG_ERR);
                throw new BuildException(ermsg,anyX);
            }
        }
        edited("setClassName");
    }


    /**
     * Returns the name of the display strategy this handle represents.
     * Return's this class's name if this printer doesn't delegate to
     * any other strategy.
     * @see #setClassName
     **/
    public final String getStrategyClassName()
    {
        if (isReference()) {
            return getPrinterRef().getStrategyClassName();
        }
        return m_printWorker!=null ?
            m_printWorker.getClass().getName() :
            AnyPrinter.class.getName();
    }



    /**
     * Convert to-be-displayed <i>thing</i> to a string. Log4J bridge
     * point. Can never return <i>null</i>. If the <i>thing</i> signals
     * an exception from its <span class="src">toString()</span> handler,
     * this method returns a generic string of form:
     * "<span class="src">classname@objectid</span>" where 'classname'
     * is the object's class name and 'objectid' is the object reference
     * identity hash value.
     **/
    public String stringFrom(Object thing)
    {
        if (isReference()) {
            return getPrinterRef().stringFrom(thing);
        }
        return Tk.stringFrom(thing,getProject());
    }


    /**
     * Prints the given <i>thing</i> to the given output stream.
     * @throws IOException if any I/O problems occur
     * @throws BuildException if any other error occurs
     **/
    public void print(DisplayRequest req, OutputStream out)
        throws IOException, BuildException
    {
        if (isReference()) {
            getPrinterRef().print(req,out);

        } else if (m_printWorker!=null) {
            m_printWorker.print(req,out);

        } else {
            String s = stringFrom(req.getObjectToBeDisplayed());
            ensure_(s!=null,"print- strFrm is non-nul");

            BufferedOutputStream bout;
            if (out instanceof BufferedOutputStream) {
                bout = (BufferedOutputStream)out;
            } else {
                bout = new BufferedOutputStream(out,s.length());//ick
            }

            if (willSplitEntries()) {
                bout.write(NL_);
            }

            byte[] raw = s.getBytes();
            bout.write(raw,0,raw.length);
            bout.flush();

            //NB:good-gravy...
            raw = null;
            s = null;
            bout = null;
        }
    }


    /**
     * Returns this printer's reference strongly typed.
     **/
    protected final AnyPrinter getPrinterRef()
    {
        return (AnyPrinter)getCheckedRef(AnyPrinter.class,"defaultprinter");
    }


    private String m_Id;
    private DisplayStrategy m_printWorker;
    private boolean m_forceNL=true;
    private static final byte[] NL_= Strings.NL.getBytes();
}

/* end-of-Printer.java */
