/**
 * $Id: TaskExaminer.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2004,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers;

import  java.util.Enumeration;
import  java.util.Iterator;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.ComponentHelper;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.ProjectComponent;
import  org.apache.tools.ant.ProjectHelper;
import  org.apache.tools.ant.RuntimeConfigurable;
import  org.apache.tools.ant.Target;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.UnknownElement;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.FixtureAdministrator;
import  com.idaremedia.antx.FixtureCore;
import  com.idaremedia.antx.FixtureOverlays;
import  com.idaremedia.antx.FixtureIds;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.TaskHandle;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Utilities to unmask the real tasks references behind an Ant 1&#46;6+
 * "<span class="src">UnknownElement</span>".
 * <p>
 * As of Ant 1.6 it is almost impossible to get task-type information as a
 * <span class="src">TaskSet</span> is being filled in. All tasksets are filled
 * with <span class="src">UnknownElement</span> proxies until the taskset is
 * actually performed. The <span class="src">TaskExaminer</span> utility is for
 * strict tasksets that must limit what gets nested inside them. The global
 * {@linkplain AntX#STRICT_SCRIPTS} option and the more-lenient
 * {@linkplain #candidateTask candidateTask} method can be used
 * by less strict tasksets that can filter nested tasks, but such filtering
 * is not necessary strictly speaking.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5.1
 * @.safety  guarded
 * @.group   impl,infra
 * @.expects Ant 1.6 or later
 **/

public final class TaskExaminer implements FixtureCore, FixtureAdministrator
{
    private static final String IAM_= AntX.utilities+"TaskExaminer:";


    /**
     * The examiner's fixture administrator identifier.
     **/
    public static final String ITID= FixtureIds.TASK_EXAMINER;



    /**
     * The shared grunt work of unmasking a task; called by public APIs.
     **/
    private static Task expose(Task t, ProjectComponent wrt)
    {
        UnknownElement ue = (UnknownElement)t;
        t = ue.getTask();//check if done already!
        if (t==null) {
            FixtureOverlays I= FixtureOverlays.getContextInstance();
            I.install(ITID,Boolean.TRUE);
            try {
                ue.maybeConfigure();
                t = ue.getTask();
                if (t==null || t==ue) {
                    throw new BuildException
                        (AntX.uistrs().get("task.cant.flip.unknown",
                                           ue.getTaskName()));
                }
            } finally {
                FixtureOverlays.uninstallIfIs(ITID,new Requester.ForComponent(wrt),IAM_);
            }
        }
        return t;
    }



    /**
     * Returns <i>true</i> if the task's name matches any of the tasks
     * of interest to caller.
     * @param t the unknown task to be matched (non-null)
     * @param candidates the classes of interest to caller (non-null)
     **/
    private static boolean match(Task t, Class[] candidates)
    {
        Class cc = trueClass(t);//NB:can-be-nul

        if (cc!=null) {
            for (int i=0;i<candidates.length;i++) {
                if (candidates[i].isAssignableFrom(cc)) {
                    return true;
                }
            }
        }
        return false;
    }



    /**
     * Lousy hack to convert a configurable's attribute map to all lowercase
     * (which is how it should've been stored in the first place&#8230;grumble).
     * @param atrs the attribute map (non-null)
     **/
    private static Map normalize(Map atrs)
    {
        Map norm = AntXFixture.newMap();
        Iterator itr = atrs.entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry mE = (Map.Entry)itr.next();
            norm.put(Tk.lowercaseFrom(mE.getKey().toString()),mE.getValue());
        }
        atrs.clear();//eyow! zap the othr copy!
        return norm;
    }



    /**
     * Tries to immediately determine the value of a UE attribute resolving
     * any context-dependent properties.
     * @param P project (non-null)
     * @param string string to resolve (non-null)
     * @return resolved string or original if could not resolve now.
     **/
    private static String tryResolve(final Project P, final String string)
    {
        try {
            return Tk.resolveString(P,string);
        } catch(BuildException tooSoonX) {
            return string;
        }
    }



    /**
     * Returns the class of a placeholder
     * (<span class="src">UnknownElement</span>) item. Will return <i>null</i>
     * if placeholder's target's definition does not yet exist in project.
     * Use this method if you do not <em>need</em> the underlying task
     * immediately. Tasksets can use this method to prevent nested tasks
     * from being configured before they need to be.
     * @param t the task placeholder (non-null)
     * @return class of placeholder (can be <i>null</i> if element undefined)
     * @throws IllegalArgumentException if task is <i>null</i>
     **/
    public static Class trueClass(Task t)
    {
        AntX.require_(t!=null && t.getProject()!=null,IAM_,"trueClas- inited prxy");

        if (t instanceof UnknownElement) {
            UnknownElement ue = (UnknownElement)t;
            ComponentHelper ch = ComponentHelper.getComponentHelper(t.getProject());
            String cn = ProjectHelper.genComponentName(ue.getNamespace(), ue.getTag());
            return ch.getComponentClass(cn);
        }

        return t.getClass();
    }



    /**
     * Given an task placeholder (<span class="src">UnknownElement</span>),
     * tries to convert to the actual task without incurring the overhead
     * of immediately configuring the real task. This configuration delay
     * is important for tasksets that may or may not execute all nested
     * sub-tasks. If the incoming task is not a placeholder, it is
     * returned as-is.
     * @param t the task to be unmasked (non-null)
     * @param wrt the task on whose behalf the "unmasking" is being done
     * @return real task (never <i>null</i>)
     * @throws IllegalArgumentException if either parameter is <i>null</i>
     **/
    public static Task trueTask(Task t, ProjectComponent wrt)
    {
        AntX.require_(t!=null && wrt!=null, IAM_,
                      "trueTask- nonzro args");

        if (t instanceof UnknownElement) {
            t = expose(t,wrt);
        }
        return t;
    }



    /**
     * Like {@linkplain #trueTask(Task,ProjectComponent) trueTask(&#8230;)}
     * but the placeholder's proxied class is first compared against a list
     * of filter classes. If the underlying task class matches one of the
     * filter classes, this method tries to unmask the placeholder. If there
     * is no match or the task is not a placeholder, it is returned as-is.
     * Note that the filter classes should be mutally exclusive (usually
     * leaf classes).
     * @param t the task to be unmasked (non-null)
     * @param coi classes of interest (non-null)
     * @param wrt the task on whose behalf the "unmasking" is being done
     * @return possible real task (never <i>null</i>)
     * @throws IllegalArgumentException if any parameter is <i>null</i>
     **/
    public static Task trueTask(Task t, Class[] coi, ProjectComponent wrt)
    {
        AntX.require_(t!=null && wrt!=null && coi!=null, IAM_,
                      "trueTask- nonzro args");

        if ((t instanceof UnknownElement) && match(t,coi)) {
            t = expose(t,wrt);
        }
        return t;
    }



    /**
     * Tries to extract an attribute (setter) value from a placeholder
     * task. If the given task is not a placeholder, this method returns
     * <i>null</i>. This method will also return <i>null</i> if the
     * attribute is unspecified on placeholder.
     * @param t the placeholder to be examined (non-null)
     * @param attr the attribute to be examined (non-null)
     * @return attribute setting (can be <i>null</i>)
     * @throws IllegalArgumentException if either parameter is <i>null</i>
     **/
    public static String trueAttr(Task t, String attr)
    {
        AntX.require_(t!=null && attr!=null, IAM_,
                      "trueAttr- nonzro args");

        String atrvalue=null;

        if (t instanceof UnknownElement) {
            Map m = ((UnknownElement)t).getWrapper().getAttributeMap();
            Object v = m.get(attr);
            if (v!=null) {
                atrvalue = v.toString();
            } else {
                m = normalize(m);//gag!
                v = m.get(attr);
                if (v!=null) {
                    atrvalue = v.toString();
                }
            }
            m.clear();//gc
            m=null;
        }

        return atrvalue;
    }



    /**
     * Tries to extract a collection of attribute (setter) values
     * from a placeholder task. Each found attribute's value is
     * in the given map. If the incoming task is not a placeholder
     * task, this method leaves the attribute map untouched and
     * returns <i>null</i>.
     * @param t the placeholder to be examined (non-null)
     * @param attrs the attributes to be examined (non-null,writable)
     * @return <i>null</i> if not placeholder or no hits, else
     *         incoming map
     * @throws IllegalArgumentException if either parameter is <i>null</i>
     **/
    public static Map trueAttrs(Task t, Map attrs)
    {
        AntX.require_(t!=null && attrs!=null, IAM_,
                      "trueAttrs- nonzro args");

        if (t instanceof UnknownElement) {
            Map m = ((UnknownElement)t).getWrapper().getAttributeMap();
            m = normalize(m);//gag!
            int N=0;
            Iterator itr= attrs.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry mE = (Map.Entry)itr.next();
                Object v = m.get(mE.getKey());
                if (v!=null) {
                    mE.setValue(v);
                    N++;
                }
            }
            if (N==0) {
                attrs=null;
            }
            m.clear();//gc
            m=null;
            return attrs;
        }
        return null;
    }



    /**
     * Called by AntX tasks being configured to determine whether
     * the request is part of an ongoing "unmask" operation. If the
     * request was triggered by this examiner, the AntX task will
     * ignore the request and wait for another from its enclosing
     * parent.
     * @param t the task being configured (non-null)
     **/
    public static boolean ignoreConfigure(Task t)
    {
        FixtureOverlays context= FixtureOverlays.getContextInstance();
        return ((Boolean)context.nearest(ITID))==Boolean.TRUE;
    }



    /**
     * A TaskSet construction helper that ensures an
     * Ant 1.6+ placeholder task is replaced if AntX's strict script
     * loading property is turned on. If unmasked, the handle will be
     * updated with the true task. If the AntX option is turned off,
     * this method leaves the placeholder task as-is.
     * @param taskH the task to be examined (non-null)
     * @param coi classes of interest (non-null)
     * @param wrt the task on whose behalf the "unmasking" is being done
     * @return the task to examine (might be flipped, might not)
     * @throws IllegalArgumentException if any parameter is <i>null</i>
     * @see AntX#STRICT_SCRIPTS
     **/
    public static Task candidateTask(TaskHandle taskH, Class[] coi,
                                     ProjectComponent wrt)
    {
        AntX.require_(taskH!=null && wrt!=null && coi!=null, IAM_,
                      "peekTask- nonzro args");

        Task task = taskH.getTask();
        AntX.verify_(task!=null, IAM_, "peekTask- nonzro task");

        if (match(task,coi) && AntX.STRICT_SCRIPTS) {
            task = trueTask(task,wrt);
            taskH.setTask(task);
        }
        return task;
    }



    /**
     * Convenience utility for initializing an enclosed worker task that
     * functions on behalf of another (public) one.
     * @param implTask the hidden worker task (non-null)
     * @param ctrlTask the controlling task (non-null)
     * @since JWare/AntX 0.5
     **/
    public static void initTaskFrom(Task implTask, Task ctrlTask)
    {
        AntX.require_(implTask!=null && ctrlTask!=null, IAM_,
                      "initTask- nonzro args");
        implTask.setProject(ctrlTask.getProject());
        implTask.setTaskName(ctrlTask.getTaskName());
        implTask.setLocation(ctrlTask.getLocation());
        implTask.setOwningTarget(ctrlTask.getOwningTarget());
        implTask.init();
    }

 

    /**
     * Initialization steps for a new
     * <span class="src">UnknownElement</span> proxy for another
     * task. The new element can adopt a specific namespace or
     * assume the standard Ant namespace. Caller must provide a
     * valid proxy parent target if necessary.
     * @param ue the UE to initialize (can be UE subclass) (non-null)
     * @param tag unqualified (task) name of new element (non-null)
     * @param attrs [optional] automatic parameters for new element
     * @param nsuri [optional] new element's namespace marker
     * @param P project with which new element associated (non-null)
     * @throws IllegalArgumentException if either project or tag
     *         is <i>null</i>
     * @since JWare/AntXtras 0.5.1
     **/
    public static UnknownElement initUEProxy(UnknownElement ue, String tag, Map attrs,
            String nsuri,
            final Project P)
    {
        AntX.require_(ue!=null,IAM_,"newUE- nonzro UE obj");
        AntX.require_(P!=null,IAM_,"newUE- nonzro proj");
        AntX.require_(tag!=null,IAM_,"newUE- nonzro element tag");
        
        String qname;
        if (!Tk.isWhitespace(nsuri)) {
            qname = nsuri+":"+tag;
            ue.setNamespace(nsuri);
        } else {
            qname = tag;
            ue.setNamespace("");//NB:required by UE's processing!
        }
        
        ue.setQName(qname);
        ue.setTaskName(tag);
        ue.setTaskType(qname);
        ue.setProject(P);
        
        dynamicConfigure(ue, tag, attrs);
        
        return ue;
    }



    /**
     * Factory method that creates a new <em>standard</em> 
     * <span class="src">UnknownElement</span> proxy for another
     * task. The new element can adopt a specific namespace or
     * assume the standard Ant namespace. Caller must provide a
     * valid proxy parent target if necessary.
     * @param tag unqualified (task) name of new element (non-null)
     * @param attrs [optional] automatic parameters for new element
     * @param nsuri [optional] new element's namespace marker
     * @param P project with which new element associated (non-null)
     * @throws IllegalArgumentException if either project or tag
     *         is <i>null</i>
     **/
    public static UnknownElement newUEProxy(String tag, Map attrs,
                                            String nsuri,
                                            final Project P)
    {
        AntX.require_(P!=null,IAM_,"newUE- nonzro proj");
        AntX.require_(tag!=null,IAM_,"newUE- nonzro element tag");

        UnknownElement ue = new UnknownElement(tag);
        initUEProxy(ue,tag,attrs,nsuri,P);
        return ue;
    }



    /**
     * Applies a set of attribute byname to a new helper object.
     * @param ue unknown helper task object (unconfigured, non-null)
     * @param tag script-facing common name for helper task
     * @param attrs attributes to apply (non-null)
     * @since JWare/AntX 0.5
     **/
    public static void dynamicConfigure(Task ue, String tag, Map attrs)
    {
        AntX.require_(ue!=null,IAM_,"dynCfg- nonzro element");
        AntX.require_(tag!=null,IAM_,"dynCfg- nonzro element tag");
        
        RuntimeConfigurable info = new RuntimeConfigurable(ue,tag);

        if (attrs!=null && !attrs.isEmpty()) {
            Iterator itr= attrs.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry mE= (Map.Entry)itr.next();
                info.setAttribute((String)mE.getKey(),(String)mE.getValue());
            }
        }
    }



    /**
     * Factory method that creates a new
     * <span class="src">UnknownElement</span> proxy for another
     * task. The new element can adopt the namespace of the
     * parent or assume the standard Ant namespace.
     * @param tag unqualified (task) name of new element (non-null)
     * @param attrs [optional] automatic parameters for new element
     * @param nsSource [optional] non-null to have new item inherit namespace
     * @param P project with which new element associated (non-null)
     * @throws IllegalArgumentException if either project or tag
     *         is <i>null</i>
     **/
    public static UnknownElement newUEProxy(String tag, Map attrs,
                                            Task nsSource,
                                            final Project P)
    {
        String nsuri=null;
        if (nsSource!=null) {
            nsuri = ProjectHelper.extractUriFromComponentName
                (nsSource.getTaskType());
        }
        return newUEProxy(tag,attrs,nsuri,P);
    }



    /**
     * Factory method for a placeholder target for a proxy
     * element.
     * @param P enclosing project
     * @since JWare/AntX 0.5
     **/
    public static Target newStubTarget(final Project P)
    {
        Target dumiTarget = new Target();
        dumiTarget.setProject(P);
        dumiTarget.setName("");
        return dumiTarget;
    }



    /**
     * Factory method that creates a new copy of an exsting
     * <span class="src">UnknownElement</span> proxy. Most of this
     * method mimics the copy code found in the standard Ant
     * <span class="src">MacroInstance</span> implementation.
     * @param orig original to be copied (non-null)
     * @param P [optional] project with which new copy associated
     * @param T [optional] project target with which new copy associated
     * @param resolveValues <i>true</i> to try attribute value resolution
     *           immediately (useful for loop-dependent values)
     * @throws IllegalArgumentException if original is <i>null</i>
     * @since JWare/AntX 0.5
     **/
    public static UnknownElement copyUEProxy(UnknownElement orig,
                                             final Project P, Target T,
                                             boolean resolveValues)
    {
        AntX.require_(orig!=null,IAM_,"copyUE- nonzro source");
        UnknownElement copy = new UnknownElement(orig.getTag());
        boolean resolve = resolveValues && P!=null;

        //1. Element programmatic members
        copy.setNamespace(orig.getNamespace());
        copy.setProject(P);
        copy.setQName(orig.getQName());
        copy.setTaskType(orig.getTaskType());
        copy.setTaskName(orig.getTaskName());
        copy.setLocation(orig.getLocation());

        //1.b Ensure target linked to copy's project!
        if (T!=null) {
            copy.setOwningTarget(T);
        } else if (P!=null && P!=orig.getProject()) {
            T = newStubTarget(P); //NB:children too!
            copy.setOwningTarget(T);
        } else {
            copy.setOwningTarget(orig.getOwningTarget());
        }

        //2. Element parameters
        RuntimeConfigurable copyconfig= new RuntimeConfigurable(copy,orig.getTaskName());
        copyconfig.setPolyType(orig.getWrapper().getPolyType());
        Map map = orig.getWrapper().getAttributeMap();
        for (Iterator itr= map.entrySet().iterator(); itr.hasNext();) {
            Map.Entry mE= (Map.Entry)itr.next();
            String value = (String)mE.getValue();
            if (resolve) {
                value = tryResolve(P,value);
            }
            copyconfig.setAttribute((String)mE.getKey(), value);
        }
        map= null;

        //3. Element nested elements and (freeform) text
        String text = orig.getWrapper().getText().toString();
        if (resolve) {
            text = tryResolve(P,text);
        }
        copyconfig.addText(text);
        

        Enumeration e = orig.getWrapper().getChildren();
        while (e.hasMoreElements()) {
            RuntimeConfigurable rc = (RuntimeConfigurable)e.nextElement();
            UnknownElement childorig = (UnknownElement)rc.getProxy();
            UnknownElement childcopy = copyUEProxy(childorig,P,T,resolveValues);
            copyconfig.addChild(childcopy.getWrapper());
            copy.addChild(childcopy);
        }
        e= null;

        return copy;
    }


    /**
     * Default version of 
     * {@linkplain #copyUEProxy(UnknownElement,Project,Target,boolean)
     * copyUEProxy(&#8230;)} that does <em>not</em> resolve any UE
     * attribute values.
     **/
    public static UnknownElement copyUEProxy(UnknownElement orig,
                                             final Project P, Target T)
    {
        return copyUEProxy(orig,P,T,false);
    }


    /** Block; not allowed. Only public utility methods. **/
    private TaskExaminer()
    {
    }
}

/* end-of-TaskExaminer.java */
