/**
 * $Id: PropertiesIterator.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers;

import  java.util.Iterator;
import  java.util.Map;
import  java.util.NoSuchElementException;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.RegularExpression;
import  org.apache.tools.ant.util.regexp.Regexp;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.parameters.IgnoreCaseEnabled;
import  com.idaremedia.antx.parameters.PropertySource;

/**
 * Iterator that will iterate a project's properties according to some
 * application-defined selection criteria. This iterator does not support
 * any mutative operation (like remove) to the project's properties. If a properties
 * iterator's selection criteria are modified during an iteration, the effect
 * on the iteration's results are undefined. <span class="src">PropertiesIterator</span>s
 * return standard <span class="src">Map.Entry</span> items.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 * @.pattern GoF.Iterator
 * @see      InnerProperties
 * @see      PropertyExaminer
 * @see      UnresolvedProperty
 **/

public final class PropertiesIterator extends ProjectDependentSkeleton
    implements Iterator, IgnoreCaseEnabled
{
    private static final String IAM_ = AntX.utilities+"PropertiesIterator:";


    /**
     * Initializes a new unattached iterator. Before it is used,
     * this iterator's project must be set.
     * @see #setProject
     * @see #setRE(RegularExpression) setRE(&#8230;)
     **/
    public PropertiesIterator()
    {
    }


    /**
     * Initializes a new iterator for a particular project
     * and properties type.
     * @param P the project (non-null)
     * @param domain which properties (non-null)
     **/
    public PropertiesIterator(Project P, PropertySource domain)
    {
        setProject(P);
        setDomain(domain);
    }


    /**
     * Initializes a new iterator for a particular
     * script declaration. The bean's project will be used.
     **/
    public PropertiesIterator(InnerProperties bean)
    {
        setProject(bean.getProject());
        setDomain(bean.getDomain());
        setIgnoreCase(bean.isIgnoreCase());
        setPrefix(bean.getPrefixFilterPattern());
        String pattern = bean.getLikeFilterPattern();
        if (pattern==null) {
            pattern = bean.getNotLikeFilterPattern();
            if (pattern!=null) {
                setExclude(true);
            }
        }
        if (pattern!=null) {
            RegularExpression re = new RegularExpression();
            re.setPattern(pattern);
            setRE(re);
        }
    }


    /**
     * Call to mark this iterator as no longer in use.
     * A GC-helper.
     **/
    public void dispose()
    {
        m_entryItr = null;
        m_re = null;
        m_nextEntry = null;
        m_formCheck = null;
        m_prefix = null;
    }


    /**
     * Initializes this iterator's target project. This iterator
     * extracts properties (and patterns) from this project.
     * @param P the project (non-null)
     **/
    public void setProject(Project P)
    {
        AntX.require_(P!=null,IAM_,"setProject- nonzro project");
        super.setProject(P);
    }


    /**
     * Initializes this iterator's properties domain to a specific
     * (sub)set of properties.
     * @param domain properties domain (non-null)
     **/
    public void setDomain(PropertySource domain)
    {
        AntX.require_(domain!=null,IAM_,"setDomain- nonzro domain");
        m_domain = domain;
    }


    /**
     * Returns this iterator's properties domain. Will never return
     * <i>null</i>; returns {@linkplain PropertySource#ALL ALL} if
     * never defined.
     **/
    public PropertySource getDomain()
    {
        return m_domain;
    }


    /**
     * Sets or clears a filtering regular expression for this
     * iterator.
     * @param re the regular expression filter (can be <i>null</i>)
     **/
    public void setRE(RegularExpression re)
    {
        m_re = re;
    }


    /**
     * Sets or clears a filtering regular expression for this
     * iterator.
     * @param re the regular expression filter (can be <i>null</i>)
     **/
    public void setRE(Regexp re)
    {
        m_re = re;
    }


    /**
     * Returns <i>true</i> if this iterator includes a filtering
     * regular expression.
     * @see #setRE(RegularExpression) setRE(&#8230;)
     **/
    public boolean willCheckRE()
    {
        return m_re!=null;
    }


    /**
     * Sets or clears a filtering property name prefix for this
     * iterator.
     * @param prefix the filter prefix (can be <i>null</i>)
     **/
    public void setPrefix(String prefix)
    {
        m_prefix = prefix;
        m_prefixLocal = null;
    }


    /**
     * Returns this iterator's filtering prefix. Returns
     * <i>null</i> if not set.
     **/
    public String getPrefix()
    {
        return m_prefix;
    }


    /**
     * Returns <i>true</i> if this iterator includes a filtering
     * prefix.
     * @see #setPrefix(String) setPrefix(&#8230;)
     **/
    public boolean willCheckPrefix()
    {
        return m_prefix!=null;
    }


    /**
     * Tells this iterator to ignore case (or not) in regular
     * expression or prefix matches.
     * @param ignore <i>true</i> to ignore
     **/
    public void setIgnoreCase(boolean ignore)
    {
        m_reOptions = ignore ? Regexp.MATCH_CASE_INSENSITIVE : 0/*?*/;
    }


    /**
     * Returns <i>true</i> if this iterator will do case-insensitive
     * regular expression or prefix matches. Is <i>false</i> by default.
     **/
    public boolean isIgnoreCase()
    {
        return m_reOptions!=0;
    }



    /**
     * Tells this iterator to exclude items matching this iterator's
     * regular expression. Exclusion does <em>not</em> apply to
     * prefix matching. If exclusion is turned on, this iterator will
     * return items that <em>do not match</em> its pattern.
     * @param exclude <i>true</i> if is exclusion
     **/
    public final void setExclude(boolean exclude)
    {
        m_exclude = exclude;
    }



    /**
     * Returns <i>true</i> if this iterator will return items that
     * do <em>not</em> match its prefix and/or regular expression.
     **/
    public final boolean isExclusion()
    {
        return m_exclude;
    }


    /**
     * Tells this iterator to check each property for an all-in-one
     * failed variable substitution. If called, this iterator will either
     * skip bad properties or return a proxy if one defined.
     * @see #setBrokenSubstitutionProxy setBrokenSubstitutionProxy(String)
     **/
    public void setCheckSimpleBrokenSubstitution()
    {
        getFormCheck().setCheckSimpleBrokenSubstitution();
    }


    /**
     * Tells this iterator to thoroughly check each property for any
     * bad (failed) variable substitution. If called, this iterator will either
     * skip bad properties or return a proxy if one defined.
     * @see #setBrokenSubstitutionProxy setBrokenSubstitutionProxy(String)
     **/
    public void setCheckBrokenSubstitution()
    {
        getFormCheck().setCheckBrokenSubstitution();
    }


    /**
     * Define a proxy value string for failed substitution
     * properties. If this iterator is not already setup to check
     * for unresolved property values, this method will turn
     * checking on.
     * @param valueProxy string to use for property value (non-null)
     * @see UnresolvedProperty#VALUE
     * @.sideeffect Will turn property definition checking on.
     **/
    public void setBrokenSubstitutionProxy(String valueProxy)
    {
        AntX.require_(valueProxy!=null,IAM_,"setProxy- nonzro valu");
        getFormCheck().setBrokenSubstitutionProxy(valueProxy);
    }


    /**
     * Returns <i>true</i> if this iterator will filter
     * unresolved properties from list of returned items.
     **/
    public boolean willCheckForm()
    {
        return m_formCheck!=null;
    }


    /**
     * Determine what is to be iterated and how the selections
     * should be filtered. Called once, when iterator's public
     * iteration API has been triggered.
     **/
    private void prepareToIterate()
    {
        Project P = getProjectNoNull();

        Map mp = FixtureExaminer.copyOfProperties(getDomain(),P);
        m_itemCount = mp.size();
        m_entryItr = mp.entrySet().iterator();
        mp = null;

        if (m_re instanceof RegularExpression) {
            m_re = ((RegularExpression)m_re).getRegexp(P);
        }
        if (isIgnoreCase() && m_prefix!=null) {
            m_prefixLocal= m_prefix.toLowerCase();
        }
    }


    /**
     * Returns the total number of items this iterator can return.
     * However, this number does not exclude items filtered by each
     * call to {@linkplain #next}, so the actual number of
     * iterations can be less than this value.
     **/
    public int candidateCount()
    {
        return m_itemCount;
    }


    /**
     * Returns <i>true</i> if this iterator has at least one
     * more property to return.
     **/
    public boolean hasNext()
    {
        if (m_entryItr==null) {
            prepareToIterate();
        }

        boolean gotOne = (m_nextEntry!=null);

        if (!gotOne) {
            while (m_entryItr.hasNext()) {
                Map.Entry candidate = (Map.Entry)m_entryItr.next();
                boolean match;

                if (willCheckPrefix()) {
                    String name = (String)candidate.getKey();
                    if (m_prefixLocal!=null) {
                        name = name.toLowerCase();
                        match = name.startsWith(m_prefixLocal);
                    } else {
                        match = name.startsWith(m_prefix);
                    }
                    //NB: -exclusion does not apply to prefix (ssmc)
                    if (!match) {
                        continue;
                    }
                }

                if (willCheckRE()) {
                    String name = (String)candidate.getKey();
                    match = ((Regexp)m_re).matches(name, m_reOptions);
                    if (m_exclude) {
                        if (match) {
                            continue;
                        }
                    } else if (!match) {
                        continue;
                    }
                }

                if (willCheckForm()) {
                    candidate = m_formCheck.verifiedPropertyValue
                        (candidate, getProject());
                }

                if (candidate!=null) {
                    m_nextEntry = candidate;
                    gotOne = true;
                    break;
                }
            }
        }

        return gotOne;
    }


    /**
     * Returns the next project property key-value pair as a
     * <span class="src">Map.Entry</span>.
     * @throws NoSuchElementException if nothing more to iterate
     **/
    public Object next()
    {
        if (m_nextEntry==null) {
            throw new NoSuchElementException();
        }
        Object result = m_nextEntry;
        m_nextEntry = null;
        return result;
    }



    /**
     * Always generates an
     * <span class="src">UnsupportedOperationException</span>.
     * @throws UnsupportedOperationException always
     **/
    public final void remove()
    {
        throw new UnsupportedOperationException();
    }



    /**
     * Resets this iterator for reuse. You can alter the selection
     * criteria before the next call to <span class="src">hasNext</span>.
     * The set of candidate properties will be recalculated.
     **/
    public void reset()
    {
        m_entryItr = null;
        m_nextEntry = null;
        m_itemCount = 0;
        m_prefixLocal = null;
    }



    /**
     * Returns this iterator's property definition checker,
     * allocating the object if necessary.
     **/
    private PropertyExaminer getFormCheck()
    {
        if (m_formCheck==null) {
            m_formCheck = new PropertyExaminer();
        }
        return m_formCheck;
    }


    private Iterator m_entryItr;
    private Map.Entry m_nextEntry;
    private int m_itemCount;

    private PropertySource m_domain= PropertySource.ALL;
    private String m_prefix, m_prefixLocal;
    private Object m_re;
    private int m_reOptions;//NB:=> case-sensitive matches
    private PropertyExaminer m_formCheck;//lazy-inited
    private boolean m_exclude;//NB:=> include not exclude
}

/* end-of-PropertiesIterator.java */
