/**
 * $Id: LocalTk.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers;

import  java.lang.reflect.Constructor;
import  java.lang.reflect.Method;
import  java.io.File;
import  java.io.InputStream;
import  java.net.URL;

import  org.apache.tools.ant.BuildEvent;
import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.DirectoryScanner;
import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.launch.Locator;
import  org.apache.tools.ant.types.Reference;
import  org.apache.tools.ant.util.ClasspathUtils;
import  org.apache.tools.ant.util.FileUtils;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.apis.ScriptLocatable;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Collection of independent but AntX-specific utility methods. These utilities can
 * use classes from the root-level 'helpers' and 'antx' packages only.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   impl,helper
 **/

public final class LocalTk
{
    private static final String IAM_= AntX.AntX+"LocalTk:";


    /**
     * Factory method that calls the right constructor based on active
     * Ant runtime. Before Ant 1.6.3, a reference's project was determined
     * by each API; post 1.6.2 you must pass in the project at construction.
     * @param refid the reference id (non-null)
     * @param project the owning project of reference (non-null)
     * @return new reference (never <i>null</i>)
     * @since JWare/AntX 0.5
     **/
    public static final Reference referenceFor(String refid, Project project)
    {
        return new Reference(refid);
    }


    /**
     * Like {@linkplain #referenceFor referenceFor(refid,project)} but
     * where the new reference is based on any existing object. Useful
     * from constructors and clone methods.
     * @param ref existing reference (non-null)
     * @return new reference (never <i>null</i>)
     * @since JWare/AntX 0.5
     **/
    public static final Reference referenceFor(Reference ref)
    {
        return referenceFor(ref.getRefId(),null/*until require 1.6.3*/);
    }



    /**
     * Used as loaderId to cache and reuse project-based class loaders.
     **/
    private static final String ANTX_SYSLOADER_ =
        AntX.ANTX_INTERNAL_ID+"SystemResourcesClassLoader";


    /**
     * Tries to load a "system" resource from within Ant environment. This
     * method uses an Ant runtime class loader to load resource to avoid
     * all of the issues Ant vs. JUnit vs. Eclipse vs. Universe+Dog seem to
     * trigger on occasion. Replaces typical calls to
     * <span class="src">ClassLoader.getSystemResource()</span>.
     * @param resource resource to be loaded (not-null, not-class)
     * @param P project on whose behalf resource loading performed
     * @return resource URL or <i>null</i> if cannot find resource.
     **/
    public static final URL getSystemResource(String resource, Project P)
    {
        AntX.require_(P!=null,IAM_,"getSysRez- nonzro project");
        AntX.require_(resource!=null,IAM_,"getSysRez- nonzro resource");

        ClassLoader cL = ClasspathUtils.getClassLoaderForPath
            (P, null/*systempath*/, ANTX_SYSLOADER_, false, true);

        return cL.getResource(resource);
    }



    /**
     * Like {@linkplain #getSystemResource getSystemResource(&#8230;)} but
     * returns resource as an input stream.
     * @param resource resource to be opened (not-null, not-class)
     * @param P project on whose behalf resource loading performed
     * @return resource stream or <i>null</i> if cannot find resource
     */
    public static final InputStream getSystemResourceAsStream
        (String resource, Project P)
    {
        AntX.require_(P!=null,IAM_,"getSysRez- nonzro project");
        AntX.require_(resource!=null,IAM_,"getSysRez- nonzro resource");

        ClassLoader cL = ClasspathUtils.getClassLoaderForPath
            (P, null/*systempath*/, ANTX_SYSLOADER_, false, true);

        return cL.getResourceAsStream(resource);
    }



    /**
     * Returns location of a named system resource. Will return <i>null</i>
     * if unable to locate resource. Like Ant's Locator method but using
     * our class loader.
     * @param resource resource to be loaded (not-null, not-class)
     * @param P project on whose behalf resource loading performed
     * @return resource's directory or <i>null</i> if cannot find resource.
     * @since JWare/AntX 0.5
     **/
    public static final File getSystemResourceSource(String resource, Project P)
    {
        URL url = getSystemResource(resource,P);
        return getFilesystemSource(url,resource,P);
    }



    /**
     * Like {@linkplain #getSystemResource getSystemResource(&#8230;)}
     * but for an arbitrary resource with a custom class path.
     * @param resource the resource to locate (non-null)
     * @param pathref the classpath to use (if null, system classpath used)
     * @param project the project on whose behalf resource loading done (non-null)
     * @return resource's URL or <i>null</i> if cannot locate resource.
     **/
    public static final URL getResource(String resource, String pathref,
        Project project)
    {
        if (pathref==null) {
            return getSystemResource(resource,project);
        }
        AntX.require_(project!=null,IAM_,"getRez- nonzro project");
        AntX.require_(resource!=null,IAM_,"getRez- nonzro resource");

        ClassLoader cL = ClasspathUtils.getClassLoaderForPath
                            (project, referenceFor(pathref, project), false);
        return cL.getResource(resource);
    }



    /**
     * Like {@linkplain #getSystemResourceAsStream getSystemResourceAsStream(&#8230;)}
     * but for an arbitrary resource with a custom class path.
     * @param resource the resource to locate (non-null)
     * @param pathref the classpath to use (if null, system classpath used)
     * @param project the project on whose behalf resource loading done (non-null)
     * @return resource's URL or <i>null</i> if cannot locate resource.
     **/
    public static final InputStream getResourceAsStream(String resource, 
        String pathref, Project project)
    {
        if (pathref==null) {
            return getSystemResourceAsStream(resource,project);
        }
        AntX.require_(project!=null,IAM_,"getRez- nonzro project");
        AntX.require_(resource!=null,IAM_,"getRez- nonzro resource");

        ClassLoader cL = ClasspathUtils.getClassLoaderForPath
                            (project, referenceFor(pathref,project), false);
        return cL.getResourceAsStream(resource);
    }



    /**
     * Like {@linkplain #getSystemResourceSource getSystemResourceSource(&#8230;)}
     * but for an arbitrary resource with a custom class path.
     * @param resource the resource to locate (non-null)
     * @param pathref the classpath to use (if null, system classpath used)
     * @param project the project on whose behalf resource loading done (non-null)
     * @return resource's URL or <i>null</i> if cannot locate resource.
     **/
    public static final File getResourceSource(String resource,
        String pathref, Project project)
    {
        URL url = getResource(resource, pathref, project);
        return getFilesystemSource(url, resource, project);
    }



    /**
     * Returns location of a named system resource. Will return <i>null</i>
     * if unable to locate resource. Like Ant's Locator method but using
     * our class loader.
     * @param url URL for resource to be translated to file (non-null)
     * @param resource the resource's id as passed by application (non-null)
     * @param P project on whose behalf resource loading performed
     * @return resource's location as a file or <i>null</i> if cannot
     *        find resource.
     * @since JWare/AntX 0.5
     **/
    public static final File getFilesystemSource(URL url, String resource, Project P)
    {
        /**
         * Following snipped (verbatim) from Ant's Locator 
         * source to deal w/ jars.
         */
        if (url != null) {
            String u = url.toString();
            if (u.startsWith("jar:file:")) {
                int pling = u.indexOf("!");
                String jarName = u.substring(4, pling);
                return new File(Locator.fromURI(jarName));
            } else if (u.startsWith("file:")) {
                int tail = u.indexOf(resource);
                String dirName = u.substring(0, tail);
                return new File(Locator.fromURI(dirName));
            }
        }
        return null;
    }



    /**
     * Ensures all file names from a directory scanner are
     * fully-resolved absolute path names.
     * @param filesOrDirs items to be resolved (non-null)
     * @param dirscan the scanner resolution based on (non-null)
     * @param basenameOnly <i>true</i> if only the items base 
     *         names should be saved
     * @since JWare/AntX 0.5
     **/
    public static final void resolveAllFilesOrDirs(String[] filesOrDirs,
            DirectoryScanner dirscan, boolean basenameOnly)
    {
        FileUtils fileUtils = AntXFixture.fileUtils();
        File wrt = dirscan.getBasedir();
        for (int i=0;i<filesOrDirs.length;i++) {
            File rf= fileUtils.resolveFile(wrt,filesOrDirs[i]);
            if (basenameOnly) {
                filesOrDirs[i]= rf.getName();
            } else {
                filesOrDirs[i]= rf.getAbsolutePath();
            }
        }
    }



    /**
     * Creates a log message that is labeled w/ task|target if possible.
     * @param e the event (non-null 'messageLogged' event)
     * @since JWare/AntX 0.4
     **/
    public static final String purtyEventMsg(BuildEvent e)
    {
        if (e.getTask()!=null || e.getTarget()!=null) {
            StringBuffer sb = new StringBuffer(e.getMessage().length()+50);
            sb.append("[");
            if (e.getTask()!=null) {
                sb.append(e.getTask().getTaskName());
            } else {
                sb.append(e.getTarget().getName());
            }
            sb.append("]  ");
            sb.append(e.getMessage());
            return sb.substring(0);
        }
        return e.getMessage();
    }



    /**
     * Makes a Location string human-readable (manageable) if 
     * application's defaults allow it.
     * @param l the location to be stringified (non-null)
     * @since JWare/AntX 0.5
     **/
    public static String purtyLocation(Location l)
    {
        return Tk.shortStringFrom
            (Iteration.defaultdefaults().isShortLocationsEnabled(),l);
    }



    /**
     * Feedback helper that replaces one build exception with another
     * one (usually script-specified). This method returns the old
     * error if it is unable to create a new one for any reason. The
     * newly generated exception is <em>always</em> a 
     * <span class="src">BuildException</em>; the incoming exception 
     * does not have to be.
     * @param oldX the current exception (non-null)
     * @param fallbackLoc location used if current exception has none
     * @param newXclass the new build exception's class (non-null)
     * @throws IllegalArgumentException if old exception or new class
     *         is <i>null</i>
     * @since JWare/AntX 0.4
     **/
    public static Exception replaceError(Exception oldX,
                                         Location fallbackLoc,
                                         Class newXclass)
    {
        AntX.require_(oldX!=null && newXclass!=null,IAM_,
                      "replErr- nonzro args");

        Constructor ctor=null;
        BuildException newX= null;
        try {
            if (!BuildException.class.isAssignableFrom(newXclass)) {
                return oldX;
            }
            Location loc;
            if (oldX instanceof BuildException) {
                BuildException oldBX = (BuildException)oldX;
                loc = oldBX.getLocation();
                if (loc==Location.UNKNOWN_LOCATION) {
                    if (oldBX.getCause() instanceof BuildException) {//only 1 down!
                        loc = ((BuildException)oldBX.getCause()).getLocation();
                        if (loc==Location.UNKNOWN_LOCATION) {
                            loc = fallbackLoc;
                        }
                    } else {
                        loc = fallbackLoc;
                    }
                }
            } else if (oldX instanceof ScriptLocatable) {
                loc = ((ScriptLocatable)oldX).getLocation();
                if (loc==Location.UNKNOWN_LOCATION) {
                    loc = fallbackLoc;
                }
            } else {
                loc = fallbackLoc;
            }
            try {
                ctor = newXclass.getConstructor(AFE_SIG2);
                newX = (BuildException)ctor.newInstance
                            (new Object[]{oldX.getMessage(),loc});
            } catch(Exception igx) {
                ctor = newXclass.getConstructor(AFE_SIG1);
                newX = (BuildException)ctor.newInstance
                            (new Object[]{oldX.getMessage()});
            }
            try {
                Method m= newXclass.getMethod("initCause",AFE_SIG0);
                m.invoke(newX, new Object[]{oldX});
            } catch(Exception jre13X) {/*burp*/}
        } catch(Exception anyX) {
            /*burp*/
        }
        return (newX!=null) ? newX : oldX;
    }



    /**
     * Feedback helper that replaces on build error with another
     * (usually script specified) one.
     * @param oldX the current exception (non-null)
     * @param fallbackLoc location used if current exception has none
     * @param newXclass the new exception's class name (non-null)
     * @since JWare/AntX 0.4
     **/
    public static Exception replaceError(Exception oldX,
                                         Location fallbackLoc,
                                         String newXclass)
    {
        AntX.require_(oldX!=null && newXclass!=null, IAM_,
                      "replErr- nonzro args");
        try {
            return replaceError(oldX,fallbackLoc,Class.forName(newXclass));
        } catch(Exception anyX) {
            return oldX;
        }
    }


    /**
     * Helper that tries to install a cause into a new Throwable
     * using reflection instead of direct JRE 1.4 APIs.
     * @param newerX the throwable to be updated (non-null)
     * @param causeX the original boo-boo (non-null)
     * @since JWare/AntX 0.4
     **/
    public static void tryInitCause(Throwable newerX, Throwable causeX)
    {
        AntX.require_(newerX!=null && causeX!=null, IAM_,
                     "initCause- nonzro args");

        try {
            Method m= newerX.getClass().getMethod("initCause",AFE_SIG0);
            m.invoke(newerX, new Object[]{causeX});
        } catch(Exception jre13OrInitedX) {/*burp*/}
    }


    private static final Class[] AFE_SIG0= new Class[]{Throwable.class};
    private static final Class[] AFE_SIG1= new Class[]{String.class};
    private static final Class[] AFE_SIG2= new Class[]{String.class,Location.class};


    /**
     * Prevent; only public static utility methods.
     **/
    private LocalTk()
    {
    }
}

/* end-of-LocalTk.java */
