/**
 * $Id: FileSetsIterator.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers;

import  java.io.File;
import  java.util.Iterator;
import  java.util.List;
import  java.util.NoSuchElementException;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.DirectoryScanner;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.FileSet;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.helpers.ArrayIterator;

/**
 * Helper that can iterate a heterogeneous collection of Ant
 * <span class="src">FileSet</span>s, <span class="src">File</span>s, and
 * path strings. The iterator returns absolute path <em>names</em>.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,helper
 * @.caveat   Both file and pathnames are returned <em>as-is</em>.
 **/

public class FileSetsIterator extends ProjectDependentSkeleton implements Iterator
{
    private static final String IAM_ = AntX.utilities+"FileSetsIterator:";


    /**
     * Initializes new iterator for given collection of file
     * declarations.
     * @param filesets list of filesets, Files, and path names(strings) (non-null)
     * @param project project from which base directory should be determined (non-null)
     **/
    public FileSetsIterator(List filesets, Project project)
    {
        super();
        AntX.require_(filesets!=null,IAM_,"ctor- nonzro fileset list");
        setProject(project);
        m_includesItr = filesets.isEmpty() ? null : filesets.iterator();
    }



    /**
     * Returns <i>true</i> if there is at least one path left
     * to iterate.
     **/
    public boolean hasNext()
    {
        if (m_includesItr==null) {
            return false;
        }

        boolean gotOne = (m_nextItem!=null);

        if (!gotOne) {
            if (m_innerItr!=null) {
                if (m_innerItr.hasNext()) {
                    m_nextItem = m_innerItr.next();
                    return true;
                }
                m_innerItr = null;
                if (m_privateCall) {
                    return false;
                }
            }
            while (m_includesItr.hasNext()) {
                Object candidate = m_includesItr.next();

                if (candidate instanceof FileSet) {
                    DirectoryScanner ds = ((FileSet)candidate).getDirectoryScanner(getProject());
                    String[] list = ds.getIncludedFiles();
                    LocalTk.resolveAllFilesOrDirs(list,ds,false);
                    m_innerItr = new ArrayIterator(list);
                    m_privateCall=true;
                    gotOne = this.hasNext();//get 1st one setup!
                    m_privateCall=false;
                    if (gotOne) {
                        return true;
                    }
                    candidate = null;
                }
                else if (candidate instanceof File) {
                    candidate = ((File)candidate).getPath();
                }
                else if (!(candidate instanceof String)) {
                    String error = AntX.uistrs().get("task.bad.path.from",
                                                     String.valueOf(candidate));
                    getProject().log(error,Project.MSG_ERR);
                    throw new BuildException(error);
                }

                if (candidate!=null) {
                    m_nextItem = candidate;
                    gotOne = true;
                    break;
                }
            }
        }

        return gotOne;
    }



    /**
     * Returns the next file path to be handled  as a
     * <span class="src">String</span>.
     * @throws NoSuchElementException if nothing left to iterate
     **/
    public Object next()
    {
        if (m_nextItem==null) {
            throw new NoSuchElementException();
        }
        Object result = m_nextItem;
        m_nextItem = null;
        return result;
    }



    /**
     * Always generates an
     * <span class="src">UnsupportedOperationException</span>.
     * @throws UnsupportedOperationException always
     **/
    public void remove()
    {
        throw new UnsupportedOperationException();
    }


    private Iterator m_includesItr; //NB:list of filesets,files,pathnames,etc.
    private Iterator m_innerItr;//NB:list of a particular fileset's included bits
    private Object   m_nextItem;//NB:next String
    private boolean  m_privateCall;
}


/* end-of-FileSetsIterator.java */