/**
 * $Id: MkNewObject.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.mktemp;

import  java.lang.reflect.Constructor;
import  java.io.File;
import  java.io.FileOutputStream;
import  java.io.IOException;
import  java.io.PrintWriter;
import  java.net.URL;
import  java.util.Iterator;
import  java.util.List;
import  java.util.Properties;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.FilterSet;
import  org.apache.tools.ant.types.FilterSetCollection;
import  org.apache.tools.ant.types.PropertySet;
import  org.apache.tools.ant.util.FileUtils;

import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.LocalTk;
import  com.idaremedia.antx.starters.StringItemListHandle;

/**
 * Skeleton starter for tasks that create or copy file system objects. Implements
 * the common "copy prototype" functionality.
 *
 * @since    JWare/AntX 0.3 (Extracted from MkTempObject)
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 * @.todo    Support for classpath when specifying copy-source resource
 **/

public abstract class MkNewObject extends AssertableTask
{
    /**
     * Initializes a new MkNewObject instance.
     **/
    protected MkNewObject(String iam)
    {
        super(iam);
    }


    /**
     * Sets the property into which a URL representation of the 
     * temp directory's path is copied.
     * @param urlproperty the property's name (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setUrlProperty(String urlproperty)
    {
        require_(urlproperty!=null,"setUrlProp- nonzro name");
        m_updateUrlProperty = urlproperty;
    }



    /**
     * Returns the name of the property into which a URL representation
     * of the temp directory's path will be stored. Returns <i>null</i>
     * if never set explicitly.
     * @since JWare/AntX 0.5
     **/
    public final String getUrlPathProperty()
    {
        return m_updateUrlProperty;
    }


    /**
     * Returns <i>true</i> if this task will try to mark all created
     * prototype copies as delete-on-exit. Returns <i>false</i> by
     * default.
     **/
    public boolean isAutoDelete()
    {
        return false;
    }


    /**
     * Sets this task's prototype source file.
     * @param copyFrom prototype file's path (non-null)
     * @see #getPrototypePlainFile
     **/
    public void setCopyFile(File copyFrom)
    {
        require_(copyFrom!=null,"setCopy- nonzro src file");
        require_(copyFrom.canRead(), "setCopy- existing readable file/dir");
        m_copyFromFile= copyFrom;
    }


    /**
     * Returns this task's prototype source file. Returns
     * <i>null</i> if never specified.
     **/
    public File getPrototypePlainFile()
    {
        return m_copyFromFile;
    }


    /**
     * Sets this task's prototype class-based resource.
     * @param copyFrom resource's name (non-null)
     * @see #getPrototypeResourceFile
     **/
    public void setCopyResource(String copyFrom)
    {
        require_(!Tk.isWhitespace(copyFrom),"setCopyRez- nonzro rez name");
        m_copyFromResource = copyFrom;
    }


    /**
     * Returns this task's prototype class-based resource. Returns
     * <i>null</i> if never specified.
     **/
    public String getPrototypeResource()
    {
        return m_copyFromResource;
    }


    /**
     * Returns this task's prototype resource's actual file.
     * Returns <i>null</i> if never defined. All prototype resources
     * must resolve to physical files.
     * @throws BuildException if cannot do conversion (or resource is
     *         not a file)
     **/
    public File getPrototypeResourceFile()
    {
        if (m_copyFromRezFile==null && getPrototypeResource()!=null) {
            File rezfile = getFileFromResource(getPrototypeResource());
            m_copyFromRezFile = rezfile;
        }
        return m_copyFromRezFile;
    }


    /**
     * Adds a new filterset to use when copying prototype files.
     * @since JWare/AntX 0.2
     **/
    public void addFilterSet(FilterSet filters)
    {
        require_(filters!=null,"addFiltrs- nonzro filtrs");
        getCopyFiltersNoNull().addFilterSet(filters);
    }


    /**
     * Returns <i>true</i> if either a prototype file or prototype
     * resource has been defined.
     * @.sideeffect Will convert prototype resource name into actual file
     **/
    protected final boolean needToCopyFile()
    {
        return (getPrototypePlainFile()!=null ||
                getPrototypeResourceFile()!=null);
    }


    /**
     * Converts a named class resource to a filesystem entity.
     * @throws BuildException if cannot do conversion (or resource is
     *         not a file)
     **/
    protected final File getFileFromResource(String resource)
        throws BuildException
    {
        require_(resource!=null,"rezToFile- nonzro rez");

        File rezfile = null;
        URL url = LocalTk.getSystemResource(resource,getProject());
        if (url!=null) {
            rezfile = Tk.toFile(url);
            if (rezfile!=null && !rezfile.canRead()) {
                rezfile=null;
            }
        }
        if (rezfile==null) {
            String ermsg = uistrs().get("mktemp.cant.read.resource",resource);
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        return rezfile;
    }


    /**
     * Copies the prototype file to a target directory. Any defined
     * filters are automatically applied to the new destination file. Will
     * mark new file for deletion-on-exit if {@linkplain #isAutoDelete
     * isAutoDelete} returns <i>true</i>.
     * @param fromFile source prototype file (non-null)
     * @param inDir target directory (non-null, must exist)
     * @throws IOException if unable to copy file
     **/
    protected File copyPrototypeFile(File fromFile, File inDir)
        throws IOException
    {
        File toFile = new File(inDir, fromFile.getName());

        getFileUtils().copyFile(fromFile,toFile,getCopyFilters(),false);

        if (isAutoDelete()) {
            toFile.deleteOnExit();
        }
        return toFile;
    }



    /**
     * Returns this task's pre-allocated Ant file utilities helper.
     **/
    protected final FileUtils getFileUtils()
    {
        return AntXFixture.fileUtils();
    }


    /**
     * Returns this task's underlying filterset collection. Will
     * return <i>null</i> if no filters ever added.
     * @.safety single
     **/
    protected final FilterSetCollection getCopyFilters()
    {
        return m_copyFilters;
    }


    /**
     * Returns this task's underlying filterset collection
     * creating a new empty collection if necessary.
     * @.safety single
     **/
    protected final FilterSetCollection getCopyFiltersNoNull()
    {
        if (m_copyFilters==null) {
            m_copyFilters = new FilterSetCollection();
        }
        return m_copyFilters;
    }


    /**
     * Returns this task's list of prototype file lines creating
     * a new empty list if necessary. Contents of this list must
     * be either InnerStrings, Strings, or some other stringifiable
     * object.
     * @since JWare/AntX 0.3
     **/
    protected List getPrototypeLinesNoNull()
    {
        if (m_prototypeLines==null) {
            m_prototypeLines = AntXFixture.newList();
        }
        return m_prototypeLines;
    }


    /**
     * Returns this task's list of prototype file line contents.
     * Will return <i>null</i> if not lines added to this task.
     * @since JWare/AntX 0.3
     **/
    protected List getPrototypeLines()
    {
        return m_prototypeLines;
    }


    /**
     * Copies this task's list of prototype file line contents
     * to existing file. Does nothing if no prototype lines defined.
     * @param newFile the target file (non-null)
     * @param append <i>true</i> if lines should be appended
     * @since JWare/AntX 0.3
     * @throws BuildException if any I/O error occurs or lines improperly
     *         defined.
     **/
    protected void copyPrototypeLines(File newFile, boolean append)
    {
        List l = getPrototypeLines();
        if (l!=null && !l.isEmpty()) {
            try {
                final Project P = getProject();
                FileOutputStream fos = newFOS(newFile,append);
                PrintWriter w = new PrintWriter(fos);
                for (int i=0,N=l.size();i<N;i++) {
                    Object o = l.get(i);
                    if (o instanceof StringItemListHandle) {
                        Iterator itr= ((StringItemListHandle)o).readonlyStringIterator(P);
                        while (itr.hasNext()) {
                            w.println(itr.next().toString());
                        }
                    } else if (o instanceof PropertySet) {
                        Properties ps = ((PropertySet)o).getProperties();
                        if (!ps.isEmpty()) {
                            w.flush();
                            ps.store(fos,null);
                        }
                        ps.clear();
                        ps=null;//gc
                    } else {
                        w.println(Iteration.lenientStringifer().stringFrom(o,P));
                    }
                }
                w.flush();
                w.close();
            } catch(IOException iox) {
                throw new BuildException(iox,getLocation());
            }
        }
    }



    /**
     * Apply any common update properties to final file system object.
     * @param finalObject file system object (non-null)
     * @param strict <i>true</i> if should fail if unable to set update properties
     * @since JWare/AntX 0.5
     **/
    protected void saveFinalPath(File finalObject, boolean strict)
    {
        if (getUrlPathProperty()!=null) {
            checkIfProperty_(getUrlPathProperty(),!strict);
            String urlPath = AntXFixture.fileUtils().toURI(finalObject.getPath());
            getProject().setNewProperty(getUrlPathProperty(),urlPath);
        }
    }



    /**
     * Create a FileOutputStream with custom append option if active
     * JVM supports it.
     **/
    private FileOutputStream newFOS(File newFile, boolean append)
        throws IOException
    {
        if (append) {
            try {
                Class[] sig = new Class[]{File.class,boolean.class};
                Constructor ctor = FileOutputStream.class.getConstructor(sig);
                Object[] args = new Object[]{newFile, (append ? Boolean.TRUE : Boolean.FALSE)};
                return (FileOutputStream)ctor.newInstance(args);
            } catch(Exception anyX) {/*burp*/}
        }
        return new FileOutputStream(newFile);
    }

    private File m_copyFromFile, m_copyFromRezFile;
    private String m_copyFromResource;
    private FilterSetCollection m_copyFilters;
    private List m_prototypeLines;
    private String m_updateUrlProperty;
}

/* end-of-MkNewObject.java */
