/**
 * $Id: MkFileTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.mktemp;

import  java.io.File;
import  java.io.IOException;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.PropertySet;
import  org.apache.tools.ant.types.Reference;
import  org.apache.tools.ant.util.FileUtils;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.starters.StringItemListHandle;

/**
 * Create a new simple file. Useful for creating simple, small files or test input
 * files. Unlike the related {@linkplain MkTempFile} task, the files created by
 * MkFileTask persist by default. The other main difference between this task and
 * the MkTempFile task is that the caller specifies the new file's path-- it is not
 * dynamically generated by the AntX TempLocator.
 * <p>
 * Prototype lines (as specified with the &lt;line&gt; or &lt;lines&gt; elements)
 * are only applied to new empty files. If a prototype copy file (or resource) has
 * been defined, these lines are <em>not</em> included to the new file.
 * <p>
 * <b>Examples:</b><pre>
 *   &lt;<b>strings</b> id="default.ignoredfiles" prefix="*."&gt;
 *     &lt;string value="last"&gt;
 *     &lt;string value="jar"&gt;
 *     &lt;string value="gz"&gt;
 *     &lt;string value="tgz"&gt;
 *   &lt;/strings&gt;
 *   &lt;<b>newfile</b> path="${work}/lib/.cvsignore"&gt;
 *     &lt;line value="classes"/&gt;
 *     &lt;lines listref="default.ignoredfiles"/&gt;
 *   &lt;/newfile&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      MkTempFile
 **/

public class MkFileTask extends MkNewObject
{
    /**
     * Initializes a new MkFileTask instance.
     **/
    public MkFileTask()
    {
        super(AntX.mktemp);
    }


    /**
     * Initializes a new CV-labeled MkFileTask instance.
     **/
    protected MkFileTask(String iam)
    {
        super(iam);
    }


    /**
     * Defines the new file's path. If the path is relative the
     * new file will be relative to the project's base directory.
     * @param fp the file with full path (non-null)
     **/
    public void setPath(File fp)
    {
        require_(fp!=null,"setPath- nonzro path");
        m_filepath = fp;
    }


    /**
     * Returns file path of new files created by this task. Will
     * return <i>null</i> if never set. Required before this task
     * is executed.
     **/
    public File getPath()
    {
        return m_filepath;
    }


    /**
     * Returns absolute file path of new files created by this task.
     * Resolves the script-specified {@linkplain #setPath path} against
     * this task's project's base directory if necessary. Will return
     * <i>null</i> if this task's path has not been defined.
     **/
    public File getEffectivePath()
    {
        if (getPath()==null || getProject()==null) {
            return null;
        }
        return getProject().resolveFile(getPath().getPath());
    }


    /**
     * Defines whether this task will mark all new files as
     * delete-on-exit. Is <i>false</i> by default.
     * @param persist <i>false</i> if items should be deleted
     * @see MkTempFile
     **/
    public void setPersist(boolean persist)
    {
        m_autoDelete = !persist;
    }


    /**
     * Returns <i>true</i> if this task will try to mark all
     * temporary objects (files and directories) as delete-on-exit.
     **/
    public boolean isAutoDelete()
    {
        return m_autoDelete;
    }


    /**
     * Tells this tasks whether existing files should be replaced
     * with fresh copies.
     * @param stompOnOlder <i>true</i> if existing files will be overwritten
     **/
    public void setOverwrite(boolean stompOnOlder)
    {
        m_overWrite = stompOnOlder;
    }


    /**
     * Returns <i>true</i> if existing files will be overwritten with
     * new (fresh) copies. Defaults to <i>true</i>.
     **/
    public boolean willOverwrite()
    {
        return m_overWrite;
    }


    /**
     * Grunt work of creating the new file in specified location.
     * If a prototype source file/resource is defined, its contents
     * are copied to new file.
     * @param newFile the new file's location (absolute path)
     * @throws BuildException if unable to create new file
     **/
    protected final void createFile(File newFile) throws BuildException
    {
        require_(newFile!=null,"newFile- nonzro fil");
        FileUtils fsu= getFileUtils();

        try {
            if (getPrototypePlainFile()!=null) {
                fsu.copyFile(getPrototypePlainFile(),newFile,
                             getCopyFilters(),willOverwrite());
            }
            else if (getPrototypeResourceFile()!=null) {
                fsu.copyFile(getPrototypeResourceFile(),newFile,
                             getCopyFilters(),willOverwrite());
            } else {
                if (willOverwrite() || !newFile.exists()) {
                    newFile.createNewFile();
                    copyPrototypeLines(newFile,false);
                }
            }
        } catch(IOException iox) {
            throw new BuildException(iox, getLocation());
        }

        if (isAutoDelete()) {
            newFile.deleteOnExit();
        }
    }


    /**
     * Adds a new prototype line to include in new file.
     * @param line the line information (non-null)
     **/
    public void addLine(InnerString line)
    {
        require_(line!=null,"addLn- nonzro ln");
        getPrototypeLinesNoNull().add(line);
    }


    /**
     * Adds a reference to prototype lines to include in new file.
     * @param lines the line information (non-null)
     **/
    public void addLines(StringItemListHandle lines)
    {
        require_(lines!=null,"addLns- nonzro lns");
        getPrototypeLinesNoNull().add(lines);
    }


    /**
     * Shortcut for {@linkplain #addLines addLines()}. The
     * refered-to list is <em>added</em> to this task's prototype
     * lines.
     * @param ref the itemlist's reference (non-null)
     **/
    public final void setLines(Reference ref)
    {
        StringItemListHandle lines = new StringItemListHandle(ref);
        getPrototypeLinesNoNull().add(lines);
    }


    /**
     * Adds a new property set to include in new file.
     * @param propertyset new propertyset line (non-null)
     * @since JWare/AntX 0.4
     **/
    public void addConfiguredPropertySet(PropertySet propertyset)
    {
        require_(propertyset!=null,"addPropSet- nonzro pset");
        getPrototypeLinesNoNull().add(propertyset);
    }


    /**
     * Tries to create a new file in script-specified location.
     * @throws BuildException if unable to create or overwrite file
     **/
    public void execute()
    {
        verifyCanExecute_("execute");
        File newFile = getEffectivePath();
        createFile(newFile);
        saveFinalPath(newFile,false);
    }


    /**
     * Ensures this task is part of a valid project and that the
     * new file's path has been specified.
     * @throws BuildException if verification fails
     **/
    protected void verifyCanExecute_(String calr)
    {
        verifyInProject_(calr);
        if (getPath()==null) {
            String error = uistrs().get("task.needs.this.attr",getTaskName(),"path");
            log(error,Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
    }


    private File m_filepath;
    private boolean m_overWrite=true;//NB:stomps-by-default
    private boolean m_autoDelete;//NB:persists-by-default
}

/* end-of-MkFileTask.java */
