/**
 * $Id: Setting.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2003 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.helpers;

import  org.apache.tools.ant.types.EnumeratedAttribute;

/**
 * Enumeration of settings: on, off, default, inherited, and undefined.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2003 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public final class Setting extends EnumeratedAttribute
{
    /** Index of {@linkplain #OFF OFF}. **/
    public static final int OFF_INDEX= 0;
    /** Index of {@linkplain #ON ON}. **/
    public static final int ON_INDEX= 1;
    /** Index of {@linkplain #DEFAULT DEFAULT}. **/
    public static final int DEFAULT_INDEX= 2;
    /** Index of {@linkplain #INHERITED INHERITED}. **/
    public static final int INHERITED_INDEX= 3;
    /** Index of {@linkplain #UNDEFINED UNDEFINED}. **/
    public static final int UNDEFINED_INDEX= 4;

    /** Singleton "off" setting. **/
    public static final Setting OFF = new Setting("off",OFF_INDEX);
    /** Singleton "on" setting. **/
    public static final Setting ON = new Setting("on",ON_INDEX);
    /** Singleton "default" setting. **/
    public static final Setting DEFAULT = new Setting("default",DEFAULT_INDEX);
    /** Singleton "inherited" setting. **/
    public static final Setting INHERITED = new Setting("inherited",INHERITED_INDEX);
    /** Singleton "undefined" setting. **/
    public static final Setting UNDEFINED = new Setting("undefined",UNDEFINED_INDEX);



    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public Setting()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensure it's initialized
     * as with default Ant Introspector helper thingy.
     **/
    private Setting(String v, int i)
    {
        super();
        Tk.initEnum(this,v);
        this.index = getIndex();//speeds up 'from'
    }


    /**
     * Returns copy of all possible settings as an ordered string
     * array. Implementation: must be in order specified in public
     * INDEX_xxx constants.
     **/
    public String[] getValues()
    {
        return new String[] {"off","on","default","inherited","undefined"};
    };



    /**
     * Helper that converts a scalar to a known setting.
     * @param i the index to be matched
     **/
    public static Setting from(int i)
    {
        if (i==ON.index)        { return ON; }
        if (i==OFF.index)       { return OFF; }
        if (i==DEFAULT.index)   { return DEFAULT; }
        if (i==INHERITED.index) { return INHERITED; }
        if (i==UNDEFINED.index) { return UNDEFINED; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known setting
     * index.
     * @param i the index to be matched
     * @param dflt the default Setting if necessary
     **/
    public static Setting from(int i, Setting dflt)
    {
        Setting st= from(i);
        return (st==null) ? dflt : st;
    }


    /**
     * Helper that converts a scalar to a known setting. Some synonyms
     * for on (true,yes) and off (false,no) as accepted. Returns <i>null</i>
     * if string unrecognized. String can be either setting's symbolic
     * name or its index.
     * @param s the string to be interpreted
     **/
    public static Setting from(String s)
    {
        if (s!=null && s.length()>1) {
            s = s.toLowerCase();
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if ("yes".equals(s))         { return ON; }
                if ("true".equals(s))        { return ON; }
                if (ON.value.equals(s))      { return ON; }

                if ("no".equals(s))          { return OFF; }
                if ("false".equals(s))       { return OFF; }
                if (OFF.value.equals(s))     { return OFF; }

                if (Strings.INHERIT.equals(s)) { return INHERITED; }
                if (INHERITED.value.equals(s)) { return INHERITED; }

                if (DEFAULT.value.equals(s))   { return DEFAULT; }
                if (UNDEFINED.value.equals(s)) { return UNDEFINED; }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(java.lang.String) from(String)} but
     * with a default value if value does not match any known
     * setting's name.
     * @param s the symbolic name to be matched
     * @param dflt the default Setting if necessary
     **/
    public static Setting from(String s, Setting dflt)
    {
        Setting st= from(s);
        return (st==null) ? dflt : st;
    }


    /**
     *  Returns <i>true</i> if this setting equals incoming
     * object (also a setting).
     **/
    public boolean equals(Object o)
    {
        if (o==this) { return true; }
        if (o==null) { return false; }
        if (o.getClass()==getClass()) {
            return ((Setting)o).getIndex()==this.getIndex();
        }
        return false;
    }



    /**
     * Returns this setting's hash value; fixed when
     * constructed.
     **/
    public int hashCode()
    {
        return this.value.hashCode();
    }

    private int index;
}

/* end-of-Setting.java */
