/**
 * $Id: SIDs.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 1997-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.helpers;

import  java.security.NoSuchAlgorithmException;
import  java.security.SecureRandom;

/**
 * Administrator of a set of pseudo-unique numeric identifers. Components that need a
 * context-specific, unique key identifier will find this class useful. If a test needs
 * more secure ids it should look at the java.rmi.server package for VMIDs
 * and/or GIDs.
 * <p>
 * The SIDs class uses a <span class="src">SecureRandom</span> as the foundation of
 * of its implementation. By default it will try to locate one of two default secure
 * random number generator algorithms ("<span class="src">SHA1PRNG</span>" and
 * "<span class="src">IBMSecureRandom</span>"). These two algorithms represent the
 * dominant JREs (Sun's and IBM's). To force SIDs to use another custom algorithm you
 * must install the SPI for the algorithm and define the system property
 * "<span class="src">jware.securerandom.algo</span>" to the algorithm's common name.
 *
 * @since    JWare/core 0.2
 * @author   ssmc, &copy;1997-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   impl,helper
 **/

public final class SIDs
{
    /** The system property that customizes the SecureRandom algorithm. **/
    public static final String SECURERANDOM_ALGORITHM_PROPERTY=
        "jware.securerandom.algo";


    /** Check if a given SPI/algorithm is available. **/
    private static SecureRandom getSRNG(String algo)
    {
        SecureRandom srng=null;
        try {
            srng= SecureRandom.getInstance(algo);
        } catch (NoSuchAlgorithmException algox) {
            return null;
        }
        srng.setSeed(System.currentTimeMillis());
        return srng;
    }


    /** A secure pseudo-random number generator (PRNG). Tries to two main ones
        automatically if not explicitly defined by application. **/
    private static final SecureRandom PRNG;
    static {
        SecureRandom srng=null;
        String preferred = System.getProperty(SECURERANDOM_ALGORITHM_PROPERTY);
        if (preferred!=null) {
            srng = getSRNG(preferred);
            if (srng==null) {
                throw new RuntimeException("Missing SPRNG algorithm: "+preferred);
            }
        } else {
            srng = getSRNG("SHA1PRNG");//Sun's jre
            if (srng==null) {
                srng = getSRNG("IBMSecureRandom");//IBM's jre
                if (srng==null) {
                    throw new RuntimeException("Missing SPRNG algorithm: SHA1PRNG");
                }
            }
        }
        PRNG= srng;
    }


    /** Shared scratch for generating string-based identifiers. **/
    private static StringBuffer m_sb= new StringBuffer(32);


    /**
     * Strategy that determines if a SID is unique within some context.
     **/
    public interface Finder {
        public boolean exists(String sid);
    }


    /**
     * Returns the next pseudo-random SIDs long integer. Both
     * positive and negative numbers are returned.
     **/
    public static long nextLong()
    {
        return PRNG.nextLong();
    }


    /**
     * Returns the next pseudo-random SIDs integer. Both
     * positive and negative numbers are returned.
     **/
    public static int nextInt()
    {
        return PRNG.nextInt();
    }


    /**
     * Returns the next pseudo-random <em>positive</em> SIDs integer
     * within a particular range. Only integers between 0(inclusive)
     * and <i>limit</i> (exclusive) are returned.
     * @return an integer (<i>0</i>..<i>limit</i>]
     **/
    public static int nextInt(int limit)
    {
        return PRNG.nextInt(limit);
    }


    /**
     * Returns a new identifier with default prefix "<i>RUT</i>;" for example,
     * "<i>RUT123456</i>".
     **/
    public static String next()
    {
        synchronized(PRNG) {
            m_sb.setLength(0);
            m_sb.append(o_SYSID);
            m_sb.append(PRNG.nextInt(Integer.MAX_VALUE));//keep positive
            m_sb.append(c_SYSID);
            return m_sb.substring(0);//NB: a string with own copy of chars!
        }
    }


    /**
     * Returns a new identifier with a customized prefix. Prefix is
     * typically at least three characters long.
     * @param pfx custom prefix (non-null)
     * @throws java.lang.IllegalArgumentException if prefix is <i>null</i>
     **/
    public static String next(String pfx)
    {
        if (pfx==null) {
            throw new IllegalArgumentException();
        }
        synchronized(PRNG) {
            m_sb.setLength(0);
            m_sb.append(pfx);
            m_sb.append(PRNG.nextInt(999999));
            return m_sb.substring(0);//NB: a string with own copy of chars!
        }
    }


    /**
     * Returns a new identifier that doesn't already exist in caller's
     * context.
     * @param uv the uniqueness checker for caller's context (non-null)
     * @throws java.lang.IllegalArgumentException if <i>uv</i> finder is <i>null</i>
     **/
    public static String next(Finder uv)
    {
        if (uv==null) {
            throw new IllegalArgumentException();
        }
        synchronized(PRNG) {
            String s;
            do {
                s= null;
                m_sb.setLength(0);
                m_sb.append(PRNG.nextInt(999999999));
                s = m_sb.substring(0);
            } while (uv.exists(s));
            return s;//NB: a string with own copy of chars!
        }
    }


    /**
     * Returns a new custom-prefixed identifier that doesn't already
     * exist in caller's context. Prefix is typically at least three
     * (3) characters long.
     * @param uv the uniqueness checker for caller's context (non-null)
     * @param pfx the prefix to append before unique part
     * @throws java.lang.IllegalArgumentException if either parameter is <i>null</i>
     **/
    public static String next(Finder uv, String pfx)
    {
        if (uv==null || pfx==null) {
            throw new IllegalArgumentException();
        }
        synchronized(PRNG) {
            String s;
            do {
                s= null;
                m_sb.setLength(0);
                m_sb.append(pfx);
                m_sb.append(PRNG.nextInt(999999));
                s = m_sb.substring(0);
            } while (uv.exists(s));
            return s;//NB: a string with own copy of chars!
        }
    }


    /**
     * Returns a given number of seed bytes as defined by the SIDs
     * underlying secure random implementation's seed generation algorithm.
     * The returned bytes can be used to seed other generators. Never
     * returns <i>null</i>
     * @param numBytes the number of seed bytes to generate
     * @since JWare/core 0.7
     **/
    public static final byte[] nextSeed(int numBytes)
    {
        return PRNG.generateSeed(numBytes);
    }



    private SIDs()
    {/*not-allowed,algorithms-only*/}


    private static final String o_SYSID= "RUT{";
    private static final String c_SYSID= "}";
}

/* end-of-SIDs.java */
