/**
 * $Id: InputFileLoader.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 1997-2003 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.helpers;

import  java.io.BufferedInputStream;
import  java.io.File;
import  java.io.FileNotFoundException;
import  java.io.FileOutputStream;
import  java.io.InputStream;
import  java.io.IOException;
import  java.net.URL;
import  java.net.URLConnection;
import  java.util.Properties;

/**
 * Helper that loads contents of an input stream into a memory-based byte buffer.
 *
 * @since    JWare/core 0.5
 * @author   ssmc, &copy;1997-2003 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,helper
 **/

public class InputFileLoader
{
    /**
     * The default transfer buffer size ~8K.
     **/
    private static final int BUFSIZ=8*1024;


    /**
     * Symbolic integer that means "NO SIZE LIMIT" on load.
     **/
    public static final int NOLIMIT= Integer.MIN_VALUE;


    /**
     * Symbolic boolean that means "NO CACHED DATA" on load.
     **/
    public static final boolean NOCACHE= false;



    /**
     * Creates new input loader.
     **/
    public InputFileLoader()
    {
        this(0);
    }


    /**
     * Creates new input loader for outer controlling object.
     **/
    public InputFileLoader(int bufsiz)
    {
        if (bufsiz<=0) {
            bufsiz=BUFSIZ;
        }
        m_transferBuffer= new byte[bufsiz];
    }


    /**
     * Loads contents of an inputstream into a new byte[] buffer with a
     * precautionary limit to the amount of information that should be read
     * into memory. Will return an empty byte[] if the stream is empty.
     * @param ins source stream (non-null)
     * @param limit the maximum number of bytes to read from stream (see {@linkplain #NOLIMIT})
     * @throws IOException if unable to load stream's contents
     **/
    public final byte[] load(InputStream ins, int limit)
        throws IOException
    {
        verify_(ins!=null,"load- nonzro inputstrm");
        byte[] bytes;

        synchronized(m_transferBuffer) {
            BufferedInputStream bins;

            if (ins instanceof BufferedInputStream) {
                bins= (BufferedInputStream)ins;
            } else {
                bins= new BufferedInputStream(ins);
            }
            bytes = transferBytes(bins,limit);
            bins = null;
        }

        return bytes;
    }


    /**
     * Loads all contents of an inputstream into a new byte[] buffer. Will
     * return an empty byte[] if the stream is empty.
     * @param ins source stream (non-null)
     * @throws IOException if unable to completely load stream's contents
     **/
    public final byte[] load(InputStream ins)
        throws IOException
    {
        return load(ins,NOLIMIT);
    }


    /**
     * Loads contents of a fetched URL into a new byte[] buffer with a
     * precautionary limit to the amount of information that should be read
     * into memory. Will return an empty byte[] if the URL returns no
     * content (but is successful).
     * @param url source URL
     * @param limit the maximum number of bytes to read from stream (see {@linkplain #NOLIMIT})
     * @param cacheOK <i>true</i> if caching data ok (see {@linkplain #NOCACHE})
     * @throws IOException if unable to load URL contents
     **/
    public final byte[] loadURL(URL url, int limit, boolean cacheOK)
        throws IOException
    {
        verify_(url!=null,"loadURL- nonzro URL");
        byte[] bytes;

        // We synchronize here to prevent concurrent calls from opening their
        // streams only to have to wait. (Avoids timeouts on connection.)
        synchronized(m_transferBuffer) {
            try {
                URLConnection urlc = url.openConnection();
                urlc.setDefaultUseCaches(cacheOK);
                urlc.setAllowUserInteraction(false);

                BufferedInputStream ins = new BufferedInputStream(urlc.getInputStream());
                bytes = transferBytes(ins,limit);

                Tk.closeQuietly(ins);
                ins = null;
                urlc= null;
            }
            catch(Exception anyx) {
                if (anyx instanceof IOException) {
                    throw (IOException)anyx;
                }
                throw new IOException("Unable to load: "+url+"("+anyx.getMessage()+")");//FIXME:
            }
        }//lock

        return bytes;
    }



    /**
     * Loads contents of a fetched URL into a new byte[] buffer using
     * cached information if possible. A precautionary limit to the amount of
     * information that should be read into memory can also be defined. Will
     * return an empty byte[] if the URL returns no content (but is successful).
     * @param url source URL
     * @param limit the maximum number of bytes to read from stream (see {@linkplain #NOLIMIT})
     * @throws IOException if unable to load URL contents
     **/
    public final byte[] loadURL(URL url, int limit)
        throws IOException
    {
        return loadURL(url,limit,!NOCACHE);
    }



    /**
     * Loads all contents of a fetched URL into a new byte[] buffer. Will
     * return an empty byte[] if the URL returns no content (but is successful).
     * @param url source URL
     * @throws IOException if unable to completely load URL contents
     **/
    public final byte[] loadURL(URL url)
        throws IOException
    {
        return loadURL(url,NOLIMIT,!NOCACHE);
    }




    /**
     * Loads contents of an established URL stream into a new local file with a
     * precautionary limit to the amount of information that should be read
     * into new file. 
     * @param urlc established connection to source (non-null)
     * @param file destination file (non-null)
     * @param limit the maximum number of bytes to read from stream (see {@linkplain #NOLIMIT})
     * @throws IOException if unable to load stream's contents
     * @since JWare/AntX 0.5
     **/
    public final void loadURLToFile(URLConnection urlc, File file, int limit)
        throws IOException
    {
        verify_(urlc!=null && file!=null,"loadURLToFile- nonzro URLcon and file");

        // We synchronize here to prevent concurrent calls from opening their
        // streams only to have to wait. (Avoids timeouts on connection.)
        synchronized(m_transferBuffer) {
            try {
                BufferedInputStream ins = new BufferedInputStream(urlc.getInputStream());
                transferBytes(ins,limit,file);
                Tk.closeQuietly(ins);
                ins = null;
            }
            catch(Exception anyx) {
                if (anyx instanceof IOException) {
                    throw (IOException)anyx;
                }
                throw new IOException("Unable to load to stream: "+urlc.getURL()+
                                      "("+anyx.getMessage()+")");
            }
        }//lock
    }



    /**
     * Loads all contents of a n established URL stream into a new byte[] buffer.
     * Will return an empty byte[] if the connection yields no content (but is
     * successfully read).
     * @param urlc established connection to source (non-null)
     * @param file destination file (non-null)
     * @throws IOException if unable to completely load stream's contents
     * @since JWare/AntX 0.5
     **/
    public final void loadURLToFile(URLConnection urlc, File file)
        throws IOException
    {
        loadURLToFile(urlc,file,NOLIMIT);
    }



    /**
     * Loads contents of a fetched URL into a new local file with a
     * precautionary limit to the amount of information that should be read
     * into new file. 
     * @param url source URL (non-null)
     * @param file destination file (non-null)
     * @param limit the maximum number of bytes to read from stream (see {@linkplain #NOLIMIT})
     * @param cacheOK <i>true</i> if caching data ok (see {@linkplain #NOCACHE})
     * @throws IOException if unable to load URL contents
     * @since JWare/AntX 0.5
     **/
    public final void loadURLToFile(URL url, File file, int limit, boolean cacheOK)
        throws IOException
    {
        verify_(url!=null && file!=null,"loadURLToFile- nonzro URL and file");

        // We synchronize here to prevent concurrent calls from opening their
        // streams only to have to wait. (Avoids timeouts on connection.)
        synchronized(m_transferBuffer) {
            try {
                URLConnection urlc = url.openConnection();
                urlc.setDefaultUseCaches(cacheOK);
                urlc.setAllowUserInteraction(false);

                BufferedInputStream ins = new BufferedInputStream(urlc.getInputStream());
                transferBytes(ins,limit,file);

                Tk.closeQuietly(ins);
                ins = null;
                urlc= null;
            }
            catch(Exception anyx) {
                if (anyx instanceof IOException) {
                    throw (IOException)anyx;
                }
                throw new IOException("Unable to load to file: "+url+"("+anyx.getMessage()+")");
            }
        }//lock
    }



    /**
     * Loads contents of a fetched URL into a new local file using
     * cached information if possible. A precautionary limit to the amount of
     * information that should be read into the file can also be defined.
     * @param url source URL (non-null)
     * @param file destination file (non-null)
     * @param limit the maximum number of bytes to read from stream (see {@linkplain #NOLIMIT})
     * @throws IOException if unable to load URL contents
     * @since JWare/AntX 0.5
     **/
    public final void loadURLToFile(URL url, File file, int limit)
        throws IOException
    {
        loadURLToFile(url,file,limit,!NOCACHE);
    }



    /**
     * Loads all contents of a fetched URL into a new byte[] buffer. Will
     * return an empty byte[] if the URL returns no content (but is successful).
     * @param url source URL (non-null)
     * @param file destination file (non-null)
     * @throws IOException if unable to completely load URL contents
     * @since JWare/AntX 0.5
     **/
    public final void loadURLToFile(URL url, File file)
        throws IOException
    {
        loadURLToFile(url,file,NOLIMIT,!NOCACHE);
    }



    /**
     * Loads contents of a file into a new byte[] buffer with a precautionary
     * limit to the amount of information that should be read into memory. Usually
     * the file descriptor is either a URL or a (same-system) file path.
     * @param descriptor path to inputfile information (URL or file path)
     * @param limit the maximum number of bytes to read from stream (see {@linkplain #NOLIMIT})
     * @return contents of file specified by descriptor
     * @throws IOException if unable to load file's contents
     **/
    public byte[] loadFile(String descriptor, int limit)
        throws IOException
    {
        verify_(!Tk.isWhitespace(descriptor), "loadFil- nonzro descrip");

        URL url=null;
        if (descriptor.indexOf("://")>0) {
            url= new URL(descriptor);
        }
        else if (descriptor.toLowerCase().startsWith("file:") ||
                 descriptor.toLowerCase().startsWith("jar:")) {
            //Try URL conversion anyway; some windows URLs are in form
            //"file:\<disk>:\<path>"; also jar urls are a funny format.
        }
        else {
            File f= new File(descriptor);
            try {
                if (f.canRead()) {
                    url= f.toURL();
                } else {
                    throw new FileNotFoundException(f.getPath());
                }
            } catch(SecurityException secx) {
                throw new IOException("Security exception caught: "+secx.getMessage());
            }
        }
        verify_(url!=null,"lod- nonzro url");
        return loadURL(url,limit,!NOCACHE);
    }


    /**
     * Loads all contents of an file into a new byte[] buffer. Usually file
     * descriptors are either URLs or (same-system) files.
     * @param descriptor path to inputfile information (URL or file name)
     * @return contents of file specified by descriptor
     * @throws IOException if unable to completely load file's contents
     **/
    public byte[] loadFile(String descriptor)
        throws IOException
    {
        return loadFile(descriptor,NOLIMIT);
    }



    /**
     * Tries to load contents of class resource file. Falls back to the system
     * class path if resource not found in class's package.
     * @param resource name of resource (non-null)
     * @param forClass the resource-owning starting class (non-null)
     * @throws java.io.FileNotFoundException if unable to find resource
     * @throws java.io.IOException if unable to completely load resource's contents
     **/
    public byte[] loadResource(String resource, Class forClass)
        throws IOException
    {
        verify_(!Tk.isWhitespace(resource), "loadRez- nonzro resname");
        verify_(forClass!=null,"loadRez- nonzro clas");

        InputStream rsrc = forClass.getResourceAsStream(resource);
        if (rsrc==null) {
            rsrc = ClassLoader.getSystemResourceAsStream(resource);
        }
        if (rsrc==null) {
            throw new FileNotFoundException("Class= "+forClass.getName()+", Resource= "+resource);
        }

        try {
            return load(rsrc,NOLIMIT);
        } finally {
            Tk.closeQuietly(rsrc);
        }
    }


    /**
     * Loads all contents of the properties file at a URL into an existing
     * <span class="src">Properties</span> reference. Assumes encoding of source
     * properties information is ISO8851-1.
     * @param url source URL (non-null)
     * @param properties [optional] properties to be updated (use <i>null</i> for new)
     * @param cacheOK <i>true</i> if URL can be read from server caches (see {@linkplain #NOCACHE})
     * @throws IOException if unable to completely load URL contents
     **/
    public final Properties loadProperties(URL url, Properties properties, boolean cacheOK)
        throws IOException
    {
        verify_(url!=null,"loadProps- nonzro URL");

        if (properties==null) {
            properties = new Properties();
        }

        try {
            URLConnection urlc = url.openConnection();
            urlc.setDefaultUseCaches(cacheOK);
            urlc.setAllowUserInteraction(false);

            properties.load(urlc.getInputStream());
            urlc= null;
        }
        catch(Exception anyx) {
            if (anyx instanceof IOException) {
                throw (IOException)anyx;
            }
            throw new IOException("Unable to load Properties: "+url+"("+anyx.getMessage()+")");
        }

        return properties;
    }


    /**
     * Labor of transferring read bytes from a buffer input to the outgoing
     * byte array. Returns an empty byte[] for an empty input stream.
     **/
    private byte[] transferBytes(BufferedInputStream ins, int limit)
        throws IOException
    {
        byte[] bytes=null;
        int cc;

        // We assume incoming data to be less-than or ~8K of data for this
        // not be be grossly inefficient (ssmc).
        final int maxChars = m_transferBuffer.length;
        if (limit<=0) {
            limit = Integer.MAX_VALUE-1;
        }
        while ((cc=ins.read(m_transferBuffer,0,maxChars))!= -1) {
            if (cc==0) { //?hmmm?
                continue;
            }
            if (bytes==null) {
                bytes= new byte[cc];
                System.arraycopy(m_transferBuffer,0,bytes,0,cc);
            } else {
                int newcc= cc+bytes.length;
                byte[] newbytes= new byte[newcc];
                System.arraycopy(bytes,0,newbytes,0,bytes.length);
                bytes=null;
                System.arraycopy(m_transferBuffer,0,newbytes,newcc-cc,cc);
                bytes= newbytes;
            }
            if (bytes.length>=limit) {
                break;
            }
        }
        if (bytes==null) {
            bytes = new byte[0];
        }
        return bytes;
    }




    /**
     * Labor of transferring read bytes from a buffer input to the outgoing
     * byte array. Returns an empty byte[] for an empty input stream.
     * @since JWare/AntX 0.5
     **/
    private void transferBytes(BufferedInputStream ins, int limit, File output)
        throws IOException
    {
        FileOutputStream fos = new FileOutputStream(output);
        int cc, cctotal=0;

        // We assume incoming data to be less-than or ~8K of data for this
        // not be be grossly inefficient (ssmc).
        final int maxChars = m_transferBuffer.length;
        if (limit<=0) {
            limit = Integer.MAX_VALUE-1;
        }
        try {
            while ((cc=ins.read(m_transferBuffer,0,maxChars))!= -1) {
                if (cc==0) { //?hmmm?
                    continue;
                }
                fos.write(m_transferBuffer,0,cc);
                fos.flush();
                cctotal += cc;
                if (cctotal>=limit) {
                    break;
                }
            }
        } finally {
            Tk.closeQuietly(fos);
        }
    }




    /**
     * Convenience; facilitates frequent need to load test files.
     * @param url the source to be loaded (non-null)
     * @throws IOException if unable to load string
     **/
    public final static String loadString(URL url)
        throws IOException
    {
        InputFileLoader fl= new InputFileLoader();
        byte[] bytes = fl.loadURL(url,NOLIMIT,NOCACHE);
        String s = bytes.length>0 ? new String(bytes) : "";
        fl.dispose();
        fl= null;
        bytes= null;
        return s;
    }


    /**
     * Convenience; facilitates frequent need to load test files.
     * @param filename the file to be loaded (non-null)
     * @throws IOException if unable to load string
     **/
    public final static String loadString(String filename)
        throws IOException
    {
        InputFileLoader fl= new InputFileLoader();
        byte[] bytes = fl.loadFile(filename);
        String s = bytes.length>0 ? new String(bytes) : "";
        fl.dispose();
        fl= null;
        bytes= null;
        return s;
    }


    /**
     * Convenience; facilitates frequent need to load tests files.
     * @param file the file to be loaded (non-null,can-read)
     * @throws IOException if unable to load string
     **/
    public final static String loadString(File file)
        throws IOException
    {
        return loadString(file.getAbsolutePath());
    }


    /**
     * Convenience; facilitates frequent need to load test files.
     * @param resource class resource name (non-null)
     * @param forClass the owning class (non-null)
     * @throws IOException if unable to load string
     **/
    public final static String loadString(String resource, Class forClass)
        throws IOException
    {
        InputFileLoader fl= new InputFileLoader();
        byte[] bytes = fl.loadResource(resource, forClass);
        String s = bytes.length>0 ? new String(bytes) : "";
        fl.dispose();
        fl= null;
        bytes= null;
        return s;
    }


    /**
     * Convenience; facilitates frequent need to load test properties files.
     * @param url the source to be loaded (non-null)
     * @param p [optional] existing properties to be updated (use <i>null</i> for new)
     * @throws IOException if unable to load Properties
     **/
    public final static Properties loadProperties(URL url, Properties p)
        throws IOException
    {
        return new InputFileLoader().loadProperties(url,p,NOCACHE);
    }


    /**
     * Clears this loader's underlying scratch buffer. This loader
     * is no longer valid after this method returns.
     **/
    public void dispose()
    {
        m_transferBuffer = null;
    }


    /**
     * Zapped assertable api.
     **/
    private void verify_(boolean c, String msg)
    {
        if (!c) {
            throw new IllegalArgumentException("REQUIRE: InputFileLoader: "+msg);
        }
    }


    private byte[] m_transferBuffer;//$mem cost
}

/* end-of-InputFileLoader.java */
