/**
 * $Id: InnerNameValuePair.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.helpers;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.apis.Nameable;

/**
 * Brain-dead luggage Name-Value bean that allows nested elements and other simple
 * "key-value holder" elements. The name-value pair's value should be specified
 * as either a single "<span class="src">value</span>" attribute or as nested text.
 * Note that these name-value pairs have empty string names and values by default. To
 * clear a pair's name and/or value to <i>null</i> you must set it to <i>null</i>
 * explicitly (see {@linkplain #setNull()}.
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public class InnerNameValuePair implements Nameable, Cloneable
{
    /**
     * Creates new empty pair (name and value are empty).
     **/
    public InnerNameValuePair()
    {
    }


    /**
     * Creates new predefined pair.
     **/
    public InnerNameValuePair(String name, String value)
    {
        setNV(name,value);
    }



    /**
     * Returns independent clone of this name-value pair.
     * @since JWare/AntX 0.5
     **/
    public Object clone()
    {
        try {
            return super.clone();
        } catch(CloneNotSupportedException clnX) {
            throw new Error();
        }
    }



    /**
     * <em>Replaces</em> this pair's current name.
     * @param s the new name (non-null)
     **/
    public void setName(String s)
    {
        m_name = s;
    }



    /**
     * Returns this pair's current name. Will return
     * the empty string if this property never set.
     **/
    public String getName()
    {
        return m_name;
    }



    /**
     * <em>Replaces</em> this pair's current value.
     * @param s the new value (non-null)
     **/
    public void setValue(String s)
    {
        m_value = s;
    }



    /**
     * Adds more text to this pair's current value.
     * @param text additional text (non-null)
     **/
    public void addText(String text)
    {
        m_value += text;
    }



    /**
     * Returns this pair's current value.
     **/
    public String getValue()
    {
        return m_value;
    }



    /**
     * Returns this pair's current value with property substition
     * first applied.
     * @param P project from which property values read
     **/
    public String getValue(final Project P)
    {
        String s = getValue();
        if (P!=null) {
            s = Tk.resolveString(P,s);
        }
        return s;
    }



    /**
     * Returns this pair's current value with property substition
     * first applied.
     * @param P project from which property values read
     * @param altForm alternative form of property references allowed
     * @since JWare/AntX 0.5
     **/
    public String getValue(final Project P, boolean altForm)
    {
        String s = getValue();
        if (P!=null) {
            s = Tk.resolveString(P,s,altForm);
        }
        return s;
    }



    /**
     * Forces both this item's name and its value to <i>null</i>.
     * @since JWare/AntX 0.5
     **/
    public void setNull()
    {
        m_name = null;
        m_value = null;
    }



    /**
     * Utility method that sets both name and value in a
     * single call. Incoming values can be <i>null</i>.
     **/
    public void setNV(String name, String value)
    {
        m_name = name;
        m_value = value;
    }



    /**
     * Returns a template string like: "<span class="src">name=value</span>"
     * where "<span class="src">name</span>" is this pair's current
     * name and "<span class="src">value</span>" is this pair's current
     * value.
     **/
    public String toString()
    {
        return ""+getName()+"="+getValue();
    }



    /**
     * Same as {@linkplain #toString toString()} but with embedded
     * properties replaced with supplied project's definitions.
     * @param P project from which properties resolved
     **/
    public String toString(final Project P)
    {
        String s = getValue();
        if (P!=null) {
            s = Tk.resolveString(P,s);
        }
        return ""+getName()+"="+s;
    }



    /**
     * Ensures this item has a non-null, non-empty name assigned.
     * @throws BuildException if name malformed.
     * @since JWare/AntX 0.5
     **/
    public void verifyNamed()
    {
        if (m_name==null || m_name.length()==0) {
            throw new BuildException("A non-empty name is required.");
        }
    }

    private String m_name="", m_value="";
}

/* end-of-InnerNameValuePair.java */
