/**
 * $Id: Locals.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.wrap;

import  java.util.Collections;
import  java.util.Iterator;
import  java.util.List;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.apis.Nameable;
import  com.idaremedia.antx.apis.Requester;


/**
 * Collection of fixture elements that should be either preserved by or limited to
 * an isolated task set. Variables are inherited from the enclosing context by
 * default; you must explicit unset pre-existing variables by setting the 
 * <span class="src">inherit</span> parameter <i>false</i>. Properties are <em>not</em>
 * inherited by default; you must explicitly copy pre-existing properties by
 * setting the <span class="src">inherit</span> parameter <i>true</i>. When default
 * values are assigned to local properties, these properties act as overlayed values
 * that replace any pre-existing settings unconditionally.
 * <p/>
 * Locals are applied en-masse at once. This means a local's definition cannot depend
 * on another local's definition since it's not guaranteed that the source item will
 * be initialized first. Of course locals can depend on pre-existing fixture values
 * including macro attribute values.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;macrodef name="fubar"&gt;
 *       &lt;attribute name="all" default="no"/&gt;
 *       &lt;sequential&gt;
 *          &lt;<b>locals name="fubar.locals"</b>&gt;
 *             &lt;property name="generate"/&gt;
 *             &lt;property name="allowtests" value="@{all}"/&gt;
 *          &lt;/locals&gt;
 *          ...
 *          &lt;isolate block="fubar.locals"&gt;
 *              &lt;do if="allowtests"&gt;
 *                 ...
 *              &lt;condition property="generate"&gt;
 *                 ...
 *          &lt;/isolate&gt;
 *       &lt;/sequential&gt;
 *    &lt;/macrodef&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 * @see       IsolatedTaskSet
 * @.impl     Because this whole notion of macro locals (and scoping in general)
 *            is destined to appear in Ant at some point, most of this class's API
 *            is kept package-private to ease transition to whatever becomes the
 *            standard Ant mechanism.
 **/

public final class Locals extends AssertableTask implements Nameable
{
    /**
     * Initialize a new locals bucket. This item's name is
     * defaulted to <span class="src">_macrolocals_</span>.
     **/
    public Locals()
    {
        super(AntX.flow+"Locals:");
    }



    /**
     * Marks this locals declaration as a blocking definition
     * or not. If blocking this locals items should be removed
     * from the enclosing project's context when the bubble
     * exits.
     * @param denied <i>true</i> if our items should be removed.
     **/
    final void setBlocking(boolean denied)
    {
        m_isDenied = denied;
    }



    /**
     * Returns <i>true</i> if this local's items should be removed
     * from the enclosing bubble's context before it exits.
     **/
    boolean isBlocking()
    {
        return m_isDenied;
    }



    /**
     * Sets this locals name so it can be referenced by a
     * bubble or net component.
     * @param name this locals handle (non-null)
     * @.sideeffect Will install this locals into the project's
     *   reference table using this name.
     */
    public void setName(String name)
    {
        require_(name!=null,"setName- nonzro name");
        m_name = name;
    }


    /**
     * Returns this locals item's name. Will never return
     * <i>null</i>; defaults to <span class="src">_macrolocals</span>.
     */
    public final String getName()
    {
        return m_name;
    }



    /**
     * Sets default inherit option for all of this local's
     * contained items. Individual items can override this 
     * setting.
     * @param inherit <i>true</i> to inherit
     **/
    public void setInherit(boolean inherit)
    {
        m_inherit = inherit ? Boolean.TRUE : Boolean.FALSE;
    }



    /**
     * Returns this local's inherit flag setting. Will return
     * <i>null</i> if never set explicitly.
     **/
    public final Boolean getInheritFlag()
    {
        return m_inherit;
    }



    /**
     * Returns a local item's inherit flag setting. Will return
     * <i>null</i> if neither the item not this bucket has 
     * an explicit setting.
     **/
    public final Boolean getInheritFlag(ConditionalLocal local)
    {
        Boolean f = local.getInheritFlag();
        if (f==null) {
            f = getInheritFlag();
        }
        return f;
    }



    /**
     * Adds a new property item to this locals bucket.
     * @return the new item
     **/
    public Object createProperty()
    {
        if (m_propertyItems==null) {
            m_propertyItems = AntXFixture.newList();
        }
        return newItem(m_propertyItems);
    }



    /**
     * Adds a new variable item to this locals bucket.
     * @return the new item
     **/
    public Object createVariable()
    {
        if (m_variableItems==null) {
            m_variableItems = AntXFixture.newList();
        }
        return newItem(m_variableItems);
    }



    /**
     * Common synonym for {@linkplain #createVariable}.
     * @return the new item
     **/
    public final Object createVar()
    {
        return createVariable();
    }



    /**
     * Ensures all local items have been named (at least). Values
     * are not required for local declarations.
     * @param calr execution method (non-null)
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);
        try {
            verifyNamed(m_propertyItems);
            verifyNamed(m_variableItems);
            //verifyNamed(m_referenceItems);
        } catch(BuildException bX) {
            log(bX.getMessage(),Project.MSG_ERR);
            if (bX.getLocation()==Location.UNKNOWN_LOCATION) {
                bX = new BuildException(bX,getLocation());
            }
            throw bX;
        }
    }


    /**
     * Ensures this locals item and all of its nested items are
     * named. Once verified, will install this locals into its
     * project's reference table using its name as the key.
     * @throws BuildException if another locals object already exists
     *     with same handle.
     * @throws BuildException if this object or any of its nested
     *     items are improperly named (or unnamed).
     */
    public void execute()
    {
        verifyCanExecute_("exec");

        Map ht = getProject().getReferences();
        synchronized(ht) {
            if (ht.containsKey(getName())) {
                String error = getAntXMsg("flow.locals.exists",getName());
                log(error,Project.MSG_ERR);
                throw new BuildException(error,getLocation());
            }
            ht.put(getName(),this);
        }
    }



    /**
     * Removes this locals item from the project's reference table. This 
     * method should be called immediately after a locals reference
     * is assigned to a bubble or net. This method does
     * <em>nothing</em> if this locals object is not installed.
     * @param clnt feedback controls.
     * @return <i>true</i> if removed from projects reference table.
     **/
    public final boolean uninstall(Requester clnt)
    {
        boolean ok=true;
        Map ht = getProject().getReferences();
        synchronized(ht) {
            Object o = ht.get(getName());
            if (o==this) {
                ht.remove(getName());
            } else {
                String warning = getAntXMsg("flow.locals.not.installed",getName());
                clnt.problem(warning,Project.MSG_WARN);
                ok = false;
            }
        }
        return ok;
    }



    List getProperties()
    {
        return m_propertyItems;
    }


    List getVariableNames()
    {
        return getItemNames(m_variableItems);
    }


    List getVariables()
    {
        return m_variableItems;
    }


    boolean hasVariables()
    {
        return m_variableItems!=null;
    }
    
    
    private ConditionalLocal newItem(List list)
    {
        ConditionalLocal item = new ConditionalLocal();
        list.add(item);
        return item;
    }


    private List getItemNames(List from)
    {
        List l = Collections.EMPTY_LIST;
        if (from!=null) {
            synchronized(from) {
                l = AntXFixture.newList(from.size());
                for (int i=0,N=from.size();i<N;i++) {
                    ConditionalLocal local = (ConditionalLocal)from.get(i);
                    if (local.isEnabled()) {
                        l.add(local.getName());
                    }
                }
            }
        }
        return l;
    }

    
    private void verifyNamed(List from)
    {
        if (from!=null) {
            synchronized(from) {
                Iterator itr= from.iterator();
                while (itr.hasNext()) {
                    ConditionalLocal local = (ConditionalLocal)itr.next();
                    local.verifyNamed();
                }
            }
        }
    }


    private String m_name = "_macrolocals_";
    private boolean m_isDenied;
    private List m_propertyItems,m_variableItems;
    private Boolean m_inherit;
}

/* end-of-Locals.java */