/**
 * $Id: IsolatedTaskSet.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.wrap;

import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.FixtureOverlay;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * A taskset that isolates the effects of its nested tasks away from the enclosing project.
 * Basically the standard AntX way to install a generic {@linkplain ExecutionBubble}.
 * <p>
 * <b>Example Usage:</b><pre>
 *   &lt;assert isnotset="a.property"/&gt;
 *   &lt;<b>isolate</b> warnproperty="whoopsies"&gt;
 *      &lt;property name="a.property" value="hi!"/&gt;
 *      &lt;assign name="a.time" op="now" scope="project"/&gt;
 *   &lt;/isolate&gt;
 *   &lt;assert isnotset="a.property"/&gt;
 *   &lt;assert isnotset="a.time" isa="reference"/&gt;
 *   &lt;assert isnotset="whoopsies" msg="No object issues found"/&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.4
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 * @see       Locals
 * @see       LocalExecutionBubble
 * @see       com.idaremedia.antx.solo.LocalFixtureTaskSet LocalFixtureTaskSet
 **/

public class IsolatedTaskSet extends TaskSet
    implements FixtureOverlay
{
    /**
     * Initializes a new isolated task set.
     **/
    public IsolatedTaskSet()
    {
        super(AntX.flow+"isolated");
    }



    /**
     * Initializes a new enclosed isolated task set.
     * @param iam CV-label
     **/
    public IsolatedTaskSet(String iam)
    {
        super(iam);
    }



    /**
     * Initializes a new subclassed isolated task set.
     * @param iam CV-label
     * @param delayConfigure <i>true</i> to delay configuration
     *        of nested items
     **/
    public IsolatedTaskSet(String iam, boolean delayConfigure)
    {
        super(iam,delayConfigure);
    }



    /**
     * Creates this taskset's execution bubble. Subclasses who
     * override this method <em>must</em> call this inherited
     * method before they access the bubble.
     * @see #newRunBubble newRunBubble(&#8230;)
     */
    public void init()
    {
        super.init();
        m_bubble = newRunBubble();
        m_bubble.setProject(getProject());
    }



    /**
     * Tells this isolated taskset to be carefully check project
     * references on exit. Will issue warnings for deleted or
     * changed references.
     * @param careful <i>true</i> to issue warnings for altered
     *        project fixture
     **/
    public void setCareful(boolean careful)
    {
        m_bubble.setCarefulObjectChecks(careful);
    }



    /**
     * Tells this isolated taskset to update a property (or a
     * set of properties) with warnings about changed
     * or deleted references. Because it's possible that multiple
     * categories of fixture items are changed, each fixture
     * category stores its warnings under a property that combines
     * the given property name and a category specific suffix;
     * for example, references store warnings under a property
     * named <span class="src">property-refs</span> while
     * context reset methods store warnings under a property named
     * <span class="src">property-<i>adminId</i></span> where
     * "<i>adminId</i>" is the name of the fixture administrator.
     * @param property name of property to update
     * @.sideeffect Turns on {@linkplain #setCareful careful} checking.
     **/
    public void setWarnProperty(String property)
    {
        m_bubble.setWarningsUpdateProperty(property);
        setCareful(true);
    }



    /**
     * Tells this isolated taskset to reset top-level iteration
     * fixture administrators. <em>DO NOT USE THIS OPTION UNLESS
     * YOU REALLY KNOW WHAT YOU'RE DOING.</em> While the ability
     * to reset fixture administrators is critical in test scripts,
     * it is almost never a good idea in a regular build situation.
     * Example Usage:<pre>
     *     &lt;isolate resetfixture="stringmanagers=default,outputrecorders=all"&gt;
     *        ...
     *     &lt;/isolate&gt;
     * </pre>
     * @param fixidList comma-delimited list of <i>component</i>=<i>aspect</i>
     *            pairs.
     **/
    public void setResetFixture(String fixidList)
    {
        require_(fixidList!=null,"resetFix- nonzro idlist");
        List l= Tk.splitList(fixidList);
        if (!l.isEmpty()) {
            int i;
            String fxid,aspect;
            for (Iterator itr=l.iterator();itr.hasNext();) {
                fxid= itr.next().toString();
                aspect = Strings.ALL;
                i= fxid.indexOf('=');
                if (i>0) {
                    aspect= fxid.substring(i+1);
                    fxid= fxid.substring(0,i);
                }
                m_bubble.addFixtureReset(fxid,aspect);
            }
        }
    }



    /**
     * Makes this task's bubble porous by allowing all modifications 
     * through except those named. See {@linkplain Locals}.
     * @param localsId name of locals definition (non-null)
     * @since JWare/AntX 0.5
     * @throws BuildException if unable to locate installed locals
     *    of given name.
     **/
    public final void setBlock(String localsId)
    {
        require_(localsId!=null,"setBlock- nonzro id");
        setFilters(localsId,true);
    }




    /**
     * Makes this task's bubble porous by allowing the named
     * modifications through. See {@linkplain Locals}.
     * @param localsId name of locals definition (non-null)
     * @since JWare/AntX 0.5
     * @throws BuildException if unable to locate installed locals
     *    of given name.
     **/
    public final void setAllow(String localsId)
    {
        require_(localsId!=null,"setAllow- nonzro id");
        setFilters(localsId,false);
    }




    /**
     * Tells this task whether default fixture exclusions should
     * be ignored. For backward compatibility, fixture exclusions 
     * are ignored unless explicitly turned on at the task level
     * or using the global defaults option.
     * @param letEmPassThru <i>true</i> to let exclusion thru
     * @since JWare/AntX 0.5
     * @see com.idaremedia.antx.Defaults
     **/
    public void setFixtureExcludes(boolean letEmPassThru)
    {
        m_bubble.setFixtureExcludes(letEmPassThru);
    }



    /**
     * Executes this taskset's nested tasks within an standard
     * execution bubble.
     */
    protected void performNestedTasks()
    {
        Requester me = new Requester.ForComponent(this);
        m_bubble.enter(me);
        try {
            performTheTasksList();
        } finally {
            m_bubble.leave(me);
        }
    }



    /**
     * Factory method for this taskset's local execution bubble.
     * Called by {@linkplain #init init()} method.
     * @see LocalExecutionBubble
     */
    protected LocalExecutionBubble newRunBubble()
    {
        return new LocalExecutionBubble();
    }




    /**
     * Common work involved to assign a set of filter controls
     * to this taskset's execution bubble.
     * @param localsId id for filter instructions (non-null)
     * @param blocking <i>true</i> if locals will be removed
     * @since JWare/AntX 0.5
     **/
    private void setFilters(String localsId, boolean blocking)
    {
        if (m_bubble.isFiltered()) {
            String error = getAntXMsg("task.one.or.other.attr","allow","deny");
            log(error, Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
        Locals locals = (Locals)getReferencedObject(null,localsId,Locals.class);
        locals.uninstall(new Requester.ForComponent(this));
        locals.setBlocking(blocking);
        m_bubble.setFilterControls(locals);
    }


    private LocalExecutionBubble m_bubble;
}


/* end-of-IsolatedTaskSet.java */
