/**
 * $Id: IfErrorTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.wrap;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ErrorSnapshot;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.apis.BuildAssertionException;
import  com.idaremedia.antx.ownhelpers.LocalTk;
import  com.idaremedia.antx.parameters.PropertySource;
import  com.idaremedia.antx.starters.StrictInnerTaskSet;
import  com.idaremedia.antx.starters.Quiet;

/**
 * Set of tasks to be executed iff an build exception is detected by a protected taskset.
 * Must be nested within a {@linkplain TolerantTaskSet TolerantTaskSet} like a
 * ProtectedTaskSet.
 * <p>
 * If you specify an error substitution (see the <span class="src">convertto</span>
 * option), this task will store the <em>converted</em> error to any snapshots and/or
 * project references. The original error will be stored as the new error's causing
 * throwable.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  n/a
 * @see      TolerantTaskSet
 **/

public final class IfErrorTask extends StrictInnerTaskSet implements Quiet
{
    /**
     * Creates a new iferror handler in a protected taskset.
     **/
    public IfErrorTask()
    {
        super(AntX.flow+"protect");
    }


    /**
     * Sets the name of property updated with either a predefined
     * {@linkplain #setFailValue value} or the thrown exception's
     * message.
     * @see #setFailValue
     **/
    public void setFailProperty(String property)
    {
        require_(property!=null,"setFailProp- nonzro propnam");
        m_failProperty = property;
    }


    /**
     * Returns name of the property to be updated iff a build
     * exception is detected by this task's enclosing protected
     * taskset. Returns <i>null</i> if never set.
     **/
    public final String getFailProperty()
    {
        return m_failProperty;
    }


    /**
     * Sets the name of an exported property updated with either
     * a predefined {@linkplain #setFailValue value} or the
     * thrown exception's message.
     * @see #setFailValue
     * @since JWare/AntX 0.2
     **/
    public void setFailVariable(String variable)
    {
        require_(variable!=null,"setFailVar- nonzro varnam");
        m_failVar = variable;
    }


    /**
     * Returns name of the exported property to be updated iff a build
     * exception is detected by this task's enclosing protected
     * taskset. Returns <i>null</i> if never set.
     **/
    public final String getFailVariable()
    {
        return m_failVar;
    }


    /**
     * Sets the value used to set the failure property if a
     * build exception is detected. By default, if requested,
     * the fail property is set to the exception's message.
     **/
    public void setFailValue(String value)
    {
        require_(value!=null,"setfailV- nonzro valu");
        m_failValue = value;
    }


    /**
     * Returns the explicit value to which the failure property
     * is set in event of detected build exception. Returns
     * <i>null</i> if never set.
     **/
    public final String getFailValue()
    {
        return m_failValue;
    }


    /**
     * Sets the name of an {@linkplain ErrorSnapshot ErrorSnapshot}
     * reference to be created iff a build exception is detected. This
     * name should not refer to an existing project reference.
     * @param reference name of <em>new</em> reference
     **/
    public void setCaptureSnapshotRefId(String reference)
    {
        m_failSnapshotRefId= reference;
    }


    /**
     * Returns the name of a <em>new</em> ErrorSnapshot reference
     * to be created iff a build exception is detected.
     **/
    public final String getCaptureSnapshotRefId()
    {
        return m_failSnapshotRefId;
    }


    /**
     * Alias for {@linkplain #setCaptureSnapshotRefId setCaptureSnapshotRefId}
     * that is typically used from build xml (easier element name).
     **/
    public final void setCaptureSnapshot(String newRefId)
    {
        setCaptureSnapshotRefId(newRefId);
    }


    /**
     * Set the name of an BuildException reference to be created iff
     * a build exception is detected. This name should not refer to
     * an existing reference (overwritten).
     **/
    public void setCaptureThrownRefId(String reference)
    {
        m_failThrownRefId = reference;
    }


    /**
     * Alias for {@linkplain #setCaptureThrownRefId setCaptureThrownRefId}
     * that is typically used from build xml (easier element name).
     **/
    public final void setCaptureThrown(String thrownRefId)
    {
        setCaptureThrownRefId(thrownRefId);
    }


    /**
     * Returns the name of the <em>new</em> BuildException reference
     * to be created iff a build exception is detected.
     **/
    public final String getCaptureThrownRefId()
    {
        return m_failThrownRefId;
    }


    /**
     * Sets a comma-delimited list of properties to be captured with
     * an error snapshot if an exception is detected. If the given
     * namesList is either one of the symbolic property domain names
     * like "<span class="src">script</span>" or "<span class="src">jre</span>"
     * all of the matching domain's properties are included in the
     * snapshot.
     * @param namesList comma-delimited list of names
     * @see #setCaptureSnapshotRefId
     **/
    public void setCapturePropertyNames(String namesList)
    {
        require_(namesList!=null,"setCapturPs- nonzro list");
        m_propertyNames= namesList;
    }


    /**
     * Returns comma-delimited list of properties to be captured
     * with an error snapshot in the event of a detected build
     * exception. Will return <i>null</i> if never set.
     **/
    public final String getCapturePropertyNames()
    {
        return m_propertyNames;
    }


    /**
     * Alias to {@linkplain #setCapturePropertyNames setCapturePropertyNames}
     * that is typically used from build xml (easier element name).
     **/
    public final void setCaptureProperties(String namesList)
    {
        setCapturePropertyNames(namesList);
    }



    /**
     * Tells this handler to convert the captured error to something
     * else on script's behalf.
     * @param what "new", "assert" or custom exception class name
     * @since JWare/AntX 0.4
     **/
    public void setConvertTo(String what)
    {
        require_(what!=null,"setCvtTo- nonzro target");
        m_convertTo= what;
    }



    /**
     * Tells this handler to be quiet when it captures an error. Will
     * prevent informative message from being written to log.
     * @param quiet <i>true</i> if be quiet.
     * @since JWare/AntX 0.5
     **/
    public final void setQuiet(boolean quiet)
    {
        m_tellCaught = quiet ? Boolean.FALSE : Boolean.TRUE;
    }



    /**
     * Returns this handler's script-supplied quiet flag. Will return
     * <i>null</i> if never set explicitly.
     * @since JWare/AntX 0.5
     **/
    public final Boolean getQuietFlag()
    {
        return m_tellCaught;
    }


    /**
     * Called by enclosing block to prepare this task for execution
     * because of a detected build exception, rtX. By default captures
     * any requested error snapshots and sets up failure properties.
     * @return the incoming exception or, if requested, a substitute.
     **/
    final RuntimeException throwOccured(RuntimeException rtX)
    {
        final Project P= getProject();
        TolerantTaskSet wrt= (TolerantTaskSet)getEnclosingTask();

        if (tellCaught()) {
            int loglevel = Project.MSG_ERR;
            if (!wrt.isHaltIfError()) {
                loglevel = Project.MSG_INFO;
            }
            log(uistrs().get("flow.caught.failure",
                             wrt.getOwningTarget().getName(), rtX.getMessage()),
                loglevel);
        }

        String what= getFailValue();
        if (what==null) {
            what= rtX.getMessage();
        }

        if (getFailProperty()!=null) {
            P.setNewProperty(getFailProperty(), what);
        }
        if (getFailVariable()!=null) {
            ExportedProperties.set(getFailVariable(), what);
        }

        rtX = getPreferredX(rtX);

        if (getCaptureThrownRefId()!=null) {
            String refid= getCaptureThrownRefId();

            if (P.getReference(refid)!=null) {
                log(uistrs().get("task.warn.refid.exists",refid),Project.MSG_WARN);
            }
            P.addReference(refid,rtX);
        }

        if (getCaptureSnapshotRefId()!=null) {
            String refid = getCaptureSnapshotRefId();

            if (P.getReference(refid)!=null) {
                log(uistrs().get("task.warn.refid.exists",refid),Project.MSG_WARN);
            }

            ErrorSnapshot es= new ErrorSnapshot(wrt, rtX);
            es.setName(refid);

            String nl= getCapturePropertyNames();
            if (nl!=null) {
                PropertySource domain = PropertySource.from(nl);
                if (domain!=null) {
                    es.captureProperties
                        (FixtureExaminer.copyOfProperties(domain,P));
                } else {
                    es.captureProperties(nl);
                }
            }
            P.addReference(refid,es);
        }

        return rtX;
    }



    /**
     * Replace the actual exception with a script requested
     * substitution if necessary. Will return the original if no
     * substitution requested or unable to create new exception.
     * @param rtX the causing exception (non-null)
     * @return script-requesed replacement or same if no such request
     * @since JWare/AntX 0.4
     **/
    private RuntimeException getPreferredX(RuntimeException rtX)
    {
        if (m_convertTo!=null) {
            if ("new".equals(m_convertTo)) {
                rtX = new BuildException(rtX.getMessage(),rtX,getLocation());
            }
            else
            if ("assert".equals(m_convertTo)) {
                if (!(rtX instanceof BuildAssertionException)) {
                    rtX = (RuntimeException)LocalTk.replaceError
                                (rtX, getLocation(), BuildAssertionException.class);
                }
            }
            else {
                rtX = (RuntimeException)LocalTk.replaceError
                                (rtX, getLocation(), m_convertTo);
            }
        }
        return rtX;
    }



    private boolean tellCaught()
    {
        return m_tellCaught==null || m_tellCaught==Boolean.TRUE;
    }


    private String  m_failProperty, m_failVar;
    private String  m_failValue;
    private String  m_failSnapshotRefId, m_failThrownRefId;
    private String  m_propertyNames;
    private String  m_convertTo;
    private Boolean m_tellCaught;//NB: null=>YES!
}

/* end-of-IfErrorTask.java */
