/**
 * $Id: SwitchTask.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2002-2004,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.match;

import  java.util.Iterator;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.TaskHandle;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;
import  com.idaremedia.antx.parameters.FlexValueSupport;
import  com.idaremedia.antx.starters.StrictOuterTask;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * A task that executes a named taskset that matches some selection criterion. The
 * selection criterion is defined like a Java-style '<i>switch</i>' statement:
 * the switch task's value is the source (or unknown) value; the possible
 * choice tasks values are the candidate matches (or known) values. Candidate tasksets
 * are all nested within a single owning SwitchTask and must all be {@linkplain ChoiceTask
 * ChoiceTasks} or ChoiceTask subclasses. Usually defined &lt;domatch&gt;.
 * <p>
 * <b>Examples:</b><pre>
 *     &lt;domatch property="dist.type"&gt;
 *        &lt;like value="public|production"&gt;
 *          ...
 *        &lt;/like&gt;
 *        &lt;like value="local|internal"&gt;
 *          ...
 *        &lt;/like&gt;
 *     &lt;/domatch&gt;
 * -OR-
 *     &lt;domatch value="${build.type}"&gt;
 *        &lt;equals value="debug"&gt;
 *          ...
 *        &lt;/equals&gt;
 *        &lt;equals value="opt"&gt;
 *          ...
 *        &lt;/equals&gt;
 *        &lt;default&gt;
 *          ...
 *        &lt;/default&gt;
 *     &lt;/domatch&gt;
 * -OR-
 *     &lt;domatch property="build.type" context="build.properties"&gt;
 *        &lt;meets criteria="is-intranet"&gt;
 *          ...
 *        &lt;/meets&gt;
 *        &lt;meets criteria="is-public"&gt;
 *          ...
 *        &lt;/meets&gt;
 *     &lt;/domatch&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5.1
 * @.safety  single
 * @.group   api,infra
 * @see      MatchAll
 * @see      MatchEquals
 * @see      MatchLike
 * @see      MatchCondition
 **/

public class SwitchTask extends TaskSet implements StrictOuterTask, FlexValueSupport
{
    /**
     * Initializes a new SwitchTask.
     **/
    public SwitchTask()
    {
        super(AntX.flow+"switch");
        m_value.setLenient(false);
    }


    /**
     * Initializes a new SwitchTask with custom delay configuration.
     **/
    protected SwitchTask(String iam, boolean delayConfiguration)
    {
        super(iam, delayConfiguration);
        m_value.setLenient(false);
    }


    /**
     * Initializes the enclosing project of this component. Updates
     * any internal project-components too.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        m_value.setProject(P);
    }

// ---------------------------------------------------------------------------------------
// Match-(Flex)Value Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets the switch literal value for this task. One of the flex
     * value types must be defined before this task is executed.
     * @param value the application value to compare (non-null)
     **/
    public final void setValue(String value)
    {
        require_(value!=null,"setValu- nonzro valu");
        if (!m_value.isLiteral()) {
            throw new BuildException
                (uistrs().get("flow.switch.needs.value",
                              getTaskName()), getLocation());
        }
        m_value.set(value);
        m_value.setIsLiteral();
    }


    /**
     * Returns the switch value for this task. Returns <i>null</i>
     * if never set or switch is not setup to use  a literal value.
     * @since JWare/AntX 0.2
     **/
    public final String getSwitchValue()
    {
        return m_value.isLiteral() ? m_value.get() : null;
    }



    /**
     * Sets the switch property for this task. The property's value is
     * determined when this task is executed (not at time of this
     * assignment).
     * @param property the application property to compare (non-null)
     **/
    public void setProperty(String property)
    {
        require_(property!=null,"setProp- nonzro name");
        if (!m_value.isUndefined() && !m_value.isProperty()) {
            throw new BuildException
                (uistrs().get("flow.switch.needs.value",
                              getTaskName()), getLocation());
        }
        m_value.set(property);
        m_value.setIsProperty(true);
    }



    /**
     * Returns the switch property for this task. Returns <i>null</i>
     * if never set or switch is not setup to use a property.
     * @since JWare/AntX 0.2
     **/
    public final String getSwitchProperty()
    {
        return m_value.isProperty() ? m_value.get() : null;
    }



    /**
     * Sets the switch variable for this task. The variables's value
     * is determined when this task is executed (not at time
     * of this assignment).
     * @param variable the variable to compare (non-null)
     * @since JWare/AntX 0.2
     **/
    public void setVariable(String variable)
    {
        require_(variable!=null,"setVar- nonzro name");
        if (!m_value.isUndefined() && !m_value.isExported()) {
            throw new BuildException
                (uistrs().get("flow.switch.needs.value",
                              getTaskName()), getLocation());
        }
        m_value.set(variable);
        m_value.setIsExported(true);
    }



    /**
     * Returns the switch variable for this task. Returns <i>null</i>
     * if never set or switch not setup to use a variable.
     * @since JWare/AntX 0.2
     **/
    public final String getSwitchVariable()
    {
        return m_value.isExported() ? m_value.get() : null;
    }



    /**
     * Sets the switch reference object for this task. The referenced
     * object's string value is determined when this task is executed
     * (not at time of this assignment). Currently only references of
     * type String, URL, File, and Number are supported.
     * @param refid the referenced item's identifier (non-null)
     * @since JWare/AntX 0.2
     **/
    public void setReference(String refid)
    {
        require_(refid!=null,"setRef- nonzro refid");
        if (!m_value.isUndefined() && !m_value.isReference()) {
            throw new BuildException
                (uistrs().get("flow.switch.needs.value",
                              getTaskName()), getLocation());
        }
        m_value.set(refid);
        m_value.setIsReference(true);
    }



    /**
     * Returns the switch reference object's id for this task. Returns
     * <i>null</i> if never set or switch not setup to use a reference
     * object.
     * @since JWare/AntX 0.2
     **/
    public final String getSwitchReference()
    {
        return m_value.isReference() ? m_value.get() : null;
    }



    /**
     * Assigns an evaluation context to this switch task. How the
     * context is used by the different kinds of choices is not known
     * to this task.
     * @param refid reference to collection of key-value settings
     * @since JWare/AntX 0.5
     **/
    public void setContext(Reference refid)
    {
        m_evalContext = refid;
    }



    /**
     * Returns this task's evaluation context. Will return <i>null</i>
     * if no context set explicitly.
     * @since JWare/AntX 0.5
     **/
    public final Reference getContext()
    {
        return m_evalContext;
    }



    /**
     * Returns <i>true</i> if none of the possible switch value types
     * have been explicitly defined. One of the valu types (literal,
     * property-name, variable-name, or reference-id) must be defined
     * before this task is executed.
     **/
    public final boolean isUndefined()
    {
        return m_value.isUndefined();
    }



    /**
     * Returns the actual value used in the switch comparisions. All
     * modifier instructions and property resolution will been applied.
     * Returns <i>null</i> if this task's switch value is undefined or
     * resolves to <i>null</i>.
     **/
    public final String getVUT()
    {
        return m_value.getValue(getProject());
    }



    /**
     * Returns a more human-friendly diagnostic description of broken
     * switch value.
     **/
    private String switchvalueString()
    {
        String s = getSwitchValue();
        if (s!=null) {
            return s;
        }
        s = getSwitchProperty();
        if (s!=null) {
            return "property="+s;
        }
        s = getSwitchVariable();
        if (s!=null) {
            return "variable="+s;
        }
        s = getSwitchReference();
        if (s!=null) {
            return "reference="+s;
        }
        return "";
    }

// ---------------------------------------------------------------------------------------
// Other Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets whether this taskset will abort if no match is found. Note
     * defining a default choice effectively nullifies this option since
     * the default will alway match.
     * @param balk <i>true</i> if no match raises a build exception
     * @see #getDefaultChoice
     **/
    public void setHaltIfNoMatch(boolean balk)
    {
        m_haltIfNoMatch = balk;
    }


    /**
     * Returns <i>true</i> if this taskset will raise an exception if
     * no match is found. Defaults to <i>false</i>
     **/
    public final boolean isHaltIfNoMatch()
    {
        return m_haltIfNoMatch;
    }


    /**
     * Sets the property to be updated with positive match information.
     * By default the property is set with the matching choice's value.
     * @see #setMatchValue
     **/
    public void setMatchProperty(String property)
    {
        require_(property!=null,"setMtchProp- nonzro name");
        m_matchProperty = property;
    }


    /**
     * Returns name of property updated if a positive match is made.
     * Returns <i>null</i> if never set.
     **/
    public final String getMatchProperty()
    {
        return m_matchProperty;
    }


    /**
     * Sets the value to use when setting the {@linkplain
     * #getMatchProperty match property}.
     * @param propertyValue value to be used (non-null)
     **/
    public void setMatchValue(String propertyValue)
    {
        require_(propertyValue!=null,"setMtchVal- nonzro valu");
        m_matchValue = propertyValue;
    }


    /**
     * Returns the property value to use when setting the match
     * property. Returns <i>null</i> if never set.
     **/
    public final String getMatchValue()
    {
        return m_matchValue;
    }


// ---------------------------------------------------------------------------------------
// Match Execution:
// ---------------------------------------------------------------------------------------

    /**
     * Returns this switch task's default choice. If non-null and no
     * explicit choice is matched, this task's nested items are performed.
     * Returns <i>null</i> if this switch task has no default choice.
     * @see #isHaltIfNoMatch
     **/
    public final MatchAll getDefaultChoice()
    {
        return m_defaultTask;
    }



    /**
     * Initializes the default choice taskset for this switch task.
     * Only one default taskset is acceptable.
     **/
    protected final void setTheDefaultChoice(MatchAll cDefault)
    {
        verifyIsUndefined_(m_defaultTask, cDefault);
        m_defaultTask = cDefault;
        m_defaultTask.setEnclosingTask(this);
    }



    /**
     * Ensure the nested choices are configured at same time as this
     * switch task. (Note the choice's sub-tasks aren't configured.)
     **/
    protected void maybeConfigureSpecialTasks()
    {
        Iterator itr= getTasksList().listIterator();
        while (itr.hasNext()) {
            ((ChoiceTask)itr.next()).maybeConfigure();
        }
        if (getDefaultChoice()!=null) {
            getDefaultChoice().maybeConfigure();
        }
    }



    /**
     * Returns <i>true</i> if task is of the allowed choice types.
     * task). The first nested choice determines the required type
     * of all subsequent choices. You cannot mix both 'like' and
     * 'equal' choices.
     * @see MatchLike
     * @see MatchEquals
     **/
    protected boolean includeTask(TaskHandle taskH)
    {
        require_(!taskH.isEmpty(),"includTsk- filld hndl");
        Task task = taskH.getTask();

        ChoiceType type = ChoiceType.from(task.getClass());
        boolean ok= type!=null;

        if (ok) {
            if (m_dominantMatchType!=ChoiceType.NONE) {
                if (type!=m_dominantMatchType) { //only nest one kind
                    String ermsg = uistrs().get("flow.switch.case.onekind",getTaskName());
                    log(ermsg, Project.MSG_ERR);
                    throw new BuildException(ermsg, task.getLocation());
                }
            } else {
                m_dominantMatchType = type;
            }
        }
        return ok;
    }



    /**
     * Ensures only ChoiceTasks added to this taskset. Looks for the
     * special {@linkplain MatchAll} (&lt;default&gt;) element.
     * @see #setTheDefaultChoice
     **/
    public void addTask(Task task)
    {
        require_(task!=null,"addTsk- nonzro tsk");

        task = TaskExaminer.trueTask(task,COI_,this);//remove placeholders!

        if (task instanceof MatchAll) {
            setTheDefaultChoice((MatchAll)task);
            addedTask();
        } else {
            super.addTask(task);
            verify_((task instanceof ChoiceTask),"addTsk- choice tsk");
            ((ChoiceTask)task).setEnclosingTask(this);
        }
    }



    /**
     * Looks (in order) for a match to this switch task's value. If
     * no match and a default choice has been defined, it's used. The
     * matching choice's sub-tasks are executed. If no match found and
     * the "haltIfNoMatch" attribute is <i>true</i>, this method raises
     * a build exception.
     * @throws BuildExceptio if matched choice does or unable to find a
     *         match and "haltIfNoMatch" is <i>true</i>
     **/
    protected void performNestedTasks()
        throws BuildException
    {
        String value = getVUT();
        Task choiceTask = null;
        String choiceValue = Strings.DEFAULT;

        if (value!=null || !m_dominantMatchType.needsLeftAndRight()) {
            Iterator itr = getTasksList().listIterator();
            while (itr.hasNext()) {
                ChoiceTask choice = (ChoiceTask)itr.next();
                if (choice.eval(value, m_evalContext)) {
                    choiceTask = choice;
                    choiceValue = choice.getVUT();
                    break;
                }
            }
        }
        if (choiceTask==null && getDefaultChoice()!=null) {
            choiceTask = getDefaultChoice();
        }
        if (choiceTask!=null) {
            choiceFound(choiceValue,(value==null ? "n/a" : value));
            choiceTask.perform();

        } else if (isHaltIfNoMatch()) {
            String ermsg = uistrs().get("flow.switch.nomatch",switchvalueString());
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        } else {
            log("Unable to match switch-value \""+switchvalueString()+"\"",
                Project.MSG_DEBUG);
        }
    }



    /**
     * Called when a positive match is found for this switch task's
     * value. By default updates 'matchProperty' if requested and logs
     * a diagnostic message.
     * @param choiceValue the hit (non-null)
     * @param switchValue the value under test (non-null)
     * @see #setMatchProperty
     **/
    protected void choiceFound(String choiceValue, String switchValue)
    {
        log("Found hit for '"+switchValue+"' from choice value='"+
            choiceValue+"'", Project.MSG_DEBUG);

        if (getMatchProperty()!=null) {
            String what = getMatchValue();
            if (what==null) {
                what = choiceValue;
            }
            getProject().setNewProperty(getMatchProperty(),what);
        }
    }



    /**
     * Ensures this switch task has its 'value' attribute defined.
     * @throws BuildException if not in valid target/project or missing
     *         'value' attribute
     * @see #setValue
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (isUndefined() && m_dominantMatchType.needsLeftAndRight()) {
            String ermsg = uistrs().get("flow.switch.needs.value",getTaskName());
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }



    private MatchAll   m_defaultTask;
    private boolean    m_haltIfNoMatch=false;//NB: don't balk if no match!
    private String     m_matchProperty, m_matchValue;
    private ChoiceType m_dominantMatchType=ChoiceType.NONE;//NB: defined by first nested item
    private FlexString m_value = new FlexString();
    private Reference  m_evalContext;

    /**
     * Controls the amount of peek-under for UnknownElement placeholders
     * nested inside task containers.
     * @since JWare/AntX 0.4
     **/
    private static final Class[] COI_= {
        MatchAll.class, ChoiceTask.class
    };
}

/* end-of-SwitchTask.java */
