/**
 * $Id: StepLauncherCaller.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.call;

import  java.util.Iterator;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Target;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.taskdefs.Property;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.flowcontrol.FlowConstants;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;

/**
 * Special TargetCaller that calls the special step-launcher target (which in turn calls
 * back into the source target's list of steps).
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 * @see      StepLauncher
 * @see      OnceTask
 **/

public final class StepLauncherCaller extends AnyTargetCaller
{
    /** Used to cache (per project) whether special target has been
     *  has been located and verified.
     *  @since JWare/AntX 0.4
     **/
    private static final String VERIFIED_FLAG=
        AntX.ANTX_INTERNAL_ID+"steplaunch.specialtargetname";


    /**
     * Creates new step-launcher caller. Will look for the default special
     * target.
     * @throws BuildException if unable to locate default special target or
     *         that target is improperly defined (missing StepLauncher task)
     **/
    public StepLauncherCaller(OnceTask from)
        throws BuildException
    {
        super(from, findSpecialTarget(from));
    }


    /**
     * Creates new custom steplauncher caller. Will look for a custom
     * target (which must be structured like a default special target
     * with single steplaunch task).
     * @throws BuildException if unable to locate special target or
     *         that target is improperly defined
     **/
    public StepLauncherCaller(OnceTask from, String targetName)
    {
        super(from,targetName);
        verifySpecialTarget(from,targetName);
    }


    /**
     * Creates pre-verified launcher caller. Used internal to this package.
     **/
    StepLauncherCaller(OnceTask from, String targetName, boolean verified)
    {
        super(from,targetName);
        if (!verified) {
            verifySpecialTarget(from,targetName);
        }
    }



    /**
     * Factory method to creates a preconfigured standard step-launcher caller.
     * The new caller's environment is automatically setup with the required
     * step-launcher variables like step-name, step-class, etc.
     * @see FlowConstants#DEFAULT_STEP_LAUNCHER_TARGET_PROP
     * @see FlowConstants#DEFAULT_STEP_LAUNCHER_STEPNAME_PROP
     **/
    public static final StepLauncherCaller create
        (OnceTask from, boolean verified, String specialTargetName,
         String stepName, String stepClass, int searchDepth)
    {
        StepLauncherCaller caller;
        caller = new StepLauncherCaller(from,specialTargetName,verified);
        caller.setStepName(stepName);

        Property p;

        p = caller.createProperty();
        p.setName(FlowConstants.DEFAULT_STEP_LAUNCHER_TARGET_PROP);
        p.setValue(from.getOwningTarget().getName());

        p = caller.createProperty();
        p.setName(FlowConstants.DEFAULT_STEP_LAUNCHER_STEPNAME_PROP);
        p.setValue(stepName);

        p = caller.createProperty();
        p.setName(FlowConstants.DEFAULT_STEP_LAUNCHER_STEPCLASS_PROP);
        p.setValue(stepClass);

        p = caller.createProperty();
        p.setName(FlowConstants.DEFAULT_STEP_LAUNCHER_SEARCHDEPTH_PROP);
        p.setValue(String.valueOf(searchDepth));

        return caller;
    }


    /**
     * Verifies the definition of a special step-launcher target.
     **/
    final static void verifySpecialTarget(OnceTask from,
                                          String specialTargetName)
    {
        Project P = from.getProject();
        String doneFlag = P.getProperty(VERIFIED_FLAG);
        if (specialTargetName.equals(doneFlag)) {
            return;
        }
        Map targets = (Map)P.getTargets().clone();//NB:MT-safer
        Iterator itr= targets.values().iterator();
        while (itr.hasNext()) {
            Target target = (Target)itr.next();
            if (specialTargetName.equals(target.getName())) {
                Task[] tasks = target.getTasks();
                if (tasks.length==1) {
                    Task task = TaskExaminer.trueTask(tasks[0],COI_,from);
                    if (task instanceof StepLauncher) {
                        P.setNewProperty(VERIFIED_FLAG,specialTargetName);
                        return;//=> this painful hack will work!
                    }
                }
                break;
            }
        }
        throw new BuildException
            (Iteration.uistrs().get("flow.bad.specialtarget", specialTargetName));
    }


    /**
     * Looks for the default special target within the caller's project.
     * The special target is used by step-callers to cope with the existing
     * file requirement of the 'antcall' and 'ant' tasks.
     **/
    final static String findSpecialTarget(OnceTask from)
    {
        final Project P= from.getProject();

        String specialTargetName =
            P.getProperty(FlowConstants.DEFAULT_STEP_CALLER_SPECIAL_TARGETNAME_PROP);

        if (specialTargetName==null) {
            specialTargetName = FlowConstants.DEFAULT_STEP_CALLER_SPECIAL_TARGETNAME;
        }
        verifySpecialTarget(from, specialTargetName);

        return specialTargetName;
    }


    /**
     * Controls the amount of peek-under for UnknownElement placeholders
     * nested inside task containers.
     * @since JWare/AntX 0.4
     **/
    private static final Class[] COI_= {
        StepLauncher.class
    };

}


/* end-of-StepLauncherCaller.java */
